/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.camera2.internal;

import android.content.Context;
import android.graphics.Point;
import android.graphics.SurfaceTexture;
import android.hardware.camera2.CameraCharacteristics;
import android.hardware.camera2.params.StreamConfigurationMap;
import android.media.CamcorderProfile;
import android.media.MediaRecorder;
import android.os.Build;
import android.util.Pair;
import android.util.Rational;
import android.util.Size;
import android.view.WindowManager;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.camera.camera2.internal.CamcorderProfileHelper;
import androidx.camera.camera2.internal.CameraUnavailableExceptionHelper;
import androidx.camera.camera2.internal.compat.CameraAccessExceptionCompat;
import androidx.camera.camera2.internal.compat.CameraCharacteristicsCompat;
import androidx.camera.camera2.internal.compat.CameraManagerCompat;
import androidx.camera.camera2.internal.compat.workaround.ExcludedSupportedSizesContainer;
import androidx.camera.camera2.internal.compat.workaround.TargetAspectRatio;
import androidx.camera.core.CameraUnavailableException;
import androidx.camera.core.Logger;
import androidx.camera.core.impl.ImageOutputConfig;
import androidx.camera.core.impl.SurfaceCombination;
import androidx.camera.core.impl.SurfaceConfig;
import androidx.camera.core.impl.SurfaceSizeDefinition;
import androidx.camera.core.impl.UseCaseConfig;
import androidx.camera.core.impl.utils.CameraOrientationUtil;
import androidx.core.util.Preconditions;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

final class SupportedSurfaceCombination {
    private static final String TAG = "SupportedSurfaceCombination";
    private static final Size MAX_PREVIEW_SIZE = new Size(1920, 1080);
    private static final Size DEFAULT_SIZE = new Size(640, 480);
    private static final Size ZERO_SIZE = new Size(0, 0);
    private static final Size QUALITY_1080P_SIZE = new Size(1920, 1080);
    private static final Size QUALITY_480P_SIZE = new Size(720, 480);
    private static final int ALIGN16 = 16;
    private static final Rational ASPECT_RATIO_4_3 = new Rational(4, 3);
    private static final Rational ASPECT_RATIO_3_4 = new Rational(3, 4);
    private static final Rational ASPECT_RATIO_16_9 = new Rational(16, 9);
    private static final Rational ASPECT_RATIO_9_16 = new Rational(9, 16);
    private final List<SurfaceCombination> mSurfaceCombinations = new ArrayList<SurfaceCombination>();
    private final Map<Integer, Size> mMaxSizeCache = new HashMap<Integer, Size>();
    private final String mCameraId;
    private final CamcorderProfileHelper mCamcorderProfileHelper;
    private final CameraCharacteristicsCompat mCharacteristics;
    private final ExcludedSupportedSizesContainer mExcludedSupportedSizesContainer;
    private final int mHardwareLevel;
    private final boolean mIsSensorLandscapeResolution;
    private final Map<Integer, List<Size>> mExcludedSizeListCache = new HashMap<Integer, List<Size>>();
    private boolean mIsRawSupported = false;
    private boolean mIsBurstCaptureSupported = false;
    private SurfaceSizeDefinition mSurfaceSizeDefinition;
    private Map<Integer, Size[]> mOutputSizesCache = new HashMap<Integer, Size[]>();

    SupportedSurfaceCombination(@NonNull Context context, @NonNull String cameraId, @NonNull CameraManagerCompat cameraManagerCompat, @NonNull CamcorderProfileHelper camcorderProfileHelper) throws CameraUnavailableException {
        this.mCameraId = (String)Preconditions.checkNotNull((Object)cameraId);
        this.mCamcorderProfileHelper = (CamcorderProfileHelper)Preconditions.checkNotNull((Object)camcorderProfileHelper);
        WindowManager windowManager = (WindowManager)context.getSystemService("window");
        this.mExcludedSupportedSizesContainer = new ExcludedSupportedSizesContainer(cameraId);
        try {
            this.mCharacteristics = cameraManagerCompat.getCameraCharacteristicsCompat(this.mCameraId);
            Integer keyValue = (Integer)this.mCharacteristics.get(CameraCharacteristics.INFO_SUPPORTED_HARDWARE_LEVEL);
            this.mHardwareLevel = keyValue != null ? keyValue : 2;
            this.mIsSensorLandscapeResolution = this.isSensorLandscapeResolution();
        }
        catch (CameraAccessExceptionCompat e) {
            throw CameraUnavailableExceptionHelper.createFrom(e);
        }
        this.generateSupportedCombinationList();
        this.generateSurfaceSizeDefinition(windowManager);
        this.checkCustomization();
    }

    String getCameraId() {
        return this.mCameraId;
    }

    boolean isRawSupported() {
        return this.mIsRawSupported;
    }

    boolean isBurstCaptureSupported() {
        return this.mIsBurstCaptureSupported;
    }

    boolean checkSupported(List<SurfaceConfig> surfaceConfigList) {
        SurfaceCombination surfaceCombination;
        boolean isSupported = false;
        Iterator<SurfaceCombination> iterator = this.mSurfaceCombinations.iterator();
        while (iterator.hasNext() && !(isSupported = (surfaceCombination = iterator.next()).isSupported(surfaceConfigList))) {
        }
        return isSupported;
    }

    SurfaceConfig transformSurfaceConfig(int imageFormat, Size size) {
        SurfaceConfig.ConfigSize configSize = SurfaceConfig.ConfigSize.NOT_SUPPORT;
        SurfaceConfig.ConfigType configType = imageFormat == 35 ? SurfaceConfig.ConfigType.YUV : (imageFormat == 256 ? SurfaceConfig.ConfigType.JPEG : (imageFormat == 32 ? SurfaceConfig.ConfigType.RAW : SurfaceConfig.ConfigType.PRIV));
        Size maxSize = this.fetchMaxSize(imageFormat);
        if (size.getWidth() * size.getHeight() <= this.mSurfaceSizeDefinition.getAnalysisSize().getWidth() * this.mSurfaceSizeDefinition.getAnalysisSize().getHeight()) {
            configSize = SurfaceConfig.ConfigSize.ANALYSIS;
        } else if (size.getWidth() * size.getHeight() <= this.mSurfaceSizeDefinition.getPreviewSize().getWidth() * this.mSurfaceSizeDefinition.getPreviewSize().getHeight()) {
            configSize = SurfaceConfig.ConfigSize.PREVIEW;
        } else if (size.getWidth() * size.getHeight() <= this.mSurfaceSizeDefinition.getRecordSize().getWidth() * this.mSurfaceSizeDefinition.getRecordSize().getHeight()) {
            configSize = SurfaceConfig.ConfigSize.RECORD;
        } else if (size.getWidth() * size.getHeight() <= maxSize.getWidth() * maxSize.getHeight()) {
            configSize = SurfaceConfig.ConfigSize.MAXIMUM;
        }
        return SurfaceConfig.create((SurfaceConfig.ConfigType)configType, (SurfaceConfig.ConfigSize)configSize);
    }

    Map<UseCaseConfig<?>, Size> getSuggestedResolutions(List<SurfaceConfig> existingSurfaces, List<UseCaseConfig<?>> newUseCaseConfigs) {
        HashMap suggestedResolutionsMap = new HashMap();
        List<Integer> useCasesPriorityOrder = this.getUseCasesPriorityOrder(newUseCaseConfigs);
        ArrayList<List<Size>> supportedOutputSizesList = new ArrayList<List<Size>>();
        for (Integer index : useCasesPriorityOrder) {
            List<Size> supportedOutputSizes = this.getSupportedOutputSizes(newUseCaseConfigs.get(index));
            supportedOutputSizesList.add(supportedOutputSizes);
        }
        List<List<Size>> allPossibleSizeArrangements = this.getAllPossibleSizeArrangements(supportedOutputSizesList);
        for (List<Size> possibleSizeList : allPossibleSizeArrangements) {
            ArrayList<SurfaceConfig> surfaceConfigList = new ArrayList<SurfaceConfig>(existingSurfaces);
            for (int i = 0; i < possibleSizeList.size(); ++i) {
                Size size = possibleSizeList.get(i);
                UseCaseConfig<?> newUseCase = newUseCaseConfigs.get(useCasesPriorityOrder.get(i));
                surfaceConfigList.add(this.transformSurfaceConfig(newUseCase.getInputFormat(), size));
            }
            if (!this.checkSupported(surfaceConfigList)) continue;
            for (UseCaseConfig<?> useCaseConfig : newUseCaseConfigs) {
                suggestedResolutionsMap.put(useCaseConfig, possibleSizeList.get(useCasesPriorityOrder.indexOf(newUseCaseConfigs.indexOf(useCaseConfig))));
            }
        }
        return suggestedResolutionsMap;
    }

    private Rational getTargetAspectRatio(@NonNull ImageOutputConfig imageOutputConfig) {
        Rational outputRatio = null;
        int targetAspectRatio = new TargetAspectRatio().get(imageOutputConfig, this.mCameraId, this.mCharacteristics);
        block0 : switch (targetAspectRatio) {
            case 0: {
                outputRatio = this.mIsSensorLandscapeResolution ? ASPECT_RATIO_4_3 : ASPECT_RATIO_3_4;
                break;
            }
            case 1: {
                outputRatio = this.mIsSensorLandscapeResolution ? ASPECT_RATIO_16_9 : ASPECT_RATIO_9_16;
                break;
            }
            case 2: {
                Size maxJpegSize = this.fetchMaxSize(256);
                outputRatio = new Rational(maxJpegSize.getWidth(), maxJpegSize.getHeight());
                break;
            }
            case 3: {
                Size targetSize = this.getTargetSize(imageOutputConfig);
                if (imageOutputConfig.hasTargetAspectRatio()) {
                    int aspectRatio = imageOutputConfig.getTargetAspectRatio();
                    switch (aspectRatio) {
                        case 0: {
                            outputRatio = this.mIsSensorLandscapeResolution ? ASPECT_RATIO_4_3 : ASPECT_RATIO_3_4;
                            break block0;
                        }
                        case 1: {
                            outputRatio = this.mIsSensorLandscapeResolution ? ASPECT_RATIO_16_9 : ASPECT_RATIO_9_16;
                            break block0;
                        }
                    }
                    Logger.e((String)TAG, (String)("Undefined target aspect ratio: " + aspectRatio));
                    break;
                }
                if (targetSize == null) break;
                outputRatio = new Rational(targetSize.getWidth(), targetSize.getHeight());
                break;
            }
        }
        return outputRatio;
    }

    SurfaceSizeDefinition getSurfaceSizeDefinition() {
        return this.mSurfaceSizeDefinition;
    }

    private Size fetchMaxSize(int imageFormat) {
        Size size = this.mMaxSizeCache.get(imageFormat);
        if (size != null) {
            return size;
        }
        Size maxSize = this.getMaxOutputSizeByFormat(imageFormat);
        this.mMaxSizeCache.put(imageFormat, maxSize);
        return maxSize;
    }

    private List<Integer> getUseCasesPriorityOrder(List<UseCaseConfig<?>> newUseCaseConfigs) {
        ArrayList<Integer> priorityOrder = new ArrayList<Integer>();
        ArrayList<Integer> priorityValueList = new ArrayList<Integer>();
        for (UseCaseConfig<?> config : newUseCaseConfigs) {
            int priority = config.getSurfaceOccupancyPriority(0);
            if (priorityValueList.contains(priority)) continue;
            priorityValueList.add(priority);
        }
        Collections.sort(priorityValueList);
        Collections.reverse(priorityValueList);
        Iterator<Object> iterator = priorityValueList.iterator();
        while (iterator.hasNext()) {
            int priorityValue = (Integer)iterator.next();
            for (UseCaseConfig<?> config : newUseCaseConfigs) {
                if (priorityValue != config.getSurfaceOccupancyPriority(0)) continue;
                priorityOrder.add(newUseCaseConfigs.indexOf(config));
            }
        }
        return priorityOrder;
    }

    @NonNull
    @VisibleForTesting
    List<Size> getSupportedOutputSizes(@NonNull UseCaseConfig<?> config) {
        ImageOutputConfig imageOutputConfig;
        int imageFormat = config.getInputFormat();
        Size[] outputSizes = this.getCustomizedSupportSizesFromConfig(imageFormat, imageOutputConfig = (ImageOutputConfig)config);
        if (outputSizes == null) {
            outputSizes = this.getAllOutputSizesByFormat(imageFormat);
        }
        ArrayList<Size> outputSizeCandidates = new ArrayList<Size>();
        Size maxSize = imageOutputConfig.getMaxResolution(null);
        Size maxOutputSizeByFormat = this.getMaxOutputSizeByFormat(imageFormat);
        if (maxSize == null || SupportedSurfaceCombination.getArea(maxOutputSizeByFormat) < SupportedSurfaceCombination.getArea(maxSize)) {
            maxSize = maxOutputSizeByFormat;
        }
        Arrays.sort(outputSizes, new CompareSizesByArea(true));
        Size targetSize = this.getTargetSize(imageOutputConfig);
        Size minSize = DEFAULT_SIZE;
        int defaultSizeArea = SupportedSurfaceCombination.getArea(DEFAULT_SIZE);
        int maxSizeArea = SupportedSurfaceCombination.getArea(maxSize);
        if (maxSizeArea < defaultSizeArea) {
            minSize = ZERO_SIZE;
        } else if (targetSize != null && SupportedSurfaceCombination.getArea(targetSize) < defaultSizeArea) {
            minSize = targetSize;
        }
        for (Size outputSize : outputSizes) {
            if (SupportedSurfaceCombination.getArea(outputSize) > SupportedSurfaceCombination.getArea(maxSize) || SupportedSurfaceCombination.getArea(outputSize) < SupportedSurfaceCombination.getArea(minSize) || outputSizeCandidates.contains(outputSize)) continue;
            outputSizeCandidates.add(outputSize);
        }
        if (outputSizeCandidates.isEmpty()) {
            throw new IllegalArgumentException("Can not get supported output size under supported maximum for the format: " + imageFormat);
        }
        Rational aspectRatio = this.getTargetAspectRatio(imageOutputConfig);
        targetSize = targetSize == null ? imageOutputConfig.getDefaultResolution(null) : targetSize;
        ArrayList<Size> supportedResolutions = new ArrayList<Size>();
        Map<Object, Object> aspectRatioSizeListMap = new HashMap();
        if (aspectRatio == null) {
            supportedResolutions.addAll(outputSizeCandidates);
            if (targetSize != null) {
                this.removeSupportedSizesByTargetSize(supportedResolutions, targetSize);
            }
        } else {
            aspectRatioSizeListMap = this.groupSizesByAspectRatio(outputSizeCandidates);
            if (targetSize != null) {
                for (Rational rational : aspectRatioSizeListMap.keySet()) {
                    this.removeSupportedSizesByTargetSize((List)aspectRatioSizeListMap.get(rational), targetSize);
                }
            }
            ArrayList<Object> aspectRatios = new ArrayList<Object>(aspectRatioSizeListMap.keySet());
            Collections.sort(aspectRatios, new CompareAspectRatiosByDistanceToTargetRatio(aspectRatio));
            for (Rational rational : aspectRatios) {
                for (Size size : (List)aspectRatioSizeListMap.get(rational)) {
                    if (supportedResolutions.contains(size)) continue;
                    supportedResolutions.add(size);
                }
            }
        }
        return supportedResolutions;
    }

    @Nullable
    private Size getTargetSize(@NonNull ImageOutputConfig imageOutputConfig) {
        int targetRotation = imageOutputConfig.getTargetRotation(0);
        Size targetSize = imageOutputConfig.getTargetResolution(null);
        targetSize = this.flipSizeByRotation(targetSize, targetRotation);
        return targetSize;
    }

    @Nullable
    private Size flipSizeByRotation(@Nullable Size size, int targetRotation) {
        Size outputSize = size;
        if (size != null && this.isRotationNeeded(targetRotation)) {
            outputSize = new Size(size.getHeight(), size.getWidth());
        }
        return outputSize;
    }

    private boolean isRotationNeeded(int targetRotation) {
        Integer sensorOrientation = (Integer)this.mCharacteristics.get(CameraCharacteristics.SENSOR_ORIENTATION);
        Preconditions.checkNotNull((Object)sensorOrientation, (Object)"Camera HAL in bad state, unable to retrieve the SENSOR_ORIENTATION");
        int relativeRotationDegrees = CameraOrientationUtil.surfaceRotationToDegrees((int)targetRotation);
        Integer lensFacing = (Integer)this.mCharacteristics.get(CameraCharacteristics.LENS_FACING);
        Preconditions.checkNotNull((Object)lensFacing, (Object)"Camera HAL in bad state, unable to retrieve the LENS_FACING");
        boolean isOppositeFacingScreen = 1 == lensFacing;
        int sensorRotationDegrees = CameraOrientationUtil.getRelativeImageRotation((int)relativeRotationDegrees, (int)sensorOrientation, (boolean)isOppositeFacingScreen);
        return sensorRotationDegrees == 90 || sensorRotationDegrees == 270;
    }

    private boolean isSensorLandscapeResolution() {
        Size pixelArraySize = (Size)this.mCharacteristics.get(CameraCharacteristics.SENSOR_INFO_PIXEL_ARRAY_SIZE);
        return pixelArraySize != null ? pixelArraySize.getWidth() >= pixelArraySize.getHeight() : true;
    }

    static boolean hasMatchingAspectRatio(Size resolution, Rational aspectRatio) {
        boolean isMatch = false;
        if (aspectRatio == null) {
            isMatch = false;
        } else if (aspectRatio.equals((Object)new Rational(resolution.getWidth(), resolution.getHeight()))) {
            isMatch = true;
        } else if (SupportedSurfaceCombination.getArea(resolution) >= SupportedSurfaceCombination.getArea(DEFAULT_SIZE)) {
            isMatch = SupportedSurfaceCombination.isPossibleMod16FromAspectRatio(resolution, aspectRatio);
        }
        return isMatch;
    }

    private static boolean isPossibleMod16FromAspectRatio(Size resolution, Rational aspectRatio) {
        int width = resolution.getWidth();
        int height = resolution.getHeight();
        Rational invAspectRatio = new Rational(aspectRatio.getDenominator(), aspectRatio.getNumerator());
        if (width % 16 == 0 && height % 16 == 0) {
            return SupportedSurfaceCombination.ratioIntersectsMod16Segment(Math.max(0, height - 16), width, aspectRatio) || SupportedSurfaceCombination.ratioIntersectsMod16Segment(Math.max(0, width - 16), height, invAspectRatio);
        }
        if (width % 16 == 0) {
            return SupportedSurfaceCombination.ratioIntersectsMod16Segment(height, width, aspectRatio);
        }
        if (height % 16 == 0) {
            return SupportedSurfaceCombination.ratioIntersectsMod16Segment(width, height, invAspectRatio);
        }
        return false;
    }

    private static int getArea(Size size) {
        return size.getWidth() * size.getHeight();
    }

    private static boolean ratioIntersectsMod16Segment(int height, int mod16Width, Rational aspectRatio) {
        Preconditions.checkArgument((mod16Width % 16 == 0 ? 1 : 0) != 0);
        double aspectRatioWidth = (double)(height * aspectRatio.getNumerator()) / (double)aspectRatio.getDenominator();
        return aspectRatioWidth > (double)Math.max(0, mod16Width - 16) && aspectRatioWidth < (double)(mod16Width + 16);
    }

    private Map<Rational, List<Size>> groupSizesByAspectRatio(List<Size> sizes) {
        HashMap<Rational, List<Size>> aspectRatioSizeListMap = new HashMap<Rational, List<Size>>();
        aspectRatioSizeListMap.put(ASPECT_RATIO_4_3, new ArrayList());
        aspectRatioSizeListMap.put(ASPECT_RATIO_16_9, new ArrayList());
        for (Size outputSize : sizes) {
            Rational matchedKey = null;
            for (Rational key : aspectRatioSizeListMap.keySet()) {
                List sizeList;
                if (!SupportedSurfaceCombination.hasMatchingAspectRatio(outputSize, key) || (sizeList = (List)aspectRatioSizeListMap.get(matchedKey = key)).contains(outputSize)) continue;
                sizeList.add(outputSize);
            }
            if (matchedKey != null) continue;
            aspectRatioSizeListMap.put(new Rational(outputSize.getWidth(), outputSize.getHeight()), new ArrayList<Size>(Collections.singleton(outputSize)));
        }
        return aspectRatioSizeListMap;
    }

    private void removeSupportedSizesByTargetSize(List<Size> supportedSizesList, Size targetSize) {
        Size outputSize;
        if (supportedSizesList == null || supportedSizesList.isEmpty()) {
            return;
        }
        int indexBigEnough = -1;
        ArrayList<Size> removeSizes = new ArrayList<Size>();
        int i = 0;
        while (i < supportedSizesList.size() && (outputSize = supportedSizesList.get(i)).getWidth() >= targetSize.getWidth() && outputSize.getHeight() >= targetSize.getHeight()) {
            if (indexBigEnough >= 0) {
                removeSizes.add(supportedSizesList.get(indexBigEnough));
            }
            indexBigEnough = i++;
        }
        supportedSizesList.removeAll(removeSizes);
    }

    private List<List<Size>> getAllPossibleSizeArrangements(List<List<Size>> supportedOutputSizesList) {
        int totalArrangementsCount = 1;
        for (List<Size> supportedOutputSizes : supportedOutputSizesList) {
            totalArrangementsCount *= supportedOutputSizes.size();
        }
        if (totalArrangementsCount == 0) {
            throw new IllegalArgumentException("Failed to find supported resolutions.");
        }
        ArrayList<List<Size>> allPossibleSizeArrangements = new ArrayList<List<Size>>();
        for (int i = 0; i < totalArrangementsCount; ++i) {
            ArrayList sizeList = new ArrayList();
            allPossibleSizeArrangements.add(sizeList);
        }
        int currentRunCount = totalArrangementsCount;
        int nextRunCount = currentRunCount / supportedOutputSizesList.get(0).size();
        for (int currentIndex = 0; currentIndex < supportedOutputSizesList.size(); ++currentIndex) {
            List<Size> supportedOutputSizes = supportedOutputSizesList.get(currentIndex);
            for (int i = 0; i < totalArrangementsCount; ++i) {
                List surfaceConfigList = (List)allPossibleSizeArrangements.get(i);
                surfaceConfigList.add(supportedOutputSizes.get(i % currentRunCount / nextRunCount));
            }
            if (currentIndex >= supportedOutputSizesList.size() - 1) continue;
            currentRunCount = nextRunCount;
            nextRunCount = currentRunCount / supportedOutputSizesList.get(currentIndex + 1).size();
        }
        return allPossibleSizeArrangements;
    }

    @NonNull
    private Size[] excludeProblematicSizes(@NonNull Size[] outputSizes, int imageFormat) {
        List<Size> excludedSizes = this.fetchExcludedSizes(imageFormat);
        ArrayList<Size> resultSizesList = new ArrayList<Size>(Arrays.asList(outputSizes));
        resultSizesList.removeAll(excludedSizes);
        return resultSizesList.toArray(new Size[0]);
    }

    @Nullable
    private Size[] getCustomizedSupportSizesFromConfig(int imageFormat, @NonNull ImageOutputConfig config) {
        Size[] outputSizes = null;
        List formatResolutionsPairList = config.getSupportedResolutions(null);
        if (formatResolutionsPairList != null) {
            for (Pair formatResolutionPair : formatResolutionsPairList) {
                if ((Integer)formatResolutionPair.first != imageFormat) continue;
                outputSizes = (Size[])formatResolutionPair.second;
                break;
            }
        }
        if (outputSizes != null) {
            outputSizes = this.excludeProblematicSizes(outputSizes, imageFormat);
            Arrays.sort(outputSizes, new CompareSizesByArea(true));
        }
        return outputSizes;
    }

    @NonNull
    private Size[] getAllOutputSizesByFormat(int imageFormat) {
        Size[] outputs = this.mOutputSizesCache.get(imageFormat);
        if (outputs == null) {
            outputs = this.doGetAllOutputSizesByFormat(imageFormat);
            this.mOutputSizesCache.put(imageFormat, outputs);
        }
        return outputs;
    }

    @NonNull
    private Size[] doGetAllOutputSizesByFormat(int imageFormat) {
        StreamConfigurationMap map = (StreamConfigurationMap)this.mCharacteristics.get(CameraCharacteristics.SCALER_STREAM_CONFIGURATION_MAP);
        if (map == null) {
            throw new IllegalArgumentException("Can not retrieve SCALER_STREAM_CONFIGURATION_MAP");
        }
        Size[] outputSizes = Build.VERSION.SDK_INT < 23 && imageFormat == 34 ? map.getOutputSizes(SurfaceTexture.class) : map.getOutputSizes(imageFormat);
        if (outputSizes == null) {
            throw new IllegalArgumentException("Can not get supported output size for the format: " + imageFormat);
        }
        outputSizes = this.excludeProblematicSizes(outputSizes, imageFormat);
        Arrays.sort(outputSizes, new CompareSizesByArea(true));
        return outputSizes;
    }

    Size getMaxOutputSizeByFormat(int imageFormat) {
        Size[] outputSizes = this.getAllOutputSizesByFormat(imageFormat);
        return Collections.max(Arrays.asList(outputSizes), new CompareSizesByArea());
    }

    List<SurfaceCombination> getLegacySupportedCombinationList() {
        ArrayList<SurfaceCombination> combinationList = new ArrayList<SurfaceCombination>();
        SurfaceCombination surfaceCombination1 = new SurfaceCombination();
        surfaceCombination1.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination1);
        SurfaceCombination surfaceCombination2 = new SurfaceCombination();
        surfaceCombination2.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.JPEG, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination2);
        SurfaceCombination surfaceCombination3 = new SurfaceCombination();
        surfaceCombination3.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination3);
        SurfaceCombination surfaceCombination4 = new SurfaceCombination();
        surfaceCombination4.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination4.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.JPEG, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination4);
        SurfaceCombination surfaceCombination5 = new SurfaceCombination();
        surfaceCombination5.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination5.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.JPEG, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination5);
        SurfaceCombination surfaceCombination6 = new SurfaceCombination();
        surfaceCombination6.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination6.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        combinationList.add(surfaceCombination6);
        SurfaceCombination surfaceCombination7 = new SurfaceCombination();
        surfaceCombination7.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination7.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        combinationList.add(surfaceCombination7);
        SurfaceCombination surfaceCombination8 = new SurfaceCombination();
        surfaceCombination8.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination8.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination8.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.JPEG, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination8);
        return combinationList;
    }

    List<SurfaceCombination> getLimitedSupportedCombinationList() {
        ArrayList<SurfaceCombination> combinationList = new ArrayList<SurfaceCombination>();
        SurfaceCombination surfaceCombination1 = new SurfaceCombination();
        surfaceCombination1.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination1.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.RECORD));
        combinationList.add(surfaceCombination1);
        SurfaceCombination surfaceCombination2 = new SurfaceCombination();
        surfaceCombination2.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination2.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.RECORD));
        combinationList.add(surfaceCombination2);
        SurfaceCombination surfaceCombination3 = new SurfaceCombination();
        surfaceCombination3.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination3.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.RECORD));
        combinationList.add(surfaceCombination3);
        SurfaceCombination surfaceCombination4 = new SurfaceCombination();
        surfaceCombination4.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination4.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.RECORD));
        surfaceCombination4.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.JPEG, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.RECORD));
        combinationList.add(surfaceCombination4);
        SurfaceCombination surfaceCombination5 = new SurfaceCombination();
        surfaceCombination5.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination5.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.RECORD));
        surfaceCombination5.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.JPEG, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.RECORD));
        combinationList.add(surfaceCombination5);
        SurfaceCombination surfaceCombination6 = new SurfaceCombination();
        surfaceCombination6.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination6.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination6.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.JPEG, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination6);
        return combinationList;
    }

    List<SurfaceCombination> getFullSupportedCombinationList() {
        ArrayList<SurfaceCombination> combinationList = new ArrayList<SurfaceCombination>();
        SurfaceCombination surfaceCombination1 = new SurfaceCombination();
        surfaceCombination1.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination1.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination1);
        SurfaceCombination surfaceCombination2 = new SurfaceCombination();
        surfaceCombination2.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination2.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination2);
        SurfaceCombination surfaceCombination3 = new SurfaceCombination();
        surfaceCombination3.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination3.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination3);
        SurfaceCombination surfaceCombination4 = new SurfaceCombination();
        surfaceCombination4.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination4.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination4.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.JPEG, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination4);
        SurfaceCombination surfaceCombination5 = new SurfaceCombination();
        surfaceCombination5.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.ANALYSIS));
        surfaceCombination5.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination5.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination5);
        SurfaceCombination surfaceCombination6 = new SurfaceCombination();
        surfaceCombination6.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.ANALYSIS));
        surfaceCombination6.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination6.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination6);
        return combinationList;
    }

    List<SurfaceCombination> getRAWSupportedCombinationList() {
        ArrayList<SurfaceCombination> combinationList = new ArrayList<SurfaceCombination>();
        SurfaceCombination surfaceCombination1 = new SurfaceCombination();
        surfaceCombination1.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.RAW, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination1);
        SurfaceCombination surfaceCombination2 = new SurfaceCombination();
        surfaceCombination2.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination2.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.RAW, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination2);
        SurfaceCombination surfaceCombination3 = new SurfaceCombination();
        surfaceCombination3.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination3.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.RAW, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination3);
        SurfaceCombination surfaceCombination4 = new SurfaceCombination();
        surfaceCombination4.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination4.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination4.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.RAW, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination4);
        SurfaceCombination surfaceCombination5 = new SurfaceCombination();
        surfaceCombination5.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination5.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination5.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.RAW, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination5);
        SurfaceCombination surfaceCombination6 = new SurfaceCombination();
        surfaceCombination6.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination6.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination6.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.RAW, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination6);
        SurfaceCombination surfaceCombination7 = new SurfaceCombination();
        surfaceCombination7.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination7.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.JPEG, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        surfaceCombination7.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.RAW, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination7);
        SurfaceCombination surfaceCombination8 = new SurfaceCombination();
        surfaceCombination8.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination8.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.JPEG, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        surfaceCombination8.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.RAW, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination8);
        return combinationList;
    }

    List<SurfaceCombination> getBurstSupportedCombinationList() {
        ArrayList<SurfaceCombination> combinationList = new ArrayList<SurfaceCombination>();
        SurfaceCombination surfaceCombination1 = new SurfaceCombination();
        surfaceCombination1.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination1.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination1);
        SurfaceCombination surfaceCombination2 = new SurfaceCombination();
        surfaceCombination2.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination2.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination2);
        SurfaceCombination surfaceCombination3 = new SurfaceCombination();
        surfaceCombination3.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination3.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination3);
        return combinationList;
    }

    List<SurfaceCombination> getLevel3SupportedCombinationList() {
        ArrayList<SurfaceCombination> combinationList = new ArrayList<SurfaceCombination>();
        SurfaceCombination surfaceCombination1 = new SurfaceCombination();
        surfaceCombination1.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination1.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.ANALYSIS));
        surfaceCombination1.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.YUV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        surfaceCombination1.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.RAW, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination1);
        SurfaceCombination surfaceCombination2 = new SurfaceCombination();
        surfaceCombination2.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.PREVIEW));
        surfaceCombination2.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.PRIV, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.ANALYSIS));
        surfaceCombination2.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.JPEG, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        surfaceCombination2.addSurfaceConfig(SurfaceConfig.create((SurfaceConfig.ConfigType)SurfaceConfig.ConfigType.RAW, (SurfaceConfig.ConfigSize)SurfaceConfig.ConfigSize.MAXIMUM));
        combinationList.add(surfaceCombination2);
        return combinationList;
    }

    private void generateSupportedCombinationList() {
        int[] availableCapabilities;
        this.mSurfaceCombinations.addAll(this.getLegacySupportedCombinationList());
        if (this.mHardwareLevel == 0 || this.mHardwareLevel == 1 || this.mHardwareLevel == 3) {
            this.mSurfaceCombinations.addAll(this.getLimitedSupportedCombinationList());
        }
        if (this.mHardwareLevel == 1 || this.mHardwareLevel == 3) {
            this.mSurfaceCombinations.addAll(this.getFullSupportedCombinationList());
        }
        if ((availableCapabilities = (int[])this.mCharacteristics.get(CameraCharacteristics.REQUEST_AVAILABLE_CAPABILITIES)) != null) {
            for (int capability : availableCapabilities) {
                if (capability == 3) {
                    this.mIsRawSupported = true;
                    continue;
                }
                if (capability != 6) continue;
                this.mIsBurstCaptureSupported = true;
            }
        }
        if (this.mIsRawSupported) {
            this.mSurfaceCombinations.addAll(this.getRAWSupportedCombinationList());
        }
        if (this.mIsBurstCaptureSupported && this.mHardwareLevel == 0) {
            this.mSurfaceCombinations.addAll(this.getBurstSupportedCombinationList());
        }
        if (this.mHardwareLevel == 3) {
            this.mSurfaceCombinations.addAll(this.getLevel3SupportedCombinationList());
        }
    }

    private void checkCustomization() {
    }

    private void generateSurfaceSizeDefinition(WindowManager windowManager) {
        Size analysisSize = new Size(640, 480);
        Size previewSize = SupportedSurfaceCombination.getPreviewSize(windowManager);
        Size recordSize = this.getRecordSize();
        this.mSurfaceSizeDefinition = SurfaceSizeDefinition.create((Size)analysisSize, (Size)previewSize, (Size)recordSize);
    }

    @NonNull
    public static Size getPreviewSize(@NonNull WindowManager windowManager) {
        Point displaySize = new Point();
        windowManager.getDefaultDisplay().getRealSize(displaySize);
        Size displayViewSize = displaySize.x > displaySize.y ? new Size(displaySize.x, displaySize.y) : new Size(displaySize.y, displaySize.x);
        Size previewSize = Collections.min(Arrays.asList(new Size(displayViewSize.getWidth(), displayViewSize.getHeight()), MAX_PREVIEW_SIZE), new CompareSizesByArea());
        return previewSize;
    }

    @NonNull
    private Size getRecordSize() {
        int cameraId;
        try {
            cameraId = Integer.parseInt(this.mCameraId);
        }
        catch (NumberFormatException e) {
            return this.getRecordSizeFromStreamConfigurationMap();
        }
        CamcorderProfile profile = null;
        if (this.mCamcorderProfileHelper.hasProfile(cameraId, 1)) {
            profile = this.mCamcorderProfileHelper.get(cameraId, 1);
        }
        if (profile != null) {
            return new Size(profile.videoFrameWidth, profile.videoFrameHeight);
        }
        return this.getRecordSizeByHasProfile(cameraId);
    }

    @NonNull
    private Size getRecordSizeFromStreamConfigurationMap() {
        StreamConfigurationMap map = (StreamConfigurationMap)this.mCharacteristics.get(CameraCharacteristics.SCALER_STREAM_CONFIGURATION_MAP);
        if (map == null) {
            throw new IllegalArgumentException("Can not retrieve SCALER_STREAM_CONFIGURATION_MAP");
        }
        Size[] videoSizeArr = map.getOutputSizes(MediaRecorder.class);
        if (videoSizeArr == null) {
            return QUALITY_480P_SIZE;
        }
        Arrays.sort(videoSizeArr, new CompareSizesByArea(true));
        for (Size size : videoSizeArr) {
            if (size.getWidth() > QUALITY_1080P_SIZE.getWidth() || size.getHeight() > QUALITY_1080P_SIZE.getHeight()) continue;
            return size;
        }
        return QUALITY_480P_SIZE;
    }

    @NonNull
    private Size getRecordSizeByHasProfile(int cameraId) {
        Size recordSize = QUALITY_480P_SIZE;
        CamcorderProfile profile = null;
        if (this.mCamcorderProfileHelper.hasProfile(cameraId, 10)) {
            profile = this.mCamcorderProfileHelper.get(cameraId, 10);
        } else if (this.mCamcorderProfileHelper.hasProfile(cameraId, 8)) {
            profile = this.mCamcorderProfileHelper.get(cameraId, 8);
        } else if (this.mCamcorderProfileHelper.hasProfile(cameraId, 12)) {
            profile = this.mCamcorderProfileHelper.get(cameraId, 12);
        } else if (this.mCamcorderProfileHelper.hasProfile(cameraId, 6)) {
            profile = this.mCamcorderProfileHelper.get(cameraId, 6);
        } else if (this.mCamcorderProfileHelper.hasProfile(cameraId, 5)) {
            profile = this.mCamcorderProfileHelper.get(cameraId, 5);
        } else if (this.mCamcorderProfileHelper.hasProfile(cameraId, 4)) {
            profile = this.mCamcorderProfileHelper.get(cameraId, 4);
        }
        if (profile != null) {
            recordSize = new Size(profile.videoFrameWidth, profile.videoFrameHeight);
        }
        return recordSize;
    }

    @NonNull
    private List<Size> fetchExcludedSizes(int imageFormat) {
        List<Size> excludedSizes = this.mExcludedSizeListCache.get(imageFormat);
        if (excludedSizes == null) {
            excludedSizes = this.mExcludedSupportedSizesContainer.get(imageFormat);
            this.mExcludedSizeListCache.put(imageFormat, excludedSizes);
        }
        return excludedSizes;
    }

    static final class CompareAspectRatiosByDistanceToTargetRatio
    implements Comparator<Rational> {
        private Rational mTargetRatio;

        CompareAspectRatiosByDistanceToTargetRatio(Rational targetRatio) {
            this.mTargetRatio = targetRatio;
        }

        @Override
        public int compare(Rational lhs, Rational rhs) {
            if (lhs.equals((Object)rhs)) {
                return 0;
            }
            Float lhsRatioDelta = Float.valueOf(Math.abs(lhs.floatValue() - this.mTargetRatio.floatValue()));
            Float rhsRatioDelta = Float.valueOf(Math.abs(rhs.floatValue() - this.mTargetRatio.floatValue()));
            int result = (int)Math.signum(lhsRatioDelta.floatValue() - rhsRatioDelta.floatValue());
            return result;
        }
    }

    static final class CompareSizesByArea
    implements Comparator<Size> {
        private boolean mReverse = false;

        CompareSizesByArea() {
        }

        CompareSizesByArea(boolean reverse) {
            this.mReverse = reverse;
        }

        @Override
        public int compare(Size lhs, Size rhs) {
            int result = Long.signum((long)lhs.getWidth() * (long)lhs.getHeight() - (long)rhs.getWidth() * (long)rhs.getHeight());
            if (this.mReverse) {
                result *= -1;
            }
            return result;
        }
    }
}

