/*
 * Copyright 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.compose.foundation.text.selection

import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Rect
import androidx.compose.ui.geometry.isUnspecified
import androidx.compose.ui.layout.LayoutCoordinates
import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.TextLayoutResult
import androidx.compose.ui.text.TextRange
import kotlin.math.max

internal class MultiWidgetSelectionDelegate(
    override val selectableId: Long,
    private val coordinatesCallback: () -> LayoutCoordinates?,
    private val layoutResultCallback: () -> TextLayoutResult?
) : Selectable {

    private var _previousTextLayoutResult: TextLayoutResult? = null

    // previously calculated `lastVisibleOffset` for the `_previousTextLayoutResult`
    private var _previousLastVisibleOffset: Int = -1

    /**
     * TextLayoutResult is not expected to change repeatedly in a BasicText composable. At least
     * most TextLayoutResult changes would likely affect Selection logic in some way. Therefore,
     * this value only caches the last visible offset calculation for the latest seen
     * TextLayoutResult instance. Object equality check is not worth the extra calculation as
     * instance check is enough to accomplish whether a text layout has changed in a meaningful
     * way.
     */
    private val TextLayoutResult.lastVisibleOffset: Int
        @Synchronized get() {
            if (_previousTextLayoutResult !== this) {
                val lastVisibleLine = when {
                    !didOverflowHeight || multiParagraph.didExceedMaxLines -> lineCount - 1
                    else -> { // size.height < multiParagraph.height
                        var finalVisibleLine = getLineForVerticalPosition(size.height.toFloat())
                            .coerceAtMost(lineCount - 1)
                        // if final visible line's top is equal to or larger than text layout
                        // result's height, we need to check above lines one by one until we find
                        // a line that fits in boundaries.
                        while (
                            finalVisibleLine >= 0 &&
                            getLineTop(finalVisibleLine) >= size.height
                        ) finalVisibleLine--
                        finalVisibleLine.coerceAtLeast(0)
                    }
                }
                _previousLastVisibleOffset = getLineEnd(lastVisibleLine, true)
                _previousTextLayoutResult = this
            }
            return _previousLastVisibleOffset
        }

    override fun appendSelectableInfoToBuilder(builder: SelectionLayoutBuilder) {
        val layoutCoordinates = getLayoutCoordinates() ?: return
        val textLayoutResult = layoutResultCallback() ?: return

        val relativePosition =
            builder.containerCoordinates.localPositionOf(layoutCoordinates, Offset.Zero)
        val localStartPosition = builder.startHandlePosition - relativePosition
        val localEndPosition = builder.endHandlePosition - relativePosition
        val localPreviousHandlePosition = if (builder.previousHandlePosition.isUnspecified) {
            Offset.Unspecified
        } else {
            builder.previousHandlePosition - relativePosition
        }

        builder.appendSelectableInfo(
            textLayoutResult = textLayoutResult,
            startPosition = localStartPosition,
            endPosition = localEndPosition,
            previousHandlePosition = localPreviousHandlePosition,
            selectableId = selectableId,
        )
    }

    override fun getSelectAllSelection(): Selection? {
        val textLayoutResult = layoutResultCallback() ?: return null
        val start = 0
        val end = textLayoutResult.layoutInput.text.length

        return Selection(
            start = Selection.AnchorInfo(
                direction = textLayoutResult.getBidiRunDirection(start),
                offset = start,
                selectableId = selectableId
            ),
            end = Selection.AnchorInfo(
                direction = textLayoutResult.getBidiRunDirection(max(end - 1, 0)),
                offset = end,
                selectableId = selectableId
            ),
            handlesCrossed = false
        )
    }

    override fun getHandlePosition(selection: Selection, isStartHandle: Boolean): Offset {
        // Check if the selection handle's selectable is the current selectable.
        if (isStartHandle && selection.start.selectableId != this.selectableId ||
            !isStartHandle && selection.end.selectableId != this.selectableId
        ) {
            return Offset.Zero
        }

        if (getLayoutCoordinates() == null) return Offset.Zero

        val textLayoutResult = layoutResultCallback() ?: return Offset.Zero
        val offset = if (isStartHandle) selection.start.offset else selection.end.offset
        val coercedOffset = offset.coerceIn(0, textLayoutResult.lastVisibleOffset)
        return getSelectionHandleCoordinates(
            textLayoutResult = textLayoutResult,
            offset = coercedOffset,
            isStart = isStartHandle,
            areHandlesCrossed = selection.handlesCrossed
        )
    }

    override fun getLayoutCoordinates(): LayoutCoordinates? {
        val layoutCoordinates = coordinatesCallback()
        if (layoutCoordinates == null || !layoutCoordinates.isAttached) return null
        return layoutCoordinates
    }

    override fun getText(): AnnotatedString {
        val textLayoutResult = layoutResultCallback() ?: return AnnotatedString("")
        return textLayoutResult.layoutInput.text
    }

    override fun getBoundingBox(offset: Int): Rect {
        val textLayoutResult = layoutResultCallback() ?: return Rect.Zero
        val textLength = textLayoutResult.layoutInput.text.length
        if (textLength < 1) return Rect.Zero
        return textLayoutResult.getBoundingBox(
            offset.coerceIn(0, textLength - 1)
        )
    }

    override fun getLineLeft(offset: Int): Float {
        val textLayoutResult = layoutResultCallback() ?: return -1f
        val line = textLayoutResult.getLineForOffset(offset)
        return textLayoutResult.getLineLeft(line)
    }

    override fun getLineRight(offset: Int): Float {
        val textLayoutResult = layoutResultCallback() ?: return -1f
        val line = textLayoutResult.getLineForOffset(offset)
        return textLayoutResult.getLineRight(line)
    }

    override fun getCenterYForOffset(offset: Int): Float {
        val textLayoutResult = layoutResultCallback() ?: return -1f
        val line = textLayoutResult.getLineForOffset(offset)
        val top = textLayoutResult.getLineTop(line)
        val bottom = textLayoutResult.getLineBottom(line)
        return ((bottom - top) / 2) + top
    }

    override fun getRangeOfLineContaining(offset: Int): TextRange {
        val textLayoutResult = layoutResultCallback() ?: return TextRange.Zero
        val visibleTextLength = textLayoutResult.lastVisibleOffset
        if (visibleTextLength < 1) return TextRange.Zero
        val line = textLayoutResult.getLineForOffset(offset.coerceIn(0, visibleTextLength - 1))
        return TextRange(
            start = textLayoutResult.getLineStart(line),
            end = textLayoutResult.getLineEnd(line, visibleEnd = true)
        )
    }

    override fun getLastVisibleOffset(): Int {
        val textLayoutResult = layoutResultCallback() ?: return 0
        return textLayoutResult.lastVisibleOffset
    }
}

/**
 * Appends a [SelectableInfo] to this [SelectionLayoutBuilder].
 *
 * @param textLayoutResult the [TextLayoutResult] for the selectable
 * @param startPosition the position of the start handle if not being draggedor the drag position if
 *                      it is
 * @param endPosition  the position of the end handle if not being dragged or the drag position if
 *                     it is
 * @param previousHandlePosition the position of the previous handle
 * @param selectableId the selectableId for the selectable
 */
internal fun SelectionLayoutBuilder.appendSelectableInfo(
    textLayoutResult: TextLayoutResult,
    startPosition: Offset,
    endPosition: Offset,
    previousHandlePosition: Offset,
    selectableId: Long,
) {
    val bounds = Rect(
        0.0f,
        0.0f,
        textLayoutResult.size.width.toFloat(),
        textLayoutResult.size.height.toFloat()
    )

    val isSelected =
        SelectionMode.Vertical.isSelected(bounds, startPosition, endPosition)

    if (!isSelected) {
        return
    }

    val textLength = textLayoutResult.layoutInput.text.length
    val rawStartHandleOffset: Int
    val rawEndHandleOffset: Int
    if (isStartHandle) {
        rawStartHandleOffset = getOffsetForPosition(startPosition, textLayoutResult)
        rawEndHandleOffset = previousSelection?.end
            ?.getPreviousAdjustedOffset(selectableIdOrderingComparator, selectableId, textLength)
            ?: getOffsetForPosition(endPosition, textLayoutResult)
    } else {
        rawStartHandleOffset = previousSelection?.start
            ?.getPreviousAdjustedOffset(selectableIdOrderingComparator, selectableId, textLength)
            ?: getOffsetForPosition(startPosition, textLayoutResult)
        rawEndHandleOffset = getOffsetForPosition(endPosition, textLayoutResult)
    }

    val rawPreviousHandleOffset = if (previousHandlePosition.isUnspecified) -1 else {
        getOffsetForPosition(previousHandlePosition, textLayoutResult)
    }

    val startHandleDirection = getDirection(startPosition, bounds)
    val endHandleDirection = getDirection(endPosition, bounds)

    appendInfo(
        selectableId = selectableId,
        rawStartHandleOffset = rawStartHandleOffset,
        startHandleDirection = startHandleDirection,
        rawEndHandleOffset = rawEndHandleOffset,
        endHandleDirection = endHandleDirection,
        rawPreviousHandleOffset = rawPreviousHandleOffset,
        textLayoutResult = textLayoutResult,
    )
}

private fun Selection.AnchorInfo.getPreviousAdjustedOffset(
    selectableIdOrderingComparator: Comparator<Long>,
    currentSelectableId: Long,
    currentTextLength: Int
): Int {
    val compareResult = selectableIdOrderingComparator.compare(
        this.selectableId,
        currentSelectableId
    )

    return when {
        compareResult < 0 -> 0
        compareResult > 0 -> currentTextLength
        else -> offset
    }
}

private fun getDirection(position: Offset, bounds: Rect): Direction = when {
    position.y < bounds.top -> Direction.BEFORE
    position.y > bounds.bottom -> Direction.AFTER
    else -> Direction.ON
}

// map offsets above/below the text to 0/length respectively
private fun getOffsetForPosition(position: Offset, textLayoutResult: TextLayoutResult): Int = when {
    position.y <= 0f -> 0
    position.y >= textLayoutResult.multiParagraph.height -> textLayoutResult.layoutInput.text.length
    else -> textLayoutResult.getOffsetForPosition(position)
}
