/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.databinding.tool.store

import com.google.gson.FieldNamingPolicy
import com.google.gson.GsonBuilder
import com.google.gson.annotations.SerializedName
import java.io.File

/**
 * serializable class that holds the information about the code generated by BaseDataBinder.
 * We use it to make the task incremental.
 */
class LayoutInfoLog {
    @SerializedName("dependencies")
    // layout name A depends on Layouts B,C,D (not their classes, layout names)
    private val dependencies = sortedMapOf<String, MutableSet<String>>()
    @SerializedName("class_names_log")
    val classInfoLog = GenClassInfoLog()

    companion object {
        private val GSON = GsonBuilder()
                .disableHtmlEscaping()
                .setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES)
                .setPrettyPrinting().create()

        @JvmStatic
        fun fromFile(file: File): LayoutInfoLog {
            if (!file.exists()) {
                return LayoutInfoLog()
            }
            return file.reader(Charsets.UTF_16).use {
                GSON.fromJson(it, LayoutInfoLog::class.java)
            }
        }
    }

    /**
     * `fromLayout` depends on `toLayout` (e.g. `fromLayout` has an include tag that imports
     * `toLayout`).
     */
    fun addDependency(fromLayout: String, toLayout: String) {
        dependencies.getOrPut(fromLayout) {
            mutableSetOf()
        }.add(toLayout)
    }

    fun getDependencies(infoFileName: String) = dependencies[infoFileName] ?: emptyList<String>()

    fun serialize(file: File) {
        if (file.exists()) {
            file.delete()
        }
        file.writer(Charsets.UTF_16).use {
            GSON.toJson(this, it)
        }
    }

    fun getLayoutsThatDependOn(layouts: Set<String>): Set<String> {
        val result = mutableSetOf<String>()
        dependencies.forEach { from, toList ->
            if (toList.any { layouts.contains(it) }) {
                result.add(from)
            }
        }
        return result
    }

    fun addAll(other : LayoutInfoLog) {
        classInfoLog.addAll(other.classInfoLog)
        other.dependencies.forEach { key, value ->
            value.forEach {
                addDependency(key, it)
            }
        }
    }
}
