/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2002 - 2007 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package flex.management.runtime.messaging.services;

import flex.management.BaseControl;
import flex.messaging.services.HTTPProxyService;

import java.util.Date;

/**
 * @exclude
 * The <code>HTTPProxyServiceControl</code> class is the MBean implementation
 * for monitoring and managing a <code>HTTPProxyService</code> at runtime.
 * 
 * @author shodgson
 */
public class HTTPProxyServiceControl extends ServiceControl implements
        HTTPProxyServiceControlMBean
{
    private static final String TYPE = "HTTPProxyService";
    private int invokeSOAPCount = 0;
    private Date lastInvokeSOAPTimestamp;
    private long invokeSOAPStart;
    private int invokeHTTPCount = 0;
    private Date lastInvokeHTTPTimestamp;
    private long invokeHTTPStart;
    
    /**
     * Constructs a <code>HTTPProxyServiceControl</code>, assigning its id, managed
     * HTTP proxy service and parent MBean.
     * 
     * @param service The <code>HTTPProxyService</code> managed by this MBean.
     * @param parent The parent MBean in the management hierarchy.
     */
    public HTTPProxyServiceControl(HTTPProxyService service, BaseControl parent)
    {
        super(service, parent);
        invokeSOAPStart = System.currentTimeMillis();
        invokeHTTPStart = invokeSOAPStart;
        

    }

    
    /**
     * @exclude
     *  (non-Javadoc)
     * @see flex.management.BaseControlMBean#getType()
     */
    public String getType()
    {
        return TYPE;
    }
    
    /**
     * @exclude
     *  (non-Javadoc)
     * @see flex.management.runtime.HTTPProxyServiceControlMBean#getInvokeSOAPCount()
     */
    public Integer getInvokeSOAPCount()
    {
        return new Integer(invokeSOAPCount);
    }
    
    /**
     * @exclude
     *  (non-Javadoc)
     * @see flex.management.runtime.HTTPProxyServiceControlMBean#resetInvokeSOAPCount()
     */
    public void resetInvokeSOAPCount()
    {
        invokeSOAPStart = System.currentTimeMillis();
        invokeSOAPCount = 0;
        lastInvokeSOAPTimestamp = null;
    }
    
    /**
     * Increments the count of Soap invocations.
     */
    public void incrementInvokeSOAPCount()
    {
        ++invokeSOAPCount;
        lastInvokeSOAPTimestamp = new Date();
    }
    
    /**
     * @exclude
     *  (non-Javadoc)
     * @see flex.management.runtime.HTTPProxyServiceControlMBean#getLastInvokeSOAPTimestamp()
     */
    public Date getLastInvokeSOAPTimestamp()
    {
        return lastInvokeSOAPTimestamp;
    }
    
    /**
     *
     * @exclude
     *  (non-Javadoc)
     * @see flex.management.runtime.HTTPProxyServiceControlMBean#getInvokeSOAPFrequency()
     */
    public Double getInvokeSOAPFrequency()
    {
        if (invokeSOAPCount > 0)
        {
            double runtime = differenceInMinutes(invokeSOAPStart, System.currentTimeMillis());
            return new Double(invokeSOAPCount/runtime);
        }
        else
        {
            return new Double(0);
        }
    }
    
    /**
     * @exclude
     *  (non-Javadoc)
     * @see flex.management.runtime.HTTPProxyServiceControlMBean#getInvokeHTTPCount()
     */
    public Integer getInvokeHTTPCount()
    {
        return new Integer(invokeHTTPCount);
    }
    
    /**
     * @exclude
     *  (non-Javadoc)
     * @see flex.management.runtime.HTTPProxyServiceControlMBean#resetInvokeHTTPCount()
     */
    public void resetInvokeHTTPCount()
    {
        invokeHTTPStart = System.currentTimeMillis();
        invokeHTTPCount = 0;
        lastInvokeHTTPTimestamp = null;
    }
    
    /**
     * Increments the count of HTTP invocations.
     */
    public void incrementInvokeHTTPCount()
    {
        ++invokeHTTPCount;
        lastInvokeHTTPTimestamp = new Date();
    }
    
    /**
     * @exclude
     *  (non-Javadoc)
     * @see flex.management.runtime.HTTPProxyServiceControlMBean#getLastInvokeHTTPTimestamp()
     */
    public Date getLastInvokeHTTPTimestamp()
    {
        return lastInvokeHTTPTimestamp;
    }

    /**
     * @exclude
     *  (non-Javadoc)
     * @see flex.management.runtime.HTTPProxyServiceControlMBean#getInvokeHTTPFrequency()
     */
    public Double getInvokeHTTPFrequency()
    {
        if (invokeHTTPCount > 0)
        {
            double runtime = differenceInMinutes(invokeHTTPStart, System.currentTimeMillis());
            return new Double(invokeHTTPCount/runtime);
        }
        else
        {
            return new Double(0);
        }
    }
}
