/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.LinkedList;
import java.util.List;

import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.io.TimeRange;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.hadoop.hbase.util.Bytes;

@InterfaceAudience.Private
public class ClientUtil {


  public static boolean areScanStartRowAndStopRowEqual(byte[] startRow, byte[] stopRow) {
    return startRow != null && startRow.length > 0 && Bytes.equals(startRow, stopRow);
  }

  public static Cursor createCursor(byte[] row) {
    return new Cursor(row);
  }

  public static Pair<TableDescriptor,TableDescriptor> makeLayeredTableDescripter(TableDescriptor hot,
      TableDescriptor cold,long ttl) {
    TableDescriptor newHot = TableDescriptorBuilder.newBuilder(hot)
        .setValue(HConstants.STORE_ENGINE_CLASS_KEY,HConstants.TRANSFER_STOREENGINE_CLASS)
        .setValue(HConstants.SINK_TABLE_NAME,cold.getTableName().toString())
        .setValue(HConstants.SOURCE_TTL,String.valueOf(ttl))
        .build();
    TableDescriptor newCold = TableDescriptorBuilder.newBuilder(cold)
        .setValue(HConstants.SOURCE_TABLE_NAME,hot.getTableName().toString())
        .setValue(HConstants.HFILE_STORAGE_POLICY,HConstants.COLD_HFLLE)
        .setMaxFileSize(HConstants.COLD_REGION_MAX_FILE_SIZE)
        .setValue(HConstants.BLOCKING_STOREFILES_KEY,
            String.valueOf(HConstants.COLD_BLOCKING_STOREFILE_COUNT))
        .build();
    return new Pair<>(newHot,newCold);
  }

  ///helper methods for hot table query
  public static Scan removeColdDataFromScan(Scan scan,long hotTtl) throws IOException {
    long minTimeStamp = System.currentTimeMillis() - hotTtl * 1000;
    Scan hotScan = new Scan(scan);
    TimeRange tr = hotScan.getTimeRange();
    if (tr.getMin() < minTimeStamp) {
      hotScan.setTimeRange(minTimeStamp, tr.getMax());
    }
    return hotScan;
  }

  public static Get removeColdDataFromGet(Get get,long hotTtl) throws IOException {
    long minTimeStamp = System.currentTimeMillis() - hotTtl * 1000;
    Get hotGet = new Get(get);
    TimeRange tr = hotGet.getTimeRange();
    if (tr.getMin() < minTimeStamp) {
      hotGet.setTimeRange(minTimeStamp, tr.getMax());
    }
    return hotGet;
  }

  public static List<Get> removeColdDataFromGets(List<Get> gets,long hotTtl) throws IOException {
    List<Get> hotGets = new LinkedList<>();
    long minTimeStamp = System.currentTimeMillis() - hotTtl * 1000;
    for (Get get : gets) {
      Get hotGet = new Get(get);
      TimeRange tr = hotGet.getTimeRange();
      if (tr.getMin() < minTimeStamp) {
        hotGet.setTimeRange(minTimeStamp, tr.getMax());
      }
      hotGets.add(hotGet);
    }
    return hotGets;
  }

  ///helper methods for cold table query
  public static Scan makeHotTableScan(Scan scan,long hotTtl) throws IOException {
    long maxTimeStamp = System.currentTimeMillis() - hotTtl * 1000;
    Scan hotScan = new Scan(scan);
    TimeRange tr = hotScan.getTimeRange();
    if (tr.getMax() > maxTimeStamp) {
      hotScan.setTimeRange(tr.getMin(), maxTimeStamp);
    }
    return hotScan;
  }

  public static Get makeHotTableGet(Get get,long hotTtl) throws IOException {
    long maxTimeStamp = System.currentTimeMillis() - hotTtl * 1000;
    Get hotGet = new Get(get);
    TimeRange tr = hotGet.getTimeRange();
    if (tr.getMax() > maxTimeStamp) {
      hotGet.setTimeRange(tr.getMin(), maxTimeStamp);
    }
    return hotGet;
  }

  public static List<Get> makeHotTableGets(List<Get> gets,long hotTtl) throws IOException {
    List<Get> hotGets = new LinkedList<>();
    long maxTimeStamp = System.currentTimeMillis() - hotTtl * 1000;
    for (Get get : gets) {
      Get hotGet = new Get(get);
      TimeRange tr = hotGet.getTimeRange();
      if (tr.getMax() > maxTimeStamp) {
        hotGet.setTimeRange(tr.getMin(), maxTimeStamp);
      }
      hotGets.add(hotGet);
    }
    return hotGets;
  }

}
