/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import org.apache.hadoop.hbase.TableName;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;

/**
 * Created by mazhenlin on 2019/1/22.
 */
@InterfaceAudience.Private
@InterfaceStability.Stable
public class RawAsyncColdTableImpl extends RawAsyncLayeredTableImpl {
  protected RawAsyncLayeredTableImpl hotTable;
  protected long hotTtl = -1;

  RawAsyncColdTableImpl(AsyncConnectionImpl conn, AsyncTableBuilderBase<?> builder,
      TableName hotTableName, long hotTtl) {
    super(conn, builder);
    this.hotTtl = hotTtl;
    this.hotTable = (RawAsyncLayeredTableImpl) conn.getTable(hotTableName);
  }

  @Override public ResultScanner getScanner(Scan scan) {
    Scan hotScan = null;
    try {
      hotScan = ClientUtil.makeHotTableScan(scan,hotTtl);
    } catch (IOException ignore) {
      /// nothing indeed thrown
    }
    return new MergeScanner(scan, hotTable.getRawScanner(hotScan), getRawScanner(scan));
  }

  @Override public CompletableFuture<Result> get(Get get) {
    Get hotGet = null;
    try {
      hotGet = ClientUtil.makeHotTableGet(get, hotTtl);
    } catch (IOException ignore) {
      /// nothing indeed thrown
    }
    return rawGet(get).thenCombine(hotTable.rawGet(hotGet), (hotr, coldr) -> {
      return MergeScanner.mergeResultsOfSameRow(hotr, coldr, get.getMaxVersions());
    });
  }

  @Override public List<CompletableFuture<Result>> get(List<Get> gets) {
    List<Get> hotGets = null;
    try {
      hotGets = ClientUtil.makeHotTableGets(gets, hotTtl);
    } catch (IOException ignore) {
      /// nothing indeed thrown
    }

    List<CompletableFuture<Result>> r1 = rawGet(gets);
    // Translate.
    List<CompletableFuture<Result>> results = new ArrayList<>(r1.size());
    List<CompletableFuture<Result>> r2 = hotTable.rawGet(hotGets);
    for (int i = 0; i < gets.size(); i++) {
      int idx = i;
      CompletableFuture<Result> r = r1.get(i).thenCombine(r2.get(i), (rr1, rr2) -> {
        return MergeScanner.mergeResultsOfSameRow(rr1, rr2, gets.get(idx).getMaxVersions());
      });
      results.add(i, r);
    }
    return results;
  }

}
