/*
 * Decompiled with CFR 0.152.
 */
package com.android.tools.rendering.imagepool;

import com.android.tools.rendering.imagepool.DisposableImage;
import com.android.tools.rendering.imagepool.ImagePool;
import com.android.tools.rendering.imagepool.ImagePoolUtil;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.FinalizablePhantomReference;
import com.google.common.base.FinalizableReferenceQueue;
import com.google.common.base.Preconditions;
import com.google.common.collect.EvictingQueue;
import com.google.common.collect.ForwardingQueue;
import com.google.common.collect.Sets;
import com.intellij.openapi.diagnostic.Logger;
import java.awt.AlphaComposite;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GraphicsConfiguration;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferInt;
import java.awt.image.WritableRaster;
import java.lang.ref.Reference;
import java.lang.ref.SoftReference;
import java.util.Arrays;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.NoSuchElementException;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

class ImagePoolImpl
implements ImagePool {
    private static final Logger LOG = Logger.getInstance(ImagePoolImpl.class);
    private static final Bucket NULL_BUCKET = new Bucket();
    private final int[] myBucketSizes;
    private final HashMap<String, Bucket> myPool = new HashMap();
    private final IdentityHashMap<Bucket, BucketStatsImpl> myBucketStats = new IdentityHashMap();
    private final BiFunction<Integer, Integer, Function<Integer, Integer>> myBucketSizingPolicy;
    private final FinalizableReferenceQueue myFinalizableReferenceQueue = new FinalizableReferenceQueue();
    private final Set<Reference<?>> myReferences = Sets.newConcurrentHashSet();
    private final LongAdder myTotalAllocatedBytes = new LongAdder();
    private final LongAdder myTotalInUseBytes = new LongAdder();
    private final ImagePool.Stats myStats = new ImagePool.Stats(){

        @Override
        public long totalBytesAllocated() {
            return ImagePoolImpl.this.myTotalAllocatedBytes.sum();
        }

        @Override
        public long totalBytesInUse() {
            return ImagePoolImpl.this.myTotalInUseBytes.sum();
        }

        @Override
        public ImagePool.BucketStats[] getBucketStats() {
            return (ImagePool.BucketStats[])ImagePoolImpl.this.myBucketStats.values().stream().toArray(ImagePool.BucketStats[]::new);
        }
    };
    private boolean isDisposed = false;

    ImagePoolImpl(@NotNull int[] bucketSizes, @NotNull BiFunction<Integer, Integer, Function<Integer, Integer>> bucketSizingPolicy) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("New ImagePool " + Arrays.toString(bucketSizes));
        }
        this.myBucketSizes = bucketSizes;
        Arrays.sort(this.myBucketSizes);
        this.myBucketSizingPolicy = bucketSizingPolicy;
    }

    @NotNull
    private static String getPoolKey(int w2, int h2, int type) {
        return "" + w2 + 'x' + h2 + '-' + type;
    }

    @NotNull
    private Bucket getTypeBucket(int w2, int h2, int type) {
        if (this.myBucketSizingPolicy.apply(w2, h2).apply(type) == 0) {
            return NULL_BUCKET;
        }
        int widthBucket = -1;
        int heightBucket = -1;
        for (int bucketMinSize : this.myBucketSizes) {
            if (widthBucket == -1 && w2 < bucketMinSize) {
                widthBucket = bucketMinSize;
                if (heightBucket != -1) break;
            }
            if (heightBucket != -1 || h2 >= bucketMinSize) continue;
            heightBucket = bucketMinSize;
            if (widthBucket != -1) break;
        }
        if (widthBucket == -1 || heightBucket == -1) {
            return NULL_BUCKET;
        }
        String poolKey = ImagePoolImpl.getPoolKey(widthBucket, heightBucket, type);
        int finalWidthBucket = widthBucket;
        int finalHeightBucket = heightBucket;
        return this.myPool.computeIfAbsent(poolKey, k2 -> {
            int size = this.myBucketSizingPolicy.apply(finalWidthBucket, finalHeightBucket).apply(type);
            if (size == 0) {
                return NULL_BUCKET;
            }
            Bucket newBucket = new Bucket(finalWidthBucket, finalHeightBucket, size);
            this.myBucketStats.put(newBucket, new BucketStatsImpl(newBucket));
            return newBucket;
        });
    }

    @VisibleForTesting
    @NotNull
    ImageImpl create(final int w2, final int h2, final int type, final @Nullable Consumer<BufferedImage> freedCallback) {
        FinalizablePhantomReference<ImagePool.Image> reference;
        BufferedImage image;
        assert (!this.isDisposed) : "ImagePool already disposed";
        final Bucket bucket = this.getTypeBucket(w2, h2, type);
        final BucketStatsImpl bucketStats = this.myBucketStats.get(bucket);
        if (LOG.isDebugEnabled()) {
            LOG.debug(String.format("create(%dx%d-%d) in bucket (%dx%d) hasStats=%b\n", w2, h2, type, bucket.myMinWidth, bucket.myMinHeight, bucketStats != null));
        }
        try {
            Bucket.Element element = (Bucket.Element)bucket.remove();
            while ((image = element.get()) == null) {
                this.myTotalAllocatedBytes.add(-element.getImageEstimatedSize());
                element = (Bucket.Element)bucket.remove();
            }
            long totalSize = image.getWidth() * image.getHeight();
            if (bucketStats != null) {
                bucketStats.bucketHit();
            }
            if (LOG.isDebugEnabled()) {
                double wasted = totalSize - (long)(w2 * h2);
                LOG.debug(String.format("  Re-used image %dx%d - %d\n  pool buffer %dx%d\n  wasted %d%%\n", w2, h2, type, image.getWidth(), image.getHeight(), (int)(wasted / (double)totalSize * 100.0)));
            }
            this.myTotalInUseBytes.add(element.getImageEstimatedSize());
            if (image.getRaster().getDataBuffer().getDataType() == 3) {
                Arrays.fill(((DataBufferInt)image.getRaster().getDataBuffer()).getData(), 0);
            } else {
                Graphics2D g2 = image.createGraphics();
                g2.setComposite(AlphaComposite.Clear);
                g2.fillRect(0, 0, w2, h2);
                g2.dispose();
            }
        }
        catch (NoSuchElementException e2) {
            if (LOG.isDebugEnabled()) {
                LOG.debug(String.format("  New image %dx%d - %d\n", w2, h2, type));
            }
            if (bucketStats != null) {
                bucketStats.bucketMiss();
            }
            int newImageWidth = Math.max(bucket.myMinWidth, w2);
            int newImageHeight = Math.max(bucket.myMinHeight, h2);
            image = new BufferedImage(newImageWidth, newImageHeight, type);
            image.setAccelerationPriority(0.9f);
            long estimatedSize = newImageWidth * newImageHeight * 4;
            this.myTotalAllocatedBytes.add(estimatedSize);
            this.myTotalInUseBytes.add(estimatedSize);
        }
        ImageImpl pooledImage = new ImageImpl(w2, h2, image);
        final BufferedImage imagePointer = image;
        pooledImage.myOwnReference = reference = new FinalizablePhantomReference<ImagePool.Image>((ImagePool.Image)pooledImage, this.myFinalizableReferenceQueue){

            @Override
            public void finalizeReferent() {
                if (ImagePoolImpl.this.myReferences.remove(this)) {
                    Bucket.Element element = new Bucket.Element(imagePointer);
                    boolean accepted = bucket.offer(element);
                    if (bucketStats != null) {
                        if (accepted) {
                            bucketStats.returnedImageAccepted();
                        } else {
                            bucketStats.returnedImageRejected();
                        }
                    }
                    if (LOG.isDebugEnabled()) {
                        LOG.debug(String.format("%s image (%dx%d-%d) in bucket (%dx%d)\n", accepted ? "Released" : "Rejected", w2, h2, type, bucket.myMinWidth, bucket.myMinHeight));
                    }
                    if (!accepted) {
                        ImagePoolImpl.this.myTotalAllocatedBytes.add(-element.getImageEstimatedSize());
                    }
                    ImagePoolImpl.this.myTotalInUseBytes.add(-element.getImageEstimatedSize());
                    if (freedCallback != null) {
                        freedCallback.accept(imagePointer);
                    }
                }
            }
        };
        this.myReferences.add(reference);
        return pooledImage;
    }

    @Override
    @NotNull
    public ImagePool.Image create(int w2, int h2, int type) {
        return this.create(w2, h2, type, null);
    }

    @Override
    @NotNull
    public ImagePool.Image copyOf(@Nullable BufferedImage origin2) {
        if (origin2 == null) {
            return NULL_POOLED_IMAGE;
        }
        int w2 = origin2.getWidth();
        int h2 = origin2.getHeight();
        int type = origin2.getType();
        ImageImpl image = this.create(w2, h2, type, null);
        image.drawFrom(origin2);
        return image;
    }

    @Override
    @Nullable
    public ImagePool.Stats getStats() {
        return this.myStats;
    }

    @Override
    public void dispose() {
        this.isDisposed = true;
        this.myFinalizableReferenceQueue.close();
        this.myReferences.clear();
        this.myPool.clear();
    }

    private static class Bucket
    extends ForwardingQueue<Element> {
        private final Queue<Element> myDelegate;
        private final int myMinWidth;
        private final int myMinHeight;
        private final int myMaxSize;

        Bucket(int minWidth, int minHeight, int maxSize) {
            Preconditions.checkArgument(maxSize > 0);
            this.myMinWidth = minWidth;
            this.myMinHeight = minHeight;
            this.myMaxSize = maxSize;
            this.myDelegate = new ArrayBlockingQueue<Element>(maxSize);
        }

        Bucket() {
            this.myMinWidth = 0;
            this.myMinHeight = 0;
            this.myMaxSize = 0;
            this.myDelegate = EvictingQueue.create(0);
        }

        @Override
        protected Queue<Element> delegate() {
            return this.myDelegate;
        }

        int getMaxSize() {
            return this.myMaxSize;
        }

        private static class Element {
            private final long myImageEstimatedSize;
            private final SoftReference<BufferedImage> myReference;

            private Element(@NotNull BufferedImage image) {
                this.myImageEstimatedSize = image.getWidth() * image.getHeight() * 4;
                this.myReference = new SoftReference<BufferedImage>(image);
            }

            @Nullable
            private BufferedImage get() {
                return this.myReference.get();
            }

            private long getImageEstimatedSize() {
                return this.myImageEstimatedSize;
            }
        }
    }

    private static final class BucketStatsImpl
    implements ImagePool.BucketStats {
        private final Bucket myBucket;
        private final AtomicLong myLastAccessMs = new AtomicLong(System.currentTimeMillis());
        private final AtomicLong myBucketMiss = new AtomicLong(0L);
        private final AtomicLong myBucketHit = new AtomicLong(0L);
        private final AtomicLong myBucketFull = new AtomicLong(0L);
        private final AtomicLong myBucketHadSpace = new AtomicLong(0L);

        BucketStatsImpl(@NotNull Bucket bucket) {
            this.myBucket = bucket;
        }

        @Override
        public int getMinWidth() {
            return this.myBucket.myMinWidth;
        }

        @Override
        public int getMinHeight() {
            return this.myBucket.myMinHeight;
        }

        @Override
        public int maxSize() {
            return this.myBucket.getMaxSize();
        }

        @Override
        public long getLastAccessTimeMs() {
            return this.myLastAccessMs.get();
        }

        @Override
        public long bucketHits() {
            return this.myBucketHit.get();
        }

        @Override
        public long bucketMisses() {
            return this.myBucketMiss.get();
        }

        @Override
        public long bucketWasFull() {
            return this.myBucketFull.get();
        }

        @Override
        public long imageWasReturned() {
            return this.myBucketHadSpace.get();
        }

        void bucketHit() {
            this.myLastAccessMs.set(System.currentTimeMillis());
            this.myBucketHit.incrementAndGet();
        }

        void bucketMiss() {
            this.myLastAccessMs.set(System.currentTimeMillis());
            this.myBucketMiss.incrementAndGet();
        }

        void returnedImageAccepted() {
            this.myBucketHadSpace.incrementAndGet();
        }

        void returnedImageRejected() {
            this.myBucketFull.incrementAndGet();
        }
    }

    static class ImageImpl
    implements ImagePool.Image,
    DisposableImage {
        private static final boolean ourTrackDisposeCall = ImageImpl.class.desiredAssertionStatus();
        private FinalizablePhantomReference<ImagePool.Image> myOwnReference = null;
        private ReadWriteLock myLock = new ReentrantReadWriteLock();
        private StackTraceElement[] myDisposeStackTrace = null;
        @VisibleForTesting
        @Nullable
        BufferedImage myBuffer;
        final int myWidth;
        final int myHeight;

        private ImageImpl(int w2, int h2, @NotNull BufferedImage image) {
            assert (w2 <= image.getWidth() && h2 <= image.getHeight());
            this.myWidth = w2;
            this.myHeight = h2;
            this.myBuffer = image;
        }

        private void assertIfDisposed() {
            if (this.myDisposeStackTrace != null) {
                LOG.warn("Accessing already disposed image\nDispose trace: \n" + ImagePoolUtil.stackTraceToAssertionString(this.myDisposeStackTrace));
            }
        }

        @Override
        public int getWidth() {
            return this.myWidth;
        }

        @Override
        public int getHeight() {
            return this.myHeight;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void drawImageTo(@NotNull Graphics g2, int dx1, int dy1, int dx2, int dy2, int sx1, int sy1, int sx2, int sy2) {
            this.assertIfDisposed();
            this.myLock.readLock().lock();
            try {
                g2.drawImage(this.myBuffer, dx1, dy1, dx2, dy2, sx1, sy1, sx2, sy2, null);
            }
            finally {
                this.myLock.readLock().unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void paint(@NotNull Consumer<Graphics2D> command2) {
            this.assertIfDisposed();
            this.myLock.readLock().lock();
            try {
                Graphics2D g2 = this.myBuffer.createGraphics();
                try {
                    command2.accept(g2);
                }
                finally {
                    g2.dispose();
                }
            }
            finally {
                this.myLock.readLock().unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        @Nullable
        public BufferedImage getCopy(@Nullable GraphicsConfiguration gc, int x2, int y2, int w2, int h2) {
            this.myLock.readLock().lock();
            if (this.myBuffer == null) {
                LOG.debug("getCopy for already disposed image");
                return null;
            }
            try {
                if (x2 + w2 > this.myWidth) {
                    throw new IndexOutOfBoundsException(String.format("x (%d) + y (%d) is out bounds (image width is = %d)", x2, y2, this.myWidth));
                }
                if (y2 + h2 > this.myHeight) {
                    throw new IndexOutOfBoundsException(String.format("y (%d) + h (%d) is out bounds (image height is = %d)", y2, h2, this.myHeight));
                }
                BufferedImage newImage = gc != null ? gc.createCompatibleImage(w2, h2) : new BufferedImage(w2, h2, this.myBuffer.getType());
                Graphics2D g2 = newImage.createGraphics();
                try {
                    g2.drawImage(this.myBuffer, 0, 0, w2, h2, x2, y2, x2 + w2, y2 + h2, null);
                }
                finally {
                    g2.dispose();
                }
                BufferedImage bufferedImage = newImage;
                return bufferedImage;
            }
            finally {
                this.myLock.readLock().unlock();
            }
        }

        @Override
        @Nullable
        public BufferedImage getCopy() {
            this.myLock.readLock().lock();
            if (this.myBuffer == null) {
                LOG.debug("getCopy for already disposed image");
                return null;
            }
            try {
                WritableRaster raster = this.myBuffer.copyData(this.myBuffer.getRaster().createCompatibleWritableRaster(0, 0, this.myWidth, this.myHeight));
                BufferedImage bufferedImage = new BufferedImage(this.myBuffer.getColorModel(), raster, this.myBuffer.isAlphaPremultiplied(), null);
                return bufferedImage;
            }
            finally {
                this.myLock.readLock().unlock();
            }
        }

        @Override
        public void dispose() {
            this.assertIfDisposed();
            this.myLock.writeLock().lock();
            if (ourTrackDisposeCall) {
                this.myDisposeStackTrace = Thread.currentThread().getStackTrace();
            }
            try {
                this.myBuffer = null;
                if (this.myOwnReference != null) {
                    this.myOwnReference.finalizeReferent();
                }
            }
            finally {
                this.myLock.writeLock().unlock();
            }
        }

        @Override
        public boolean isValid() {
            this.myLock.readLock().lock();
            try {
                boolean bl2 = this.myBuffer != null;
                return bl2;
            }
            finally {
                this.myLock.readLock().unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void drawFrom(@NotNull BufferedImage origin2) {
            this.assertIfDisposed();
            this.myLock.readLock().lock();
            try {
                Graphics g2 = this.myBuffer.getGraphics();
                try {
                    g2.drawImage(origin2, 0, 0, null);
                }
                finally {
                    g2.dispose();
                }
            }
            finally {
                this.myLock.readLock().unlock();
            }
        }
    }
}

