/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.io;

import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.J2ktIncompatible;
import com.google.common.base.Preconditions;
import com.google.common.io.ByteArrayDataInput;
import com.google.common.io.ByteArrayDataOutput;
import com.google.common.io.ByteProcessor;
import com.google.common.io.ElementTypesAreNonnullByDefault;
import com.google.common.io.Java8Compatibility;
import com.google.common.io.ParametricNullness;
import com.google.common.math.IntMath;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.Queue;
import javax.annotation.CheckForNull;

@ElementTypesAreNonnullByDefault
@J2ktIncompatible
@GwtIncompatible
public final class ByteStreams {
    private static final int BUFFER_SIZE = 8192;
    private static final int ZERO_COPY_CHUNK_SIZE = 524288;
    private static final int MAX_ARRAY_LEN = 0x7FFFFFF7;
    private static final int TO_BYTE_ARRAY_DEQUE_SIZE = 20;
    private static final OutputStream NULL_OUTPUT_STREAM = new OutputStream(){

        @Override
        public void write(int b2) {
        }

        @Override
        public void write(byte[] b2) {
            Preconditions.checkNotNull(b2);
        }

        @Override
        public void write(byte[] b2, int off, int len) {
            Preconditions.checkNotNull(b2);
            Preconditions.checkPositionIndexes(off, off + len, b2.length);
        }

        public String toString() {
            return "ByteStreams.nullOutputStream()";
        }
    };

    static byte[] createBuffer() {
        return new byte[8192];
    }

    private ByteStreams() {
    }

    @CanIgnoreReturnValue
    public static long copy(InputStream from2, OutputStream to) throws IOException {
        int r2;
        Preconditions.checkNotNull(from2);
        Preconditions.checkNotNull(to);
        byte[] buf = ByteStreams.createBuffer();
        long total = 0L;
        while ((r2 = from2.read(buf)) != -1) {
            to.write(buf, 0, r2);
            total += (long)r2;
        }
        return total;
    }

    @CanIgnoreReturnValue
    public static long copy(ReadableByteChannel from2, WritableByteChannel to) throws IOException {
        Preconditions.checkNotNull(from2);
        Preconditions.checkNotNull(to);
        if (from2 instanceof FileChannel) {
            long copied;
            long oldPosition;
            FileChannel sourceChannel = (FileChannel)from2;
            long position = oldPosition = sourceChannel.position();
            do {
                copied = sourceChannel.transferTo(position, 524288L, to);
                sourceChannel.position(position += copied);
            } while (copied > 0L || position < sourceChannel.size());
            return position - oldPosition;
        }
        ByteBuffer buf = ByteBuffer.wrap(ByteStreams.createBuffer());
        long total = 0L;
        while (from2.read(buf) != -1) {
            Java8Compatibility.flip(buf);
            while (buf.hasRemaining()) {
                total += (long)to.write(buf);
            }
            Java8Compatibility.clear(buf);
        }
        return total;
    }

    private static byte[] toByteArrayInternal(InputStream in, Queue<byte[]> bufs, int totalLen) throws IOException {
        int initialBufferSize;
        int bufSize = initialBufferSize = Math.min(8192, Math.max(128, Integer.highestOneBit(totalLen) * 2));
        while (totalLen < 0x7FFFFFF7) {
            byte[] buf = new byte[Math.min(bufSize, 0x7FFFFFF7 - totalLen)];
            bufs.add(buf);
            int off = 0;
            while (off < buf.length) {
                int r2 = in.read(buf, off, buf.length - off);
                if (r2 == -1) {
                    return ByteStreams.combineBuffers(bufs, totalLen);
                }
                off += r2;
                totalLen += r2;
            }
            bufSize = IntMath.saturatedMultiply(bufSize, bufSize < 4096 ? 4 : 2);
        }
        if (in.read() == -1) {
            return ByteStreams.combineBuffers(bufs, 0x7FFFFFF7);
        }
        throw new OutOfMemoryError("input is too large to fit in a byte array");
    }

    private static byte[] combineBuffers(Queue<byte[]> bufs, int totalLen) {
        if (bufs.isEmpty()) {
            return new byte[0];
        }
        byte[] result2 = bufs.remove();
        if (result2.length == totalLen) {
            return result2;
        }
        int remaining = totalLen - result2.length;
        result2 = Arrays.copyOf(result2, totalLen);
        while (remaining > 0) {
            byte[] buf = bufs.remove();
            int bytesToCopy = Math.min(remaining, buf.length);
            int resultOffset = totalLen - remaining;
            System.arraycopy(buf, 0, result2, resultOffset, bytesToCopy);
            remaining -= bytesToCopy;
        }
        return result2;
    }

    public static byte[] toByteArray(InputStream in) throws IOException {
        Preconditions.checkNotNull(in);
        return ByteStreams.toByteArrayInternal(in, new ArrayDeque<byte[]>(20), 0);
    }

    static byte[] toByteArray(InputStream in, long expectedSize) throws IOException {
        int read2;
        Preconditions.checkArgument(expectedSize >= 0L, "expectedSize (%s) must be non-negative", expectedSize);
        if (expectedSize > 0x7FFFFFF7L) {
            throw new OutOfMemoryError(expectedSize + " bytes is too large to fit in a byte array");
        }
        byte[] bytes2 = new byte[(int)expectedSize];
        for (int remaining = (int)expectedSize; remaining > 0; remaining -= read2) {
            int off = (int)expectedSize - remaining;
            read2 = in.read(bytes2, off, remaining);
            if (read2 != -1) continue;
            return Arrays.copyOf(bytes2, off);
        }
        int b2 = in.read();
        if (b2 == -1) {
            return bytes2;
        }
        ArrayDeque<byte[]> bufs = new ArrayDeque<byte[]>(22);
        bufs.add(bytes2);
        bufs.add(new byte[]{(byte)b2});
        return ByteStreams.toByteArrayInternal(in, bufs, bytes2.length + 1);
    }

    @CanIgnoreReturnValue
    public static long exhaust(InputStream in) throws IOException {
        long read2;
        long total = 0L;
        byte[] buf = ByteStreams.createBuffer();
        while ((read2 = (long)in.read(buf)) != -1L) {
            total += read2;
        }
        return total;
    }

    public static ByteArrayDataInput newDataInput(byte[] bytes2) {
        return ByteStreams.newDataInput(new ByteArrayInputStream(bytes2));
    }

    public static ByteArrayDataInput newDataInput(byte[] bytes2, int start) {
        Preconditions.checkPositionIndex(start, bytes2.length);
        return ByteStreams.newDataInput(new ByteArrayInputStream(bytes2, start, bytes2.length - start));
    }

    public static ByteArrayDataInput newDataInput(ByteArrayInputStream byteArrayInputStream) {
        return new ByteArrayDataInputStream(Preconditions.checkNotNull(byteArrayInputStream));
    }

    public static ByteArrayDataOutput newDataOutput() {
        return ByteStreams.newDataOutput(new ByteArrayOutputStream());
    }

    public static ByteArrayDataOutput newDataOutput(int size) {
        if (size < 0) {
            throw new IllegalArgumentException(String.format("Invalid size: %s", size));
        }
        return ByteStreams.newDataOutput(new ByteArrayOutputStream(size));
    }

    public static ByteArrayDataOutput newDataOutput(ByteArrayOutputStream byteArrayOutputStream) {
        return new ByteArrayDataOutputStream(Preconditions.checkNotNull(byteArrayOutputStream));
    }

    public static OutputStream nullOutputStream() {
        return NULL_OUTPUT_STREAM;
    }

    public static InputStream limit(InputStream in, long limit2) {
        return new LimitedInputStream(in, limit2);
    }

    public static void readFully(InputStream in, byte[] b2) throws IOException {
        ByteStreams.readFully(in, b2, 0, b2.length);
    }

    public static void readFully(InputStream in, byte[] b2, int off, int len) throws IOException {
        int read2 = ByteStreams.read(in, b2, off, len);
        if (read2 != len) {
            throw new EOFException("reached end of stream after reading " + read2 + " bytes; " + len + " bytes expected");
        }
    }

    public static void skipFully(InputStream in, long n2) throws IOException {
        long skipped = ByteStreams.skipUpTo(in, n2);
        if (skipped < n2) {
            throw new EOFException("reached end of stream after skipping " + skipped + " bytes; " + n2 + " bytes expected");
        }
    }

    static long skipUpTo(InputStream in, long n2) throws IOException {
        long totalSkipped;
        long skipped;
        byte[] buf = null;
        for (totalSkipped = 0L; totalSkipped < n2; totalSkipped += skipped) {
            long remaining = n2 - totalSkipped;
            skipped = ByteStreams.skipSafely(in, remaining);
            if (skipped != 0L) continue;
            int skip = (int)Math.min(remaining, 8192L);
            if (buf == null) {
                buf = new byte[skip];
            }
            if ((skipped = (long)in.read(buf, 0, skip)) == -1L) break;
        }
        return totalSkipped;
    }

    private static long skipSafely(InputStream in, long n2) throws IOException {
        int available = in.available();
        return available == 0 ? 0L : in.skip(Math.min((long)available, n2));
    }

    @ParametricNullness
    @CanIgnoreReturnValue
    public static <T> T readBytes(InputStream input2, ByteProcessor<T> processor) throws IOException {
        int read2;
        Preconditions.checkNotNull(input2);
        Preconditions.checkNotNull(processor);
        byte[] buf = ByteStreams.createBuffer();
        while ((read2 = input2.read(buf)) != -1 && processor.processBytes(buf, 0, read2)) {
        }
        return processor.getResult();
    }

    @CanIgnoreReturnValue
    public static int read(InputStream in, byte[] b2, int off, int len) throws IOException {
        int total;
        int result2;
        Preconditions.checkNotNull(in);
        Preconditions.checkNotNull(b2);
        if (len < 0) {
            throw new IndexOutOfBoundsException(String.format("len (%s) cannot be negative", len));
        }
        Preconditions.checkPositionIndexes(off, off + len, b2.length);
        for (total = 0; total < len && (result2 = in.read(b2, off + total, len - total)) != -1; total += result2) {
        }
        return total;
    }

    private static final class LimitedInputStream
    extends FilterInputStream {
        private long left;
        private long mark = -1L;

        LimitedInputStream(InputStream in, long limit2) {
            super(in);
            Preconditions.checkNotNull(in);
            Preconditions.checkArgument(limit2 >= 0L, "limit must be non-negative");
            this.left = limit2;
        }

        @Override
        public int available() throws IOException {
            return (int)Math.min((long)this.in.available(), this.left);
        }

        @Override
        public synchronized void mark(int readLimit) {
            this.in.mark(readLimit);
            this.mark = this.left;
        }

        @Override
        public int read() throws IOException {
            if (this.left == 0L) {
                return -1;
            }
            int result2 = this.in.read();
            if (result2 != -1) {
                --this.left;
            }
            return result2;
        }

        @Override
        public int read(byte[] b2, int off, int len) throws IOException {
            if (this.left == 0L) {
                return -1;
            }
            int result2 = this.in.read(b2, off, len = (int)Math.min((long)len, this.left));
            if (result2 != -1) {
                this.left -= (long)result2;
            }
            return result2;
        }

        @Override
        public synchronized void reset() throws IOException {
            if (!this.in.markSupported()) {
                throw new IOException("Mark not supported");
            }
            if (this.mark == -1L) {
                throw new IOException("Mark not set");
            }
            this.in.reset();
            this.left = this.mark;
        }

        @Override
        public long skip(long n2) throws IOException {
            n2 = Math.min(n2, this.left);
            long skipped = this.in.skip(n2);
            this.left -= skipped;
            return skipped;
        }
    }

    private static class ByteArrayDataOutputStream
    implements ByteArrayDataOutput {
        final DataOutput output;
        final ByteArrayOutputStream byteArrayOutputStream;

        ByteArrayDataOutputStream(ByteArrayOutputStream byteArrayOutputStream) {
            this.byteArrayOutputStream = byteArrayOutputStream;
            this.output = new DataOutputStream(byteArrayOutputStream);
        }

        @Override
        public void write(int b2) {
            try {
                this.output.write(b2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void write(byte[] b2) {
            try {
                this.output.write(b2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void write(byte[] b2, int off, int len) {
            try {
                this.output.write(b2, off, len);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeBoolean(boolean v2) {
            try {
                this.output.writeBoolean(v2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeByte(int v2) {
            try {
                this.output.writeByte(v2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeBytes(String s2) {
            try {
                this.output.writeBytes(s2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeChar(int v2) {
            try {
                this.output.writeChar(v2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeChars(String s2) {
            try {
                this.output.writeChars(s2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeDouble(double v2) {
            try {
                this.output.writeDouble(v2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeFloat(float v2) {
            try {
                this.output.writeFloat(v2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeInt(int v2) {
            try {
                this.output.writeInt(v2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeLong(long v2) {
            try {
                this.output.writeLong(v2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeShort(int v2) {
            try {
                this.output.writeShort(v2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public void writeUTF(String s2) {
            try {
                this.output.writeUTF(s2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public byte[] toByteArray() {
            return this.byteArrayOutputStream.toByteArray();
        }
    }

    private static class ByteArrayDataInputStream
    implements ByteArrayDataInput {
        final DataInput input;

        ByteArrayDataInputStream(ByteArrayInputStream byteArrayInputStream) {
            this.input = new DataInputStream(byteArrayInputStream);
        }

        @Override
        public void readFully(byte[] b2) {
            try {
                this.input.readFully(b2);
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }

        @Override
        public void readFully(byte[] b2, int off, int len) {
            try {
                this.input.readFully(b2, off, len);
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }

        @Override
        public int skipBytes(int n2) {
            try {
                return this.input.skipBytes(n2);
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }

        @Override
        public boolean readBoolean() {
            try {
                return this.input.readBoolean();
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }

        @Override
        public byte readByte() {
            try {
                return this.input.readByte();
            }
            catch (EOFException e2) {
                throw new IllegalStateException(e2);
            }
            catch (IOException impossible) {
                throw new AssertionError((Object)impossible);
            }
        }

        @Override
        public int readUnsignedByte() {
            try {
                return this.input.readUnsignedByte();
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }

        @Override
        public short readShort() {
            try {
                return this.input.readShort();
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }

        @Override
        public int readUnsignedShort() {
            try {
                return this.input.readUnsignedShort();
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }

        @Override
        public char readChar() {
            try {
                return this.input.readChar();
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }

        @Override
        public int readInt() {
            try {
                return this.input.readInt();
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }

        @Override
        public long readLong() {
            try {
                return this.input.readLong();
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }

        @Override
        public float readFloat() {
            try {
                return this.input.readFloat();
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }

        @Override
        public double readDouble() {
            try {
                return this.input.readDouble();
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }

        @Override
        @CheckForNull
        public String readLine() {
            try {
                return this.input.readLine();
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }

        @Override
        public String readUTF() {
            try {
                return this.input.readUTF();
            }
            catch (IOException e2) {
                throw new IllegalStateException(e2);
            }
        }
    }
}

