/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.tools.lint.checks

import com.android.SdkConstants.ATTR_VALUE
import com.android.tools.lint.client.api.LintClient
import com.android.tools.lint.detector.api.AnnotationInfo
import com.android.tools.lint.detector.api.AnnotationOrigin
import com.android.tools.lint.detector.api.AnnotationUsageInfo
import com.android.tools.lint.detector.api.AnnotationUsageType
import com.android.tools.lint.detector.api.AnnotationUsageType.ASSIGNMENT_LHS
import com.android.tools.lint.detector.api.AnnotationUsageType.ASSIGNMENT_RHS
import com.android.tools.lint.detector.api.Category
import com.android.tools.lint.detector.api.Implementation
import com.android.tools.lint.detector.api.Issue
import com.android.tools.lint.detector.api.JavaContext
import com.android.tools.lint.detector.api.Location
import com.android.tools.lint.detector.api.Scope
import com.android.tools.lint.detector.api.Severity
import com.android.tools.lint.detector.api.SourceCodeScanner
import com.android.tools.lint.detector.api.UastLintUtils
import com.android.tools.lint.detector.api.isKotlin
import com.intellij.lang.jvm.annotation.JvmAnnotationConstantValue
import com.intellij.psi.PsiClass
import com.intellij.psi.PsiCompiledElement
import com.intellij.psi.PsiField
import com.intellij.psi.PsiLiteralExpression
import com.intellij.psi.PsiMember
import com.intellij.psi.PsiMethod
import com.intellij.psi.PsiModifierListOwner
import com.intellij.psi.impl.compiled.ClsAnnotationImpl
import com.intellij.psi.util.PsiTypesUtil
import org.jetbrains.uast.UAnnotated
import org.jetbrains.uast.UAnnotation
import org.jetbrains.uast.UCallExpression
import org.jetbrains.uast.UClass
import org.jetbrains.uast.UElement
import org.jetbrains.uast.UExpression
import org.jetbrains.uast.ULiteralExpression
import org.jetbrains.uast.UReferenceExpression
import org.jetbrains.uast.UTypeReferenceExpression
import org.jetbrains.uast.UastEmptyExpression
import org.jetbrains.uast.getContainingUFile
import org.jetbrains.uast.getParentOfType
import org.jetbrains.uast.java.UnknownJavaExpression
import org.jetbrains.uast.util.isArrayInitializer

class RestrictToDetector : AbstractAnnotationDetector(), SourceCodeScanner {
    override fun applicableAnnotations(): List<String> = listOf(
        RESTRICT_TO_ANNOTATION.oldName(),
        RESTRICT_TO_ANNOTATION.newName(),
        GMS_HIDE_ANNOTATION,
        VISIBLE_FOR_TESTING_ANNOTATION.oldName(),
        VISIBLE_FOR_TESTING_ANNOTATION.newName(),
        GUAVA_VISIBLE_FOR_TESTING
    )

    override fun inheritAnnotation(annotation: String): Boolean {
        // Require restriction annotations to be annotated everywhere
        return false
    }

    override fun isApplicableAnnotationUsage(type: AnnotationUsageType): Boolean {
        return type != ASSIGNMENT_LHS && type != ASSIGNMENT_RHS &&
            super.isApplicableAnnotationUsage(type)
    }

    override fun visitAnnotationUsage(
        context: JavaContext,
        element: UElement,
        annotationInfo: AnnotationInfo,
        usageInfo: AnnotationUsageInfo
    ) {
        val type = usageInfo.type
        if (type == AnnotationUsageType.EXTENDS && element is UTypeReferenceExpression) {
            // If it's a constructor reference we don't need to also check the type
            // reference. Ideally we'd do a "parent is KtConstructorCalleeExpression"
            // here, but that points to impl classes in its hierarchy which leads to
            // class loading trouble.
            val sourcePsi = element.sourcePsi
            if (isKotlin(sourcePsi) && sourcePsi?.parent?.toString() == "CONSTRUCTOR_CALLEE") {
                return
            }
        }

        val member = usageInfo.referenced as? PsiMember
        val annotation = annotationInfo.annotation
        when (annotationInfo.qualifiedName) {
            RESTRICT_TO_ANNOTATION.oldName(), RESTRICT_TO_ANNOTATION.newName() -> {
                checkRestrictTo(
                    context, element, member, annotation, usageInfo, true
                )
            }
            GMS_HIDE_ANNOTATION -> {
                val method = member as? PsiMethod
                val isConstructor = method == null || method.isConstructor
                val isStatic = if (method == null) false else context.evaluator.isStatic(method)
                checkRestrictTo(
                    context, element, method, annotation, usageInfo, isConstructor || isStatic
                )
            }
            VISIBLE_FOR_TESTING_ANNOTATION.oldName(), VISIBLE_FOR_TESTING_ANNOTATION.newName(),
            GUAVA_VISIBLE_FOR_TESTING -> {
                if (member != null && type != AnnotationUsageType.METHOD_OVERRIDE && type != AnnotationUsageType.METHOD_CALL_PARAMETER) {
                    checkVisibleForTesting(
                        context,
                        element,
                        member,
                        annotation,
                        usageInfo
                    )
                }
            }
        }
    }

    /**
     * Returns true if we should report usages of APIs annotated with
     * com.google.android.gms.common.internal.Hide.
     */
    private fun flagHide(
        context: JavaContext,
        containingClass: PsiClass,
        member: PsiMember?,
        node: UElement
    ): Boolean {
        if (isGmsContext(context, node)) return false
        return !isAllowedFirstParty(context, containingClass, member)
    }

    /**
     * Is the annotated @Hide element *also* annotated with
     * @ShowFirstParty, *and* we're in a context where @ShowFirstParty
     * is enabled (e.g. g3).
     */
    private fun isAllowedFirstParty(
        context: JavaContext,
        containingClass: PsiClass,
        member: PsiMember?
    ): Boolean {
        if (LintClient.isGradle) return false
        val binDir = System.getProperty("com.android.tools.lint.bindir") ?: return false
        if (!binDir.contains("third_party/java/android/")) return false

        val evaluator = context.evaluator
        fun PsiModifierListOwner.isFirstParty(): Boolean {
            evaluator.findAnnotationInHierarchy(this, GMS_SHOW_FIRST_PARTY_ANNOTATION)
                ?: return false
            return true
        }

        if (member !== containingClass && member?.isFirstParty() == true) {
            return true
        }
        return containingClass.isFirstParty()
    }

    /** Checks whether the client code is in the GMS codebase. */
    private fun isGmsContext(
        context: JavaContext,
        element: UElement
    ): Boolean {
        val evaluator = context.evaluator
        val pkg = evaluator.getPackage(element) ?: return false

        val qualifiedName = pkg.qualifiedName
        if (!qualifiedName.startsWith("com.google.")) {
            return false
        }

        return qualifiedName.startsWith("com.google.firebase") ||
            qualifiedName.startsWith("com.google.android.gms") ||
            qualifiedName.startsWith("com.google.ads") ||
            qualifiedName.startsWith("com.google.mlkit")
    }

    private fun isTestContext(
        context: JavaContext,
        element: UElement
    ): Boolean {
        var current = element
        // (1) Is this compilation unit in a test source path?
        if (context.isTestSource) {
            return true
        }

        // (2) Is this AST node surrounded by a test-only annotation?
        while (true) {
            val owner = current.getParentOfType<UAnnotated>(true) ?: break

            //noinspection AndroidLintExternalAnnotations
            for (annotation in owner.uAnnotations) {
                val name = annotation.qualifiedName
                if (RESTRICT_TO_ANNOTATION.isEquals(name)) {
                    val restrictionScope = getRestrictionScope(annotation)
                    if (restrictionScope and RESTRICT_TO_TESTS != 0) {
                        return true
                    }
                } else if (VISIBLE_FOR_TESTING_ANNOTATION.isEquals(name) || GUAVA_VISIBLE_FOR_TESTING == name) {
                    return true
                }
            }

            current = owner
        }

        return false
    }

    private fun checkVisibleForTesting(
        context: JavaContext,
        node: UElement,
        method: PsiMember,
        annotation: UAnnotation,
        usageInfo: AnnotationUsageInfo
    ) {

        val visibility = getVisibilityForTesting(annotation)
        if (visibility == VISIBILITY_NONE) { // not the default
            checkRestrictTo(context, node, method, annotation, usageInfo, RESTRICT_TO_TESTS)
        } else {
            // Check that the target method is available
            // (1) private is available in the same compilation unit
            // (2) package private is available in the same package
            // (3) protected is available either from subclasses or in same package

            val uFile = node.getContainingUFile()
            val containingFile1 = UastLintUtils.getPsiFile(uFile)
            val containingFile2 = UastLintUtils.getContainingFile(method)
            if (containingFile1 == containingFile2 || containingFile2 == null) {
                // Same compilation unit
                return
            }

            // Validity check (since Kotlin UAST creates several light classes around
            // PSI files that sometimes fail equality tests)
            if (containingFile1?.virtualFile == containingFile2.virtualFile) {
                return
            }

            if (visibility == VISIBILITY_PRIVATE) {
                if (!isTestContext(context, node)) {
                    reportVisibilityError(context, node, "private")
                }
                return
            }

            val evaluator = context.evaluator
            val pkg = evaluator.getPackage(node)
            val methodPackage = evaluator.getPackage(method)
            // can't compare pkg == methodPackage because PsiPackageImpl#equals only returns
            // true for other instances of the exact same class
            if (pkg?.qualifiedName == methodPackage?.qualifiedName) {
                // Same package
                return
            }
            if (visibility == VISIBILITY_PACKAGE_PRIVATE) {
                if (!isTestContext(context, node)) {
                    reportVisibilityError(context, node, "package private")
                }
                return
            }

            assert(visibility == VISIBILITY_PROTECTED)

            val methodClass = method.containingClass
            val thisClass = node.getParentOfType<UClass>(UClass::class.java, true)
            if (thisClass == null || methodClass == null) {
                return
            }
            val qualifiedName = methodClass.qualifiedName
            if (qualifiedName == null || evaluator.inheritsFrom(
                    thisClass,
                    qualifiedName,
                    false
                )
            ) {
                return
            }

            if (!isTestContext(context, node)) {
                reportVisibilityError(context, node, "protected")
            }
        }
    }

    private fun reportVisibilityError(
        context: JavaContext,
        node: UElement,
        desc: String
    ) {
        val message = "This method should only be accessed from tests or within $desc scope"
        val location: Location = if (node is UCallExpression) {
            context.getCallLocation(node, false, false)
        } else {
            context.getLocation(node)
        }

        report(context, TEST_VISIBILITY, node, location, message)
    }

    // TODO: Test XML access of restricted classes
    private fun checkRestrictTo(
        context: JavaContext,
        node: UElement,
        method: PsiMember?,
        annotation: UAnnotation,
        usageInfo: AnnotationUsageInfo,
        applyClassAnnotationsToMembers: Boolean = true
    ) {
        val scope = getRestrictionScope(annotation)
        if (scope != 0) {
            checkRestrictTo(context, node, method, annotation, usageInfo, scope, applyClassAnnotationsToMembers)
        }
    }

    private fun checkRestrictTo(
        context: JavaContext,
        node: UElement,
        member: PsiMember?,
        annotation: UAnnotation,
        usageInfo: AnnotationUsageInfo,
        scope: Int,
        applyClassAnnotationsToMembers: Boolean = true
    ) {

        val containingClass = when {
            node is UTypeReferenceExpression -> PsiTypesUtil.getPsiClass(node.type)
            member != null -> member.containingClass
            node is UCallExpression -> node.classReference?.resolve() as? PsiClass?
            node is PsiClass -> node
            else -> null
        }

        containingClass ?: return

        val annotations = usageInfo.annotations
        if (!applyClassAnnotationsToMembers) {
            val origin = annotations[usageInfo.index].origin
            if (member is PsiClass) {
                if (origin != AnnotationOrigin.CLASS) {
                    return
                }
            } else if (origin != AnnotationOrigin.METHOD && origin != AnnotationOrigin.FIELD) {
                return
            }
        }

        if (usageInfo.anyCloser {
            it.qualifiedName == RESTRICT_TO_ANNOTATION.oldName() ||
                it.qualifiedName == RESTRICT_TO_ANNOTATION.newName()
        }
        ) {
            return
        }

        if (scope and RESTRICT_TO_LIBRARY_GROUP != 0 && member != null) {
            val evaluator = context.evaluator
            val thisCoordinates = evaluator.getLibrary(node) ?: context.project.mavenCoordinate
            val methodCoordinates = evaluator.getLibrary(member) ?: run {
                if (thisCoordinates != null && member !is PsiCompiledElement) {
                    // Local source?
                    context.evaluator.getProject(member)?.mavenCoordinate
                } else {
                    null
                }
            }
            val thisGroup = thisCoordinates?.groupId
            val methodGroup = methodCoordinates?.groupId
            if (thisGroup != methodGroup && methodGroup != null) {
                val thisGroupDisplayText = thisGroup ?: "<unknown>"
                val where = "from within the same library group (referenced groupId=`$methodGroup` from groupId=`$thisGroupDisplayText`)"
                reportRestriction(where, containingClass, member, context, node, usageInfo)
            }
        } else if (scope and RESTRICT_TO_LIBRARY_GROUP_PREFIX != 0 && member != null) {
            val evaluator = context.evaluator
            val thisCoordinates = evaluator.getLibrary(node) ?: context.project.mavenCoordinate
            val methodCoordinates = evaluator.getLibrary(member) ?: run {
                if (thisCoordinates != null && member !is PsiCompiledElement) {
                    // Local source?
                    context.evaluator.getProject(member)?.mavenCoordinate
                } else {
                    null
                }
            }
            val thisGroup = thisCoordinates?.groupId
            val methodGroup = methodCoordinates?.groupId
            if (methodGroup != null &&
                (thisGroup == null || !sameLibraryGroupPrefix(thisGroup, methodGroup))
            ) {
                val expectedPrefix = methodGroup.lastIndexOf('.').let {
                    if (it < 0) { "\"\"" } else { methodGroup.substring(0, it) }
                }
                val where =
                    "from within the same library group prefix (referenced groupId=`$methodGroup` with prefix $expectedPrefix${if (thisGroup != null) " from groupId=`$thisGroup`" else ""})"
                reportRestriction(where, containingClass, member, context, node, usageInfo)
            }
        } else if (scope and RESTRICT_TO_LIBRARY != 0 && member != null) {
            val evaluator = context.evaluator
            val thisCoordinates = evaluator.getLibrary(node) ?: context.project.mavenCoordinate
            val methodCoordinates = evaluator.getLibrary(member)
            val thisGroup = thisCoordinates?.groupId
            val methodGroup = methodCoordinates?.groupId
            if (thisGroup != methodGroup && methodGroup != null) {
                val thisArtifact = thisCoordinates?.artifactId
                val methodArtifact = methodCoordinates.artifactId
                if (thisArtifact != methodArtifact) {
                    val where = "from within the same library ($methodGroup:$methodArtifact)"
                    reportRestriction(where, containingClass, member, context, node, usageInfo)
                }
            } else if (member !is PsiCompiledElement) {
                // If the resolved method is source, make sure they're part
                // of the same Gradle project
                val project = context.evaluator.getProject(member)
                if (project != null && project != context.project) {
                    val coordinates = project.mavenCoordinate
                    val name = if (coordinates != null) {
                        "${coordinates.groupId}:${coordinates.artifactId}"
                    } else {
                        project.name
                    }
                    val where = "from within the same library ($name)"
                    reportRestriction(where, containingClass, member, context, node, usageInfo)
                }
            }
        }

        if (scope and RESTRICT_TO_TESTS != 0) {
            if (!isTestContext(context, node)) {
                reportRestriction("from tests", containingClass, member, context, node, usageInfo)
            }
        }

        if (scope and RESTRICT_TO_ALL != 0) {
            val fqn = annotation.qualifiedName
            if (fqn != GMS_HIDE_ANNOTATION || flagHide(context, containingClass, member, node)) {
                reportRestriction(null, containingClass, member, context, node, usageInfo)
            }
        }

        if (scope and RESTRICT_TO_SUBCLASSES != 0) {
            val qualifiedName = containingClass.qualifiedName
            if (qualifiedName != null) {
                val evaluator = context.evaluator

                var outer: UClass?
                var isSubClass = false
                var prev = node

                while (true) {
                    outer = prev.getParentOfType(UClass::class.java, true)
                    if (outer == null) {
                        break
                    }
                    if (evaluator.inheritsFrom(outer, qualifiedName, false)) {
                        isSubClass = true
                        break
                    }

                    if (evaluator.isStatic(outer)) {
                        break
                    }
                    prev = outer
                }

                if (!isSubClass) {
                    reportRestriction("from subclasses", containingClass, member, context, node, usageInfo)
                }
            }
        }
    }

    private fun reportRestriction(
        where: String?,
        containingClass: PsiClass,
        member: PsiMember?,
        context: JavaContext,
        node: UElement,
        usageInfo: AnnotationUsageInfo
    ) {
        var api: String
        api = if (member == null || member is PsiMethod && member.isConstructor) {
            member?.name ?: (containingClass.name + " constructor")
        } else
        //noinspection LintImplPsiEquals
            if (containingClass == member) {
                member.name ?: "class"
            } else {
                containingClass.name + "." + member.name
            }

        var locationNode = node
        if (node is UCallExpression) {
            val nameElement = node.methodIdentifier
            if (nameElement != null) {
                locationNode = nameElement
            }

            // If the annotation was reported on the class, and the left hand side
            // expression is that class, use it as the name node?
            val annotation = usageInfo.annotations[usageInfo.index]
            val annotated = annotation.annotated
            //noinspection LintImplPsiEquals
            if (where == null && annotated is PsiClass && annotated != usageInfo.referenced) {
                val qualifier = node.receiver
                val className = annotated.name
                if (qualifier != null && className != null && qualifier.asSourceString() == className) {
                    locationNode = qualifier
                    api = className
                }
            }
        }

        // If this error message changes, you need to also update ResourceTypeInspection#guessLintIssue
        var message: String
        if (where == null) {
            message = "$api is marked as internal and should not be accessed from apps"
        } else {
            val refType = if (member is PsiMethod) "called" else "accessed"
            message = "$api can only be $refType $where"

            // Most users will encounter this for the support library; let's have a clearer error message
            // for that specific scenario
            if (where == "from within the same library (groupId=com.android.support)") {
                // If this error message changes, you need to also update ResourceTypeInspection#guessLintIssue
                message =
                    "This API is marked as internal to the support library and should not be accessed from apps"
            }
        }

        val location = if (locationNode is UCallExpression) {
            context.getCallLocation(locationNode, false, false)
        } else {
            context.getLocation(locationNode)
        }
        report(context, RESTRICTED, node, location, message, null)
    }

    companion object {
        private val IMPLEMENTATION = Implementation(
            RestrictToDetector::class.java,
            Scope.JAVA_FILE_SCOPE
        )

        private const val ATTR_OTHERWISE = "otherwise"
        private const val ATTR_PRODUCTION_VISIBILITY = "productionVisibility"

        // Must match constants in @VisibleForTesting:
        private const val VISIBILITY_PRIVATE = 2
        private const val VISIBILITY_PACKAGE_PRIVATE = 3
        private const val VISIBILITY_PROTECTED = 4
        private const val VISIBILITY_NONE = 5
        // TODO: Kotlin "module" visibility

        private fun getVisibilityForTesting(annotation: UAnnotation): Int {
            val value = annotation.findDeclaredAttributeValue(ATTR_OTHERWISE)
                // Guava within Google3:
                ?: annotation.findDeclaredAttributeValue(ATTR_PRODUCTION_VISIBILITY)
            if (value is ULiteralExpression) {
                val v = value.value
                if (v is Int) {
                    return v
                }
            } else if (value is UReferenceExpression) {
                // Not compiled; this is unlikely (but can happen when editing the support
                // library project itself)
                when (value.resolvedName) {
                    "NONE" -> return VISIBILITY_NONE
                    "PRIVATE" -> return VISIBILITY_PRIVATE
                    "PROTECTED" -> return VISIBILITY_PROTECTED
                    "PACKAGE_PRIVATE" -> return VISIBILITY_PACKAGE_PRIVATE
                }
            } else if (value is UnknownJavaExpression) {
                // Workaround for https://youtrack.jetbrains.com/issue/KT-47290 -- see
                // https://issuetracker.google.com/190113936 for an applicable scenario
                val sourcePsi = value.sourcePsi
                if (sourcePsi is PsiLiteralExpression) {
                    val v = sourcePsi.value
                    if (v is Int) {
                        return v
                    }
                }
            } else if (value is UastEmptyExpression) {
                // Some kind of error in UAST; try harder. JavaUAnnotation is used to wrap
                // class file annotations and in findDeclaredAttributeValue it returns
                // UastEmptyExpression if it cannot convert it to UAST.
                // (This may be an older version of KT-47290; it doesn't seem to trigger
                // from the tests anymore)
                val psi = annotation.sourcePsi
                if (psi is ClsAnnotationImpl) {
                    val otherwise = psi.findAttribute(ATTR_OTHERWISE)
                        ?: psi.findAttribute(ATTR_PRODUCTION_VISIBILITY)
                    val v = otherwise?.attributeValue
                    if (v is JvmAnnotationConstantValue) {
                        val constant = v.constantValue
                        if (constant is Number) {
                            return constant.toInt()
                        }
                    }
                }
            }

            return VISIBILITY_PRIVATE // the default
        }

        /** `RestrictTo(RestrictTo.Scope.GROUP_ID` */
        private const val RESTRICT_TO_LIBRARY_GROUP = 1 shl 0

        /** `RestrictTo(RestrictTo.Scope.GROUP_ID` */
        private const val RESTRICT_TO_LIBRARY = 1 shl 1

        /** `RestrictTo(RestrictTo.Scope.GROUP_ID` */
        private const val RESTRICT_TO_LIBRARY_GROUP_PREFIX = 1 shl 2

        /** `RestrictTo(RestrictTo.Scope.TESTS` */
        private const val RESTRICT_TO_TESTS = 1 shl 3

        /** `RestrictTo(RestrictTo.Scope.SUBCLASSES` */
        private const val RESTRICT_TO_SUBCLASSES = 1 shl 4
        private const val RESTRICT_TO_ALL = 1 shl 5

        private fun getRestrictionScope(annotation: UAnnotation): Int {
            val value = annotation.findDeclaredAttributeValue(ATTR_VALUE)
            if (value != null) {
                return getRestrictionScope(value, annotation)
            } else if (GMS_HIDE_ANNOTATION == annotation.qualifiedName) {
                return RESTRICT_TO_ALL
            }
            return 0
        }

        private fun getRestrictionScope(expression: UExpression?, annotation: UAnnotation): Int {
            var scope = 0
            if (expression != null) {
                if (expression.isArrayInitializer()) {
                    val initializerExpression = expression as UCallExpression?
                    val initializers = initializerExpression!!.valueArguments
                    for (initializer in initializers) {
                        scope = scope or getRestrictionScope(initializer, annotation)
                    }
                } else if (expression is UReferenceExpression) {
                    val resolved = expression.resolve()
                    if (resolved is PsiField) {
                        val name = resolved.name
                        if ("GROUP_ID" == name || "LIBRARY_GROUP" == name) {
                            scope = scope or RESTRICT_TO_LIBRARY_GROUP
                        } else if ("SUBCLASSES" == name) {
                            scope = scope or RESTRICT_TO_SUBCLASSES
                        } else if ("TESTS" == name) {
                            scope = scope or RESTRICT_TO_TESTS
                        } else if ("LIBRARY" == name) {
                            scope = scope or RESTRICT_TO_LIBRARY
                        } else if ("LIBRARY_GROUP_PREFIX" == name) {
                            scope = scope or RESTRICT_TO_LIBRARY_GROUP_PREFIX
                        }
                    }
                } else if (expression is UastEmptyExpression) {
                    // See JavaUAnnotation.findDeclaredAttributeValue
                    val psi = annotation.sourcePsi
                    if (psi is ClsAnnotationImpl) {
                        val otherwise = psi.findAttribute(ATTR_VALUE)
                        val v = otherwise?.attributeValue
                        if (v is JvmAnnotationConstantValue) {
                            val constant = v.constantValue
                            if (constant is Number) {
                                scope = scope or constant.toInt()
                            }
                        }
                    }
                }
            }

            return scope
        }

        /**
         * Implements the group prefix equality that is described in the
         * documentation for the RestrictTo.Scope.LIBRARY_GROUP_PREFIX
         * enum constant.
         */
        fun sameLibraryGroupPrefix(group1: String, group2: String): Boolean {
            // TODO: Allow group1.startsWith(group2) || group2.startsWith(group1) ?

            if (group1 == group2) {
                return true
            }

            val i1 = group1.lastIndexOf('.')
            val i2 = group2.lastIndexOf('.')
            if (i2 != i1 || i1 == -1) {
                return false
            }

            return group1.regionMatches(0, group2, 0, i1)
        }

        /** Using a restricted API. */
        @JvmField
        val RESTRICTED = Issue.create(
            id = "RestrictedApi",
            briefDescription = "Restricted API",
            explanation = """
                This API has been flagged with a restriction that has not been met.

                Examples of API restrictions:
                * Method can only be invoked by a subclass
                * Method can only be accessed from within the same library (defined by the Gradle library group id)
                * Method can only be accessed from tests.

                You can add your own API restrictions with the `@RestrictTo` annotation.""",
            category = Category.CORRECTNESS,
            priority = 4,
            severity = Severity.ERROR,
            implementation = IMPLEMENTATION
        )

        /** Using an intended-for-tests API. */
        @JvmField
        val TEST_VISIBILITY = Issue.create(
            id = "VisibleForTests",
            briefDescription = "Visible Only For Tests",
            explanation = """
                With the `@VisibleForTesting` annotation you can specify an `otherwise=` \
                attribute which specifies the intended visibility if the method had not \
                been made more widely visible for the tests.

                This check looks for accesses from production code (e.g. not tests) where \
                the access would not have been allowed with the intended production \
                visibility.""",
            category = Category.CORRECTNESS,
            priority = 4,
            severity = Severity.WARNING,
            implementation = IMPLEMENTATION
        )
    }
}
