package com.applovin.mediation;

/**
 * This interface defines a listener intended to be notified about ad events.
 */
public interface MaxAdListener
{
    /**
     * This method is called when a new ad has been loaded.
     *
     * @param ad Newly received ad. Guaranteed not to be null.
     */
    void onAdLoaded(final MaxAd ad);

    /**
     * This method is called when an ad could not be retrieved.
     * <p>
     * Common error codes are: </br> <code>204</code> -- no ad is available</br> <code>5xx</code> -- internal server error</br> <code>negative number</code> -- internal errors </br>
     * </p>
     *
     * @param adUnitId  Ad unit identifier for which the ad was requested. Guaranteed not to be null.
     * @param errorCode An error code representing the failure reason. Common error codes are defined in {@link MaxErrorCodes}.
     */
    void onAdLoadFailed(final String adUnitId, final int errorCode);

    /**
     * This method is invoked when an ad is displayed.
     * <p>
     * This method is invoked on the main UI thread.
     *
     * @param ad Ad that was just displayed. Guaranteed not to be null.
     */
    void onAdDisplayed(final MaxAd ad);

    /**
     * This method is invoked when an ad is hidden.
     * <p>
     * This method is invoked on the main UI thread.
     *
     * @param ad Ad that was just hidden. Guaranteed not to be null.
     */
    void onAdHidden(final MaxAd ad);

    /**
     * This method is invoked when the ad is clicked.
     * <p>
     * This method is invoked on the main UI thread.
     *
     * @param ad Ad that was just clicked. Guaranteed not to be null.
     */
    void onAdClicked(final MaxAd ad);

    /**
     * This method is invoked when the ad failed to displayed.
     * <p>
     * This method is invoked on the main UI thread.
     *
     * @param ad        Ad that was just failed to display. Guaranteed not to be null.
     * @param errorCode Error that indicates display failure. Common error codes are defined in {@link MaxErrorCodes}.
     */
    void onAdDisplayFailed(final MaxAd ad, final int errorCode);
}
