package com.applovin.nativeAds;

import android.content.Context;

import com.applovin.sdk.AppLovinPostbackListener;

/**
 * Defines an in-content ad which should be rendered by you into a custom view that matches the look and feel of your application.
 */
public interface AppLovinNativeAd
{
    /**
     * Retrieve a unique identifier for this ad, which can be provided to AppLovin should you see issues with this particular ad.
     */
    long getAdId();

    /**
     * The zone identifier for the ad, if any.
     */
    String getZoneId();

    /**
     * Retrieve the title of the application being advertised.
     */
    String getTitle();

    /**
     * Retrieve the description of the application being advertised.
     */
    String getDescriptionText();

    /**
     * Retrieve the caption of the application being advertised. May be null.
     */
    String getCaptionText();

    /**
     * Retrieve the call-to-action / button text of the application being advertised. May be null.
     */
    String getCtaText();

    /**
     * Retrieve the URL of an image containing the launcher icon of the application being advertised
     */
    String getIconUrl();

    /**
     * Retrieve the URL of an image containing a promotional image or screenshot of the application being advertised.
     */
    String getImageUrl();

    /**
     * Retrieve the URL of an image containing the current Google Play star rating of the application being advertised.
     */
    float getStarRating();

    /**
     * Retrieve the URL of a promotional video, if this ad contains a video.
     * <p>
     * Not all native advertisements will contain a video. Therefore, may be null.
     */
    String getVideoUrl();

    /**
     * Retrieve the URL which should be fired upon ad impression.
     *
     * @deprecated Use <code>trackImpression() or trackImpression(AppLovinPostbackListener postbackListener)</code> rather than firing this Url yourself.
     */
    String getImpressionTrackingUrl();

    /**
     * Fires the impression asynchronously.
     */
    void trackImpression();

    /**
     * Fires the impression asynchronously and notifies the provided delegate.
     */
    void trackImpression(AppLovinPostbackListener postbackListener);

    /**
     * Retrieve the URL which should be fired upon ad click.
     *
     * @deprecated Use <code>launchClickTarget()</code> rather than opening this URL yourself.
     */
    @Deprecated
    String getClickUrl();

    /**
     * Retrieve the URL which should be fired upon video start.
     */
    String getVideoStartTrackingUrl();

    /**
     * Retrieve the URL which should be fired upon video completion.
     *
     * @param percentViewed The percentage of the video (0 - 100) that was viewed by the user.
     * @param firstPlay     Whether or not this postback represents initial playback of the video. The first time you begin playback, you should pass true. If the video is paused for any reason and then later resumed mid-playback, you should fire this postback a second time, passing false to firstPlay.
     */
    String getVideoEndTrackingUrl(final int percentViewed, boolean firstPlay);

    /**
     * Check whether or not all image assets for this ad have been precached.
     *
     * @return True if image resources have been pre-cached. In this case, resource URLs retrieved from this instance will point to the local filesystem. False if image resources have not been pre-cached. In this case, resource URLs will point to HTTP or HTTPS resources on the web.
     */
    boolean isImagePrecached();

    /**
     * Check whether or not this ad's video have been precached.
     *
     * @return True if this ad has a video resource, and it has been pre-cached. In this case, getVideoUrl will point to the local filesystem for this instance. False if video resources have not been pre-cached. In this case, getVideoUrl will point to an HTTP or HTTPS resource on the web.
     */
    boolean isVideoPrecached();

    /**
     * Handle a click on this native ad by launching the ad's destination.
     * <p>
     * You should call this method anytime the user taps anywhere on your native ad.
     * Calling this method launches Chrome or the Google Play Store and will result in your app being paused.
     */
    void launchClickTarget(Context context);
}