package com.applovin.sdk;

import com.applovin.impl.sdk.utils.CollectionUtils;

import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

/**
 * This class defines a size of an ad to be displayed. It is recommended to use default sizes that are declared in this class (<code>BANNER</code>, <code>MREC</code>, <code>INTERSTITIAL</code>)
 *
 * @author Basil Shikin
 */
public class AppLovinAdSize
{
    public static final int SPAN = -1;

    /**
     * Represents a 350x50 banner advertisement.
     */
    public static final AppLovinAdSize BANNER = new AppLovinAdSize( SPAN, 50, "BANNER" );

    /**
     * Represents a 300x250 rectangular advertisement.
     */
    public static final AppLovinAdSize MREC = new AppLovinAdSize( 300, 250, "MREC" );

    /**
     * Represents a 728x90 leaderboard advertisement indented for tablets.
     */
    public static final AppLovinAdSize LEADER = new AppLovinAdSize( SPAN, 90, "LEADER" );

    /**
     * Represents a full-screen advertisement.
     */
    public static final AppLovinAdSize INTERSTITIAL = new AppLovinAdSize( SPAN, SPAN, "INTER" );

    /**
     * Represents a native advertisement
     */
    public static final AppLovinAdSize NATIVE = new AppLovinAdSize( "NATIVE" );

    private final int    width;
    private final int    height;
    private final String label;

    /**
     * Create an ad size.
     * <p>
     * <b>Please note:</b> This constructor is for internal use only.
     * </p>
     *
     * @param label Size name. Must not be null. Should be shorter then 9 characters.
     *
     * @hide
     */
    public AppLovinAdSize(String label)
    {
        this( 0, 0, label );
    }

    /**
     * Create an ad size.
     * <p>
     * <b>Please note:</b> This constructor is for internal use only.
     * </p>
     *
     * @param width  Ad width. Must be a positive number or 0 if label is specified.
     * @param height Ad height. Must be a positive number or 0 if label is specified.
     * @param label  Size name. Must not be null. Should be shorter then 9 characters.
     *
     * @hide
     */
    protected AppLovinAdSize(int width, int height, String label)
    {
        // Check input
        if ( width < 0 && width != SPAN ) throw new IllegalArgumentException( "Ad width must be a positive number. Number provided: " + width );
        if ( width > 9999 ) throw new IllegalArgumentException( "Ad width must be less then 9999. Number provided: " + width );
        if ( height < 0 && height != SPAN ) throw new IllegalArgumentException( "Ad height must be a positive number. Number provided: " + height );
        if ( height > 9999 ) throw new IllegalArgumentException( "Ad height must be less then 9999. Number provided: " + height );
        if ( label == null ) throw new IllegalArgumentException( "No label specified" );
        if ( label.length() > 9 ) throw new IllegalArgumentException( "Provided label is too long. Label provided: " + label );

        this.width = width;
        this.height = height;
        this.label = label;
    }

    /**
     * Get requested ad width
     *
     * @return Ad width
     */
    public int getWidth()
    {
        return width;
    }

    /**
     * Get requested ad height
     *
     * @return Ad height
     */
    public int getHeight()
    {
        return height;
    }

    /**
     * Get a name of the ad size
     *
     * @return Name of the ad size.
     */
    public String getLabel()
    {
        return label.toUpperCase( Locale.ENGLISH );
    }

    /**
     * Simply calls <code>getLabel()</code>.
     *
     * @return Name of the ad size.
     */
    @Override
    public String toString()
    {
        return getLabel();
    }

    /**
     * Create a size object from a string representation
     *
     * @param sizeName Name of the size.
     *
     * @return Ad size or null in case input string is null or unknown.
     */
    public static AppLovinAdSize fromString(String sizeName)
    {
        if ( sizeName == null || sizeName.length() < 1 ) return null;

        final String sizeAttributeLower = sizeName.toLowerCase( Locale.ENGLISH );

        if ( sizeAttributeLower.equals( "banner" ) )
        {
            return AppLovinAdSize.BANNER;
        }
        else if ( sizeAttributeLower.equals( "interstitial" ) || sizeAttributeLower.equals( "inter" ) )
        {
            return AppLovinAdSize.INTERSTITIAL;
        }
        else if ( sizeAttributeLower.equals( "mrec" ) )
        {
            return AppLovinAdSize.MREC;
        }
        else if ( sizeAttributeLower.equals( "leader" ) )
        {
            return AppLovinAdSize.LEADER;
        }
        else
        {
            final List<String> components = CollectionUtils.explode( sizeName, "x" );
            if ( components.size() == 2 )
            {
                final int width = parseSize( components.get( 0 ) );
                final int height = parseSize( components.get( 1 ) );

                return new AppLovinAdSize( width, height, sizeName );
            }
            else
            {
                return new AppLovinAdSize( 0, 0, sizeName );
            }
        }
    }

    private static int parseSize(final String sizeStr)
    {
        if ( "span".equalsIgnoreCase( sizeStr ) )
        {
            return SPAN;
        }
        else
        {
            try
            {
                return Integer.parseInt( sizeStr );
            }
            catch ( NumberFormatException ignore )
            {
                return 0;
            }
        }
    }

    /**
     * Obtain a set of all available Ad Sizes.
     *
     * @return A set containing all available Ad Size objects.
     */
    public static Set<AppLovinAdSize> allSizes()
    {
        final Set<AppLovinAdSize> allSizes = new HashSet<>( 4 );

        allSizes.add( AppLovinAdSize.BANNER );
        allSizes.add( AppLovinAdSize.MREC );
        allSizes.add( AppLovinAdSize.INTERSTITIAL );
        allSizes.add( AppLovinAdSize.LEADER );
        allSizes.add( AppLovinAdSize.NATIVE );

        return allSizes;
    }
}
