package com.atlassian.logging.log4j;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.helpers.FormattingInfo;
import org.apache.log4j.helpers.PatternConverter;
import org.apache.log4j.helpers.PatternParser;
import org.apache.log4j.spi.LoggingEvent;

/**
 * This pattern parser can interpret %q{n} and shorten the category more smartly than log4j does
 *
 */
public class CategoryCollapsingPatternParser extends PatternParser
{
    private final int fqNameCollapsePrecision;
    private FqNameCollapser fqNameCollapser;

    public CategoryCollapsingPatternParser(final String pattern, final int fqNameCollapsePrecision)
    {
        super(pattern);
        this.fqNameCollapsePrecision = fqNameCollapsePrecision;
        fqNameCollapser = new FqNameCollapser(fqNameCollapsePrecision, FqNameCollapser.Strategy.PACKAGE_SEGMENTS);
    }

    @Override
    protected void finalizeConverter(final char character)
    {
        switch (character)
        {
            //
            // Q is the character we chose mostly because its not used in log4j in upper or lower case and t
            // its somewehere in fully Qualified name
            case 'Q':
                fqNameCollapser = new FqNameCollapser(extractPrecisionOption(), FqNameCollapser.Strategy.PACKAGE_LENGTH);
                addConverter(new CategoryCollapsingPatternConverter(formattingInfo, fqNameCollapser));
                break;
            case 'q':
                fqNameCollapser = new FqNameCollapser(extractPrecisionOption(), FqNameCollapser.Strategy.PACKAGE_SEGMENTS);
                addConverter(new CategoryCollapsingPatternConverter(formattingInfo, fqNameCollapser));
                break;
            default:
                super.finalizeConverter(character);
                break;
        }
    }

    @Override
    protected int extractPrecisionOption()
    {
        String opt = extractOption();
        if (StringUtils.isBlank(opt))
        {
            return fqNameCollapsePrecision;
        }
        try
        {
            return Integer.parseInt(opt);
        }
        catch (NumberFormatException e)
        {
            return fqNameCollapsePrecision;
        }
    }

    /**
     * A converter that collapses it input
     */
    protected static class CategoryCollapsingPatternConverter extends PatternConverter
    {
        private final FqNameCollapser collapser;

        CategoryCollapsingPatternConverter(final FormattingInfo info, FqNameCollapser fqNameCollapser)
        {
            super(info);
            this.collapser = fqNameCollapser;
        }

        @Override
        public String convert(LoggingEvent event)
        {
            return collapser.collapse(event.getLoggerName());
        }
    }

}
