/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bamboo.plugins.ssh;

import com.atlassian.bamboo.executor.RetryingTaskExecutor;
import com.atlassian.bamboo.plugins.ssh.SshConfig;
import com.atlassian.bamboo.plugins.ssh.util.NoCloseInputStream;
import com.atlassian.bamboo.plugins.ssh.util.NoCloseOutputStream;
import com.atlassian.bamboo.security.TrustedKey;
import com.atlassian.bamboo.ssh.ProxyConnectionData;
import com.atlassian.bamboo.ssh.ProxyConnectionDataProvider;
import com.atlassian.bamboo.ssh.UntrustedKeyException;
import com.atlassian.bamboo.utils.RecentLazyReference;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.atlassian.util.concurrent.ThreadFactories;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.time.Duration;
import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.apache.commons.lang3.SystemUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.sshd.client.SshClient;
import org.apache.sshd.client.channel.ChannelExec;
import org.apache.sshd.client.channel.ClientChannelEvent;
import org.apache.sshd.client.future.AuthFuture;
import org.apache.sshd.client.future.ConnectFuture;
import org.apache.sshd.client.future.OpenFuture;
import org.apache.sshd.client.session.ClientSession;
import org.apache.sshd.common.channel.Channel;
import org.apache.sshd.common.channel.PtyChannelConfiguration;
import org.apache.sshd.common.channel.PtyChannelConfigurationHolder;
import org.apache.sshd.common.session.ConnectionService;
import org.apache.sshd.server.Environment;
import org.apache.sshd.server.ExitCallback;
import org.apache.sshd.server.channel.ChannelSession;
import org.apache.sshd.server.command.Command;

public class SshProxyCommand
implements Command {
    private static final Logger log = LogManager.getLogger(SshProxyCommand.class);
    private static final int SSH_CONNECTION_RETRIES = 4;
    private static final Duration INITIAL_SSH_RETRY_DELAY = Duration.ofSeconds(1L);
    private static final ExecutorService threadPool = Executors.newCachedThreadPool(ThreadFactories.namedThreadFactory((String)"SSH connection monitoring thread"));
    public static final String BAMBOO_UNTRUSTED_KEY = "bamboo_untrusted_key";
    static final String AUTHENTICATING_REMOTE_SESSION_FAILED = "Authenticating remote session failed";
    private final String command;
    private final ProxyConnectionDataProvider proxyConnectionDataProvider;
    private static final Supplier<SshConfig> recentSshConfig = new RecentLazyReference<SshConfig>(60L, TimeUnit.SECONDS){

        protected SshConfig createInstance() throws Exception {
            return SshConfig.getDefaultSshConfig();
        }
    };
    private OutputStream outputStream;
    private InputStream inputStream;
    private OutputStream errorStream;
    private ExitCallback exitCallback;
    private final SshClient client;
    private ClientSession session;
    private ProxyConnectionData connectionData;
    private ChannelExec channel;

    public SshProxyCommand(SshClient client, ProxyConnectionDataProvider proxyConnectionDataProvider, String command) {
        this.client = client;
        this.command = command;
        this.proxyConnectionDataProvider = proxyConnectionDataProvider;
    }

    public void setInputStream(InputStream in) {
        this.inputStream = in;
    }

    public void setOutputStream(OutputStream out) {
        this.outputStream = out;
    }

    public void setErrorStream(OutputStream err) {
        this.errorStream = err;
    }

    public void setExitCallback(ExitCallback callback) {
        this.exitCallback = callback;
    }

    @VisibleForTesting
    public boolean isChannelClosed() {
        return this.channel.isClosed();
    }

    public void start(ChannelSession serverSession, Environment env) throws IOException {
        log.debug("Start called: {}", (Object)env.getEnv().toString());
        String proxyUserName = (String)env.getEnv().get("USER");
        if (proxyUserName == null) {
            String msg = "Expected USER to be set: " + env.getEnv().toString();
            log.error(msg);
            this.writeToErrorStream(msg);
            this.exitCallback.onExit(255);
            return;
        }
        ProxyConnectionData suppliedConnectionData = this.proxyConnectionDataProvider.getConnectionData(proxyUserName);
        if (suppliedConnectionData == null) {
            String msg = "Cannot map user to connection data: " + proxyUserName;
            log.error(msg);
            this.writeToErrorStream(msg);
            this.exitCallback.onExit(255);
            return;
        }
        SshConfig sshConfig = recentSshConfig.get();
        try {
            this.connectionData = sshConfig.apply(suppliedConnectionData, SystemUtils.USER_NAME);
        }
        catch (GeneralSecurityException e) {
            this.finishWithError("Error while opening SSH session", e);
        }
        threadPool.submit(() -> {
            log.debug("Monitoring " + proxyUserName);
            Callable<Object> startConnectionTask = () -> {
                this.startRemoteConnection();
                this.connectSession();
                return null;
            };
            try {
                String taskDescription = "Connecting to " + String.valueOf(this.connectionData);
                RetryingTaskExecutor.retry((String)taskDescription, (int)4, (Duration)INITIAL_SSH_RETRY_DELAY, startConnectionTask, IllegalStateException.class::isInstance);
            }
            catch (Exception e) {
                this.finishWithError("Error while opening SSH session", e);
            }
        });
    }

    public void destroy(ChannelSession serverSession) {
        log.debug("Destroy called");
        threadPool.submit(() -> {
            Thread.currentThread().setName("Connection killer");
            this.closeSession();
        });
    }

    private synchronized void startRemoteConnection() {
        log.debug("Connecting client");
        try {
            ConnectFuture connectFuture = this.client.connect(this.connectionData.getRemoteUserName(), this.connectionData.getRemoteAddress());
            connectFuture.awaitUninterruptibly();
            this.session = (ClientSession)connectFuture.getSession();
            if (!connectFuture.isConnected()) {
                this.finishWithError("Can't connect session, connectFuture is not connected...", connectFuture.getException());
                return;
            }
        }
        catch (Exception e) {
            this.finishWithError("Remote connection failed", e);
            return;
        }
    }

    private void connectSession() {
        AuthFuture authFuture;
        if (this.session == null) {
            return;
        }
        try {
            KeyPair keyPair = this.connectionData.getKeyPair();
            if (keyPair == null) {
                String remotePassword = this.connectionData.getRemotePassword();
                Preconditions.checkNotNull((Object)remotePassword, (Object)"neither the key pair nor the password is set");
                log.debug("Authenticating user [{}] using a password", (Object)this.connectionData.getRemoteUserName());
                this.session.addPasswordIdentity(remotePassword);
                authFuture = this.session.auth();
            } else {
                log.debug("Authenticating user [{}] using a key pair", (Object)this.connectionData.getRemoteUserName());
                this.session.addPublicKeyIdentity(keyPair);
                authFuture = this.session.auth();
            }
        }
        catch (IOException e) {
            this.finishWithError("I/O exception during authentication of the remote session", e);
            return;
        }
        authFuture.awaitUninterruptibly();
        if (authFuture.isSuccess()) {
            this.createExecChannel();
        } else if (this.session.getMetadataMap().containsKey(BAMBOO_UNTRUSTED_KEY)) {
            this.finishWithError("Host key not trusted by Bamboo", (Throwable)new UntrustedKeyException("Host key verification failed", (TrustedKey)this.session.getMetadataMap().get(BAMBOO_UNTRUSTED_KEY)));
        } else {
            this.finishWithError(AUTHENTICATING_REMOTE_SESSION_FAILED, authFuture.getException());
        }
    }

    private void createExecChannel() {
        OpenFuture openFuture;
        log.debug("Creating exec channel");
        try {
            PtyChannelConfiguration configurationHolder = new PtyChannelConfiguration();
            configurationHolder.setPtyType("dummy");
            this.channel = new ChannelExec(this.mapPaths(this.command), (PtyChannelConfigurationHolder)configurationHolder, null){

                protected void preClose() {
                    switch (SshProxyCommand.this.session.getTimeoutStatus().getStatus()) {
                        case AuthTimeout: {
                            SshProxyCommand.this.finishWithError(String.format("Session has timed out waiting for authentication after %d ms.", SshProxyCommand.this.session.getAuthTimeout().toMillis()));
                            break;
                        }
                        case IdleTimeout: {
                            SshProxyCommand.this.finishWithError(String.format("User session has timed out idling after %d ms.", SshProxyCommand.this.session.getIdleTimeout().toMillis()));
                        }
                    }
                    super.preClose();
                }
            };
            ((ConnectionService)this.session.getService(ConnectionService.class)).registerChannel((Channel)this.channel);
        }
        catch (Exception e) {
            this.finishWithError("Error creating exec channel", e);
            return;
        }
        this.channel.setIn((InputStream)new NoCloseInputStream(this.inputStream));
        this.channel.setErr((OutputStream)new NoCloseOutputStream(this.errorStream));
        this.channel.setOut((OutputStream)new NoCloseOutputStream(this.outputStream));
        try {
            openFuture = this.channel.open();
        }
        catch (Exception e) {
            this.finishWithError("Error opening exec channel", e);
            return;
        }
        openFuture.awaitUninterruptibly();
        if (openFuture.isOpened()) {
            Set ret = this.channel.waitFor(EnumSet.of(ClientChannelEvent.CLOSED, ClientChannelEvent.TIMEOUT), 0L);
            log.debug("Session wait: {}", (Object)ret);
            this.finishCleanly();
        } else {
            this.finishWithError("Error opening exec channel", openFuture.getException());
        }
    }

    private String mapPaths(String command) {
        if (this.connectionData.getMapPathFrom() != null) {
            String replacement;
            String toReplace = "'" + this.connectionData.getMapPathFrom() + "'";
            String commandWithPathsReplaced = command.replaceAll(toReplace, replacement = "'" + this.connectionData.getMapPathTo() + "'");
            if (commandWithPathsReplaced.equals(command)) {
                if (command.contains(replacement)) {
                    log.info("Already replaced " + command);
                } else {
                    log.warn("Did not replace anything in " + command);
                }
            } else {
                log.debug("Replaced paths from: {} to {}", (Object)command, (Object)commandWithPathsReplaced);
            }
            return commandWithPathsReplaced;
        }
        return command;
    }

    synchronized void closeSession() {
        log.debug("Closing session");
        if (this.session != null) {
            this.session.close(true);
            this.session = null;
        }
    }

    private void finishCleanly() {
        log.debug("Connection finished");
        this.closeSession();
        this.exitCallback.onExit(0);
    }

    private void finishWithError(String msg, Throwable t) {
        this.connectionData.reportProxyError(msg, t);
        StringBuilder sb = new StringBuilder("SSH proxy tried to connect to [").append(this.connectionData.getRemoteUserName()).append("@").append(this.connectionData.getRemoteAddress()).append("]: and failed due to the following error: [").append(msg).append(']');
        if (t == null) {
            sb.append(" no exception reported");
        }
        String logMessage = sb.toString();
        log.debug(logMessage, t);
        String errorMessage = t == null ? logMessage : sb.append(": ").append(t.getMessage()).toString();
        this.finishWithError(errorMessage);
    }

    private void finishWithError(String errorMessage) {
        this.writeToErrorStream(errorMessage);
        this.exitCallback.onExit(255);
        this.closeSession();
    }

    private void writeToErrorStream(String errorMessage) {
        try {
            String msg = "BAMBOO-SSH-PROXY: [" + errorMessage + "]\n";
            this.errorStream.write(msg.getBytes(StandardCharsets.UTF_8));
            this.errorStream.flush();
        }
        catch (IOException e) {
            log.warn("Cannot write error message to the client", (Throwable)e);
        }
    }
}

