package com.atlassian.crowd.search.builder;

import com.atlassian.crowd.embedded.api.SearchRestriction;
import com.atlassian.crowd.search.query.entity.restriction.BooleanRestriction;
import com.atlassian.crowd.search.query.entity.restriction.BooleanRestrictionImpl;
import com.atlassian.crowd.search.query.entity.restriction.NullRestriction;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Builder for MultiTermRestrictions.
 *
 * For usage see <code>QueryBuilder</code>.
 */
public class Combine {
    /**
     * Returns an <tt>OR</tt> boolean search restriction where only one or more of the search restrictions have to be
     * satisfied.
     *
     * @param restrictions search restrictions
     * @return <tt>OR</tt> boolean search restriction
     */
    public static BooleanRestriction anyOf(SearchRestriction... restrictions) {
        return new BooleanRestrictionImpl(BooleanRestriction.BooleanLogic.OR, restrictions);
    }

    /**
     * Returns an <tt>AND</tt> boolean search restriction where all of the search restrictions have to be satisfied.
     *
     * @param restrictions search restrictions
     * @return <tt>AND</tt> boolean search restriction
     */
    public static BooleanRestriction allOf(SearchRestriction... restrictions) {
        return new BooleanRestrictionImpl(BooleanRestriction.BooleanLogic.AND, restrictions);
    }

    /**
     * Returns an <tt>OR</tt> boolean search restriction where only one or more of the search restrictions have to be
     * satisfied.
     *
     * @param restrictions search restrictions
     * @return <tt>OR</tt> boolean search restriction
     */
    public static BooleanRestriction anyOf(Collection<SearchRestriction> restrictions) {
        return new BooleanRestrictionImpl(BooleanRestriction.BooleanLogic.OR, restrictions);
    }

    /**
     * Returns an <tt>AND</tt> boolean search restriction where all of the search restrictions have to be satisfied.
     *
     * @param restrictions search restrictions
     * @return <tt>AND</tt> boolean search restriction
     */
    public static BooleanRestriction allOf(Collection<SearchRestriction> restrictions) {
        return new BooleanRestrictionImpl(BooleanRestriction.BooleanLogic.AND, restrictions);
    }


    /**
     * Returns a combined restriction where all of the non-empty search restrictions have to be satisfied.
     * @param optionalRestrictions restrictions, {@link com.atlassian.crowd.search.query.entity.restriction.NullRestriction#INSTANCE}
     *                             are considered empty and skipped
     * @return combined search restriction
     */
    public static SearchRestriction optionalAllOf(SearchRestriction... optionalRestrictions) {
        List<SearchRestriction> restrictions = Stream.of(optionalRestrictions)
                .filter(restriction -> restriction != NullRestriction.INSTANCE)
                .collect(Collectors.toList());
        return allOfIfNeeded(restrictions);
    }

    public static SearchRestriction allOfIfNeeded(Collection<SearchRestriction> restrictions) {
        return combineIfNeeded(restrictions, BooleanRestriction.BooleanLogic.AND);
    }

    public static SearchRestriction anyOfIfNeeded(Collection<SearchRestriction> restrictions) {
        return combineIfNeeded(restrictions, BooleanRestriction.BooleanLogic.OR);
    }

    private static SearchRestriction combineIfNeeded(Collection<SearchRestriction> restrictions, BooleanRestriction.BooleanLogic booleanLogic) {
        if (restrictions.isEmpty()) {
            return NullRestriction.INSTANCE;
        } else if (restrictions.size() == 1) {
            return restrictions.iterator().next();
        } else {
            return new BooleanRestrictionImpl(booleanLogic, restrictions);
        }
    }
}
