package com.atlassian.crowd.integration.rest.service;

import java.net.URI;
import java.net.URISyntaxException;

import com.atlassian.crowd.service.client.ClientProperties;

import org.apache.commons.lang3.math.NumberUtils;
import org.apache.http.HttpHost;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.conn.UnsupportedSchemeException;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.cache.CacheConfig;
import org.apache.http.impl.client.cache.CachingHttpClients;
import org.apache.http.impl.conn.DefaultSchemePortResolver;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Default implementation of {@link HttpClientProvider} that creates http clients with client-side caching support.
 * <p>
 * This implementation uses a {@link PoolingHttpClientConnectionManager}, which pools connections on a per-route basis,
 * and is able to service connection requests from multiple execution threads. Clients should call {@link CloseableHttpClient#close()}
 * to cleanup resources held by the connection manager.
 */
public class DefaultHttpClientProvider implements HttpClientProvider {
    /**
     * The maximum number of cache entries the http client cache will retain.
     */
    protected static final int MAX_CACHE_ENTRIES = 10;

    /**
     * The maximum response body size in bytes that will be eligible for caching in the http client.
     */
    protected static final int MAX_OBJECT_SIZE = 16384;

    /**
     * Default maximum number of connections in the pool.
     */
    protected static final int DEFAULT_MAX_TOTAL_CONNECTIONS = 20;

    @Override
    public CloseableHttpClient getClient(ClientProperties clientProperties) {
        checkNotNull(clientProperties, "clientProperties is required");

        HttpRoute httpRoute = routeFor(clientProperties);
        PoolingHttpClientConnectionManager connectionManager = getHttpClientConnectionManager(clientProperties, httpRoute);
        RequestConfig requestConfig = getRequestConfig(clientProperties, connectionManager, httpRoute);

        return getHttpClientBuilder(connectionManager, requestConfig).build();
    }

    protected Registry<ConnectionSocketFactory> getConnectionSocketFactories() {
        return RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", PlainConnectionSocketFactory.getSocketFactory())
                .register("https", SSLConnectionSocketFactory.getSystemSocketFactory()).build();
    }

    protected PoolingHttpClientConnectionManager getHttpClientConnectionManager(ClientProperties clientProperties, HttpRoute httpRoute) {
        PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager(getConnectionSocketFactories(), null, null);
        connectionManager.setMaxTotal(NumberUtils.toInt(clientProperties.getHttpMaxConnections(), DEFAULT_MAX_TOTAL_CONNECTIONS));

        // Also set per host connection limit because HttpClient has a default maximum connections per host of 2
        // (see client.getHttpConnectionManager().getParams().getMaxConnectionsPerHost())
        connectionManager.setMaxPerRoute(httpRoute, NumberUtils.toInt(clientProperties.getHttpMaxConnections(), connectionManager.getMaxTotal()));
        return connectionManager;
    }

    protected HttpClientBuilder getHttpClientBuilder(HttpClientConnectionManager connectionManager, RequestConfig requestConfig) {
        return CachingHttpClients.custom()
                .setCacheConfig(getCacheConfig())
                .setDefaultRequestConfig(requestConfig)
                .setConnectionManager(connectionManager)
                .useSystemProperties();
    }

    protected CacheConfig getCacheConfig() {
        return CacheConfig.custom()
                .setMaxCacheEntries(MAX_CACHE_ENTRIES)
                .setMaxObjectSize(MAX_OBJECT_SIZE)
                .setHeuristicCachingEnabled(false)
                .setSharedCache(false)
                .setAsynchronousWorkersMax(0)
                .build();
    }

    protected RequestConfig getRequestConfig(ClientProperties clientProperties, PoolingHttpClientConnectionManager connectionManager, HttpRoute httpRoute) {
        RequestConfig.Builder requestConfigBuilder = RequestConfig.custom();
        setRequestConfig(requestConfigBuilder, clientProperties, connectionManager, httpRoute);
        initProxyConfiguration(requestConfigBuilder, clientProperties);
        return requestConfigBuilder.build();
    }

    /**
     * Create an {@link HttpRoute} with a specific port by filling
     * in the default with {@link DefaultSchemePortResolver}.
     */
    protected HttpRoute routeFor(ClientProperties clientProperties) {
        try {
            final URI uri = new URI(clientProperties.getBaseURL());
            HttpHost host = new HttpHost(uri.getHost(), uri.getPort(), uri.getScheme());
            int port = DefaultSchemePortResolver.INSTANCE.resolve(host);

            HttpHost portedHost = new HttpHost(host.getHostName(), port, host.getSchemeName());
            return new HttpRoute(portedHost, null, "https".equalsIgnoreCase(portedHost.getSchemeName()));
        } catch (URISyntaxException | UnsupportedSchemeException e) {
            throw new IllegalArgumentException(e);
        }
    }

    private void setRequestConfig(RequestConfig.Builder builder, ClientProperties clientProperties,
                                  PoolingHttpClientConnectionManager connectionManager, HttpRoute httpRoute) {
        builder.setConnectTimeout(NumberUtils.toInt(clientProperties.getHttpTimeout(), 5000));
        builder.setSocketTimeout(NumberUtils.toInt(clientProperties.getSocketTimeout(), 10 * 60 * 1000));

        connectionManager.setMaxTotal(NumberUtils.toInt(clientProperties.getHttpMaxConnections(), DEFAULT_MAX_TOTAL_CONNECTIONS));

        // Also set per host connection limit because HttpClient has a default maximum connections per host of 2
        // (see client.getHttpConnectionManager().getParams().getMaxConnectionsPerHost())
        connectionManager.setMaxPerRoute(httpRoute,
                NumberUtils.toInt(clientProperties.getHttpMaxConnections(), connectionManager.getMaxTotal()));
    }

    private void initProxyConfiguration(RequestConfig.Builder builder, ClientProperties clientProperties) {
        if (clientProperties.getHttpProxyHost() != null) {
            HttpHost proxy = new HttpHost(clientProperties.getHttpProxyHost(), NumberUtils.toInt(clientProperties.getHttpProxyPort(), -1));
            builder.setProxy(proxy);
        }
    }
}
