package com.atlassian.crowd.embedded.api;

import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableList;

import javax.annotation.Nullable;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static org.apache.commons.lang3.StringUtils.stripToNull;

/**
 * Information of a directory synchronisation round.
 */
public class DirectorySynchronisationRoundInformation implements Serializable {
    private static final long serialVersionUID = 1L;

    private final long startTime;
    private final long durationMs;
    @Nullable
    private final String statusKey;
    @Nullable
    private final List<Serializable> statusParameters;
    @Nullable
    private final String nodeId;
    @Nullable
    private final String nodeName;
    @Nullable
    private final String incrementalSyncError;
    @Nullable
    private final String fullSyncError;

    @Deprecated
    public DirectorySynchronisationRoundInformation(long startTime, long durationMs, @Nullable String statusKey,
                                                    @Nullable List<Serializable> statusParameters) {
        this(builder()
                .setStartTime(startTime)
                .setDurationMs(durationMs)
                .setStatusKey(statusKey)
                .setStatusParameters(statusParameters));
    }

    protected DirectorySynchronisationRoundInformation(Builder builder) {
        this.startTime = builder.startTime;
        this.durationMs = builder.durationMs;
        this.statusKey = builder.statusKey;
        this.statusParameters = builder.statusParameters != null ? ImmutableList.copyOf(builder.statusParameters) : null;
        this.nodeId = stripToNull(builder.nodeId);
        this.nodeName = stripToNull(builder.nodeName);
        this.incrementalSyncError = stripToNull(builder.incrementalSyncError);
        this.fullSyncError = stripToNull(builder.fullSyncError);
    }

// The code below has been generated by BoB the Builder of Beans based on the class' fields.
// Everything after this comment will be regenerated if you invoke BoB again.
// If you don't know who BoB is, you can find him here: https://bitbucket.org/atlassianlabs/bob-the-builder-of-beans

    /**
     * Returns the time in milliseconds of the directory synchronisation time.
     *
     * @return the time in milliseconds of the directory synchronisation time
     */
    public long getStartTime() {
        return startTime;
    }

    /**
     * Returns in milliseconds the duration of the synchronisation.
     *
     * @return duration of the synchronisation in milliseconds
     */
    public long getDurationMs() {
        return durationMs;
    }

    /**
     * Returns a status key that can be used to get a human readable synchronisation status message.
     *
     * @return status key
     */
    @Nullable
    public String getStatusKey() {
        return statusKey;
    }

    /**
     * Returns parameters for status key retrieved from {@link #getStatusKey()}.
     *
     * @return parameters for status key
     */
    @Nullable
    public List<Serializable> getStatusParameters() {
        return statusParameters;
    }

    /**
     * Returns the id of the node that performed the synchronization.
     *
     * @return node id
     */
    public Optional<String> getNodeId() {
        return Optional.ofNullable(nodeId);
    }

    /**
     * Returns the name of the node that performed the synchronization.
     *
     * @return node name
     */
    public Optional<String> getNodeName() {
        return Optional.ofNullable(nodeName);
    }

    /**
     * Returns the incremental synchronisation failure reason.
     *
     * @return failure reason or null
     */
    public Optional<String> getIncrementalSyncError() {
        return Optional.ofNullable(incrementalSyncError);
    }

    /**
     * Returns the full synchronisation failure reason.
     *
     * @return failure reason or null
     */
    public Optional<String> getFullSyncError() {
        return Optional.ofNullable(fullSyncError);
    }

    public static DirectorySynchronisationRoundInformation.Builder builder() {
        return new DirectorySynchronisationRoundInformation.Builder();
    }

    public static DirectorySynchronisationRoundInformation.Builder builder(DirectorySynchronisationRoundInformation data) {
        return new DirectorySynchronisationRoundInformation.Builder(data);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        DirectorySynchronisationRoundInformation that = (DirectorySynchronisationRoundInformation) o;

        return Objects.equals(this.getStartTime(), that.getStartTime()) && Objects.equals(this.getDurationMs(), that.getDurationMs()) && Objects.equals(this.getStatusKey(), that.getStatusKey()) && Objects.equals(this.getStatusParameters(), that.getStatusParameters()) && Objects.equals(this.getNodeId(), that.getNodeId()) && Objects.equals(this.getNodeName(), that.getNodeName()) && Objects.equals(this.getIncrementalSyncError(), that.getIncrementalSyncError()) && Objects.equals(this.getFullSyncError(), that.getFullSyncError());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getStartTime(), getDurationMs(), getStatusKey(), getStatusParameters(), getNodeId(), getNodeName(), getIncrementalSyncError(), getFullSyncError());
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("startTime", getStartTime())
                .add("durationMs", getDurationMs())
                .add("statusKey", getStatusKey())
                .add("statusParameters", getStatusParameters())
                .add("nodeId", getNodeId())
                .add("nodeName", getNodeName())
                .add("incrementalSyncError", getIncrementalSyncError())
                .add("fullSyncError", getFullSyncError())
                .toString();
    }

    public static final class Builder {

        private long startTime;
        private long durationMs;
        private String statusKey;
        private List<Serializable> statusParameters = null;
        private String nodeId;
        private String nodeName;
        private String incrementalSyncError;
        private String fullSyncError;

        private Builder() {
        }

        private Builder(DirectorySynchronisationRoundInformation initialData) {
            this.startTime = initialData.getStartTime();
            this.durationMs = initialData.getDurationMs();
            this.statusKey = initialData.getStatusKey();
            this.statusParameters = initialData.getStatusParameters() != null ? new ArrayList<>(initialData.getStatusParameters()) : null;
            this.nodeId = initialData.getNodeId().orElse(null);
            this.nodeName = initialData.getNodeName().orElse(null);
            this.incrementalSyncError = initialData.getIncrementalSyncError().orElse(null);
            this.fullSyncError = initialData.getFullSyncError().orElse(null);
        }

        public Builder setStartTime(long startTime) {
            this.startTime = startTime;
            return this;
        }

        public Builder setDurationMs(long durationMs) {
            this.durationMs = durationMs;
            return this;
        }

        public Builder setStatusKey(@Nullable String statusKey) {
            this.statusKey = statusKey;
            return this;
        }

        public Builder setStatusParameters(@Nullable List<Serializable> statusParameters) {
            this.statusParameters = statusParameters;
            return this;
        }

        public Builder addStatusParameter(Serializable statusParameter) {
            this.statusParameters.add(statusParameter);
            return this;
        }

        public Builder addStatusParameters(Iterable<Serializable> statusParameters) {
            for (Serializable statusParameter : statusParameters) {
                addStatusParameter(statusParameter);
            }
            return this;
        }

        public Builder setNodeId(@Nullable String nodeId) {
            this.nodeId = nodeId;
            return this;
        }

        public Builder setNodeName(@Nullable String nodeName) {
            this.nodeName = nodeName;
            return this;
        }

        public Builder setIncrementalSyncError(@Nullable String incrementalSyncError) {
            this.incrementalSyncError = incrementalSyncError;
            return this;
        }

        public Builder setFullSyncError(@Nullable String fullSyncError) {
            this.fullSyncError = fullSyncError;
            return this;
        }

        public DirectorySynchronisationRoundInformation build() {
            return new DirectorySynchronisationRoundInformation(this);
        }
    }
}
