package com.atlassian.devrel.servlet;

import com.atlassian.devrel.plugin.PlatformComponents;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.sal.api.UrlMode;
import com.atlassian.sal.api.auth.LoginUriProvider;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.templaterenderer.TemplateRenderer;
import com.google.common.collect.Maps;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import java.util.Locale;
import java.util.Map;

/**
 * Serves the developer toolbox home page.
 */
public class HomepageServlet extends RequiresAdminServlet {

    private static final String HOMEPAGE_TEMPLATE = "/templates/homepage.vm";

    private static final String I18N_AVAILABLE_KEY = "i18nAvailable";
    private static final String I18N_STATE_KEY = "i18nActive";
    private static final String SDK_VERSION_KEY = "sdkVersion";    
    private static final String SPEAKEASY_AVAILABLE_KEY = "speakeasyAvailable";
    private static final String ACTIVEOBJECTS_AVAILABLE_KEY = "aoAvailable";
    private static final String PLUGIN_DATA_EDITOR_AVAILABLE_KEY = "pdeAvailable";
    private static final String SYSTEM_INFO_LINK_KEY = "systemInfoLink";
    private static final String PLATFORM_COMPONENTS_KEY = "platformComponents";
    private static final String JAVADOC_KEY = "javadocUrl";
    private static final String APP_DOC_KEY = "appDevUrl";
    private static final String DISPLAY_NAME = "displayName";
    private static final String VERSION_KEY = "version";
    private static final String BASEURL_KEY = "baseUrl";
    private static final String PRODUCT_KEY = "product";
    private static final String DOWNLOAD_KEY = "downloadUrl";

    private static final String JAVADOC_TEMPLATE = "http://developer.atlassian.com/static/javadoc/%s/%s?utm_source=developer-toolbox-plugin&utm_medium=homepage-link&utm_campaign=developer-toolbox";
    private static final String BITBUCKET_DOC_TEMPLATE = "https://developer.atlassian.com/bitbucket/server/docs/%s";
    private static final String STASH_DOC_TEMPLATE = "https://developer.atlassian.com/stash/docs/%s";
    private static final String DOC_TEMPLATE = "http://developer.atlassian.com/display/%s?utm_source=developer-toolbox-plugin&utm_medium=homepage-link&utm_campaign=developer-toolbox";
    private static final String DOWNLOAD_TEMPLATE = "http://www.atlassian.com/software/%s/download?utm_source=developer-toolbox-plugin&utm_medium=homepage-link&utm_campaign=developer-toolbox";

    private final ApplicationProperties applicationProperties;
    private final PluginAccessor pluginAccessor;
    private final PlatformComponents platformComponents;

    public HomepageServlet(UserManager userManager, TemplateRenderer renderer, LoginUriProvider loginUriProvider,
                           ApplicationProperties applicationProperties, PluginAccessor pluginAccessor,
                           PlatformComponents platformComponents) {
        super(userManager, renderer, loginUriProvider);
        this.applicationProperties = applicationProperties;
        this.pluginAccessor = pluginAccessor;
        this.platformComponents = platformComponents;
    }

    @Override
    public Map<String, Object> getContext(HttpServletRequest req) {
        Map<String, Object> context = Maps.newHashMap();

        Application application = Application.getApplication(applicationProperties);

        boolean available = isI18nTranslationAvailable(application);
        context.put(I18N_AVAILABLE_KEY, available);
        if (available) {
            context.put(I18N_STATE_KEY, getI18nState(req, application));
        }
        
        context.put(SDK_VERSION_KEY, getSdkVersion());
        context.put(SPEAKEASY_AVAILABLE_KEY, isSpeakeasyAvailable());
        context.put(ACTIVEOBJECTS_AVAILABLE_KEY, isActiveObjectsAvailable());
        context.put(PLUGIN_DATA_EDITOR_AVAILABLE_KEY, isPluginDataEditorAvailable());        
        context.put(SYSTEM_INFO_LINK_KEY, getSystemInfoLink(application));
        context.put(PLATFORM_COMPONENTS_KEY, platformComponents.getPlatformComponents());
        context.put(JAVADOC_KEY, getJavadocLink(application));
        context.put(APP_DOC_KEY, getAppDevDocLink(application));
        context.put(DISPLAY_NAME, getDisplayName(application));
        context.put(VERSION_KEY, applicationProperties.getVersion());
        context.put(BASEURL_KEY, getBaseUrl());
        context.put(PRODUCT_KEY, application == null ? "" : application.name().toLowerCase(Locale.ENGLISH));
        context.put(DOWNLOAD_KEY, getDownloadUrl(application));

        return context;
    }

    private String getDownloadUrl(Application application) {
        if (application == null) {
            return String.format(DOWNLOAD_TEMPLATE, applicationProperties.getDisplayName().toLowerCase(Locale.ENGLISH));
        } else {
            if (application == Application.STASH) {
                application = Application.BITBUCKET;
            }
            return String.format(DOWNLOAD_TEMPLATE, application.name().toLowerCase(Locale.ENGLISH));
        }
    }

    @Override
    public String getTemplatePath() {
        return HOMEPAGE_TEMPLATE;
    }

    private boolean isI18nTranslationAvailable(Application application) {
        return application == Application.JIRA || application == Application.CONFLUENCE;
    }

    private boolean getI18nState(HttpServletRequest req, Application application) {
        HttpSession session = req.getSession(false);
        if (application == Application.JIRA) {
            return session.getAttribute("com.atlassian.jira.util.i18n.I18nTranslationModeSwitch") != null;
        } else if (application == Application.CONFLUENCE) {
            Object o = session.getAttribute("confluence.i18n.mode");
            if (o != null) {
                return o.getClass().getSimpleName().equals("LightningTranslationMode");
            }
        }

        return false;
    }

    private String getAppDevDocLink(Application application) {
        if (application == Application.BITBUCKET) {
            return String.format(BITBUCKET_DOC_TEMPLATE, applicationProperties.getVersion());
        } else if (application == Application.STASH) {
            return String.format(STASH_DOC_TEMPLATE, applicationProperties.getVersion());
        } else {
            String spaceNameForProduct;
            if (application == Application.FISHEYE || application == Application.CRUCIBLE) {
                spaceNameForProduct = "FECRUDEV";
            } else if (application == Application.CONFLUENCE) {
                spaceNameForProduct = "CONFDEV";
            } else if (application == null) {
                spaceNameForProduct = applicationProperties.getDisplayName().toUpperCase(Locale.ENGLISH);
            } else {
                spaceNameForProduct = application.name() + "DEV";
            }
            return String.format(DOC_TEMPLATE, spaceNameForProduct);
        }
    }

    private String getJavadocLink(Application application) {
        String javadocContext;
        if (application == null) {
            javadocContext = applicationProperties.getDisplayName().toLowerCase(Locale.ENGLISH);
        } else if (application == Application.BITBUCKET) {
            javadocContext = "bitbucket-server";
        } else {
            javadocContext = application.name().toLowerCase(Locale.ENGLISH);
        }
        return String.format(JAVADOC_TEMPLATE, javadocContext, applicationProperties.getVersion());
    }

    private String getSdkVersion() {
        return System.getProperty("atlassian.sdk.version", "3.7 or earlier");
    }    
    
    private boolean isSpeakeasyAvailable() {
        return pluginAccessor.getEnabledPlugin("com.atlassian.labs.speakeasy-plugin") != null;
    }
    
    private boolean isActiveObjectsAvailable() {
        return pluginAccessor.getEnabledPlugin("com.atlassian.activeobjects.activeobjects-plugin") != null;
    }

    private boolean isPluginDataEditorAvailable() {
        return pluginAccessor.getEnabledPlugin("com.atlassian.plugins.plugin-data-editor") != null;
    }
    
    private String getSystemInfoLink(Application application) {
        if (application == Application.JIRA) {
            return getBaseUrl() + "/secure/admin/ViewSystemInfo.jspa";
        } else if (application == Application.CONFLUENCE || application == Application.BAMBOO) {
            return getBaseUrl() + "/admin/systeminfo.action";
        } else if (application == Application.FISHEYE || application == Application.CRUCIBLE) {
            return getBaseUrl() + "/admin/sysinfo.do";
        } else {
            // can't get right now
            return null;
        }
    }

    private String getBaseUrl() {
        return applicationProperties.getBaseUrl(UrlMode.RELATIVE);
    }

    private String getDisplayName(Application application) {
        String name = applicationProperties.getDisplayName();
        if (application == Application.BITBUCKET) {
            return name + " Server";
        } else {
            return name;
        }
    }

    private enum Application {
        JIRA, CONFLUENCE, FISHEYE, CRUCIBLE, BAMBOO, STASH, BITBUCKET;

        private static Application getApplication(ApplicationProperties properties) {
            String displayName = properties.getDisplayName();
            for (Application application : values()) {
                if (application.name().equalsIgnoreCase(displayName)) {
                    return application;
                }
            }
            return null;
        }
    }
}
