package com.atlassian.diagnostics.internal.platform.monitor.operatingsystem;

import com.atlassian.diagnostics.MonitoringService;
import com.atlassian.diagnostics.Severity;
import com.atlassian.diagnostics.internal.InitializingMonitor;
import com.atlassian.diagnostics.internal.platform.monitor.operatingsystem.cpu.HighCPUUsageEvent;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import java.io.File;
import java.text.DecimalFormat;
import java.time.Instant;

/**
 * Monitor for Operating System related alerts.
 * {@link OperatingSystemMonitor#init} must be called to begin operations
 */
public class OperatingSystemMonitor extends InitializingMonitor {

    private static final int CPU_HIGH_USAGE_ISSUE_ID = 2001;
    private static final int SYSTEM_LOAD_AVERAGE_INCREASED_ISSUE_ID = 2002;
    private static final int LOW_RAM_MEMORY_ID = 2003;
    private static final int LOW_FREE_DIRECTORY_DISK_SPACE_ID = 2004;
    private static final int INACCESSIBLE_SYSTEM_DIRECTORY_ID = 2005;
    private static final String KEY_PREFIX = "diagnostics.operating.system.issue";
    private static final String CPU_USAGE_KEY = "cpuUsagePercentage";

    private final OperatingSystemMonitorConfiguration operatingSystemMonitorConfiguration;

    public OperatingSystemMonitor(final OperatingSystemMonitorConfiguration operatingSystemMonitorConfiguration) {
        this.operatingSystemMonitorConfiguration = operatingSystemMonitorConfiguration;
    }

    @Override
    public void init(final MonitoringService monitoringService) {
        monitor = monitoringService.createMonitor("CPU", "diagnostics.operating.system.name", operatingSystemMonitorConfiguration);
        defineIssue(KEY_PREFIX, CPU_HIGH_USAGE_ISSUE_ID, Severity.WARNING);
        defineIssue(KEY_PREFIX, SYSTEM_LOAD_AVERAGE_INCREASED_ISSUE_ID, Severity.WARNING);
        defineIssue(KEY_PREFIX, LOW_RAM_MEMORY_ID, Severity.WARNING, null);
        defineIssue(KEY_PREFIX, LOW_FREE_DIRECTORY_DISK_SPACE_ID, Severity.WARNING, null);
        defineIssue(KEY_PREFIX, INACCESSIBLE_SYSTEM_DIRECTORY_ID, Severity.WARNING, null);
    }

    /**
     * Raises an alert for high CPU usage.
     *
     * @param timestamp         the timestamp to be used on the alert
     * @param highCpuUsageEvent the snapshot of the CPU
     */
    public void raiseAlertForHighCpu(@Nonnull final Instant timestamp, @Nonnull final HighCPUUsageEvent highCpuUsageEvent) {
        alert(CPU_HIGH_USAGE_ISSUE_ID, builder -> builder
                .timestamp(timestamp)
                .details(() -> ImmutableMap.of(
                        CPU_USAGE_KEY, new DecimalFormat("#.##").format(highCpuUsageEvent.getPercentage())
                ))
        );
    }

    /**
     * Raises an alert for low free memory.
     *
     * @param free    the free space available on Disk
     * @param total   the total space available on Disk
     * @param minimum the minimum free space for Alert
     */
    public void alertLowFreeMemory(final long free, final long total, final long minimum) {
        alert(LOW_RAM_MEMORY_ID, builder -> builder
                .timestamp(Instant.now())
                .details(() -> createBaseAlert(free, total, minimum).build())
                .build()
        );
    }

    /**
     * Raises an alert for low free disk space.
     *
     * @param directory the directory checked for the Alert
     * @param free      the free space available in the Directory
     * @param total     the total space available in the Directory
     * @param minimum   the minimum free space for Alert
     */
    public void alertLowFreeDiskSpace(final File directory, final long free, final long total, final long minimum) {
        alert(LOW_FREE_DIRECTORY_DISK_SPACE_ID, builder -> builder
                .timestamp(Instant.now())
                .details(() -> createBaseAlert(free, total, minimum).put("directory", directory.getAbsolutePath()).build()
                ));
    }

    /**
     * Raises an alert for an inaccessible system directory.
     *
     * @param file the directory that is inaccessible
     */
    public void alertFileSystemInaccessible(final File file) {
        alert(INACCESSIBLE_SYSTEM_DIRECTORY_ID, builder -> builder
                .timestamp(Instant.now())
                .details(() -> ImmutableMap.of("directory", file.getAbsolutePath()))
                .build()
        );
    }

    private ImmutableMap.Builder<Object, Object> createBaseAlert(final long freeMemory, final long totalMemory, final long minimumMemory) {
        return ImmutableMap.builder()
                .put("freeMemoryInMegabytes", freeMemory)
                .put("totalMemoryInMegaBytes", totalMemory)
                .put("minimumMemoryInMegaBytes", minimumMemory);
    }

}
