package com.atlassian.diagnostics.internal.platform.monitor.scheduler;

import com.atlassian.diagnostics.MonitoringService;
import com.atlassian.diagnostics.Severity;
import com.atlassian.diagnostics.internal.InitializingMonitor;
import com.atlassian.diagnostics.internal.platform.plugin.AlertTriggerResolver;
import com.atlassian.diagnostics.internal.platform.plugin.BundleFinder;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static java.util.Objects.requireNonNull;

/**
 * Monitor for the scheduler related events.
 *
 * {@link SchedulerMonitor#init} must be called to begin operations.
 */
public class SchedulerMonitor extends InitializingMonitor {

    private static final String KEY_PREFIX = "diagnostics.scheduler.issue";
    private static final int HIGH_UTILIZATION_ISSUE_ID = 3001;
    private static final int SLOW_JOB_ISSUE_ID = 3002;

    private final SchedulerMonitorConfiguration schedulerMonitorConfiguration;
    private final AlertTriggerResolver alertTriggerResolver;
    private final BundleFinder bundleFinder;

    public SchedulerMonitor(final SchedulerMonitorConfiguration schedulerMonitorConfiguration,
                            final AlertTriggerResolver alertTriggerResolver,
                            final BundleFinder bundleFinder) {
        this.schedulerMonitorConfiguration = schedulerMonitorConfiguration;
        this.alertTriggerResolver = requireNonNull(alertTriggerResolver, "alertTriggerResolver");
        this.bundleFinder = bundleFinder;
    }

    @Override
    public void init(@Nonnull final MonitoringService monitoringService) {
        monitor = monitoringService.createMonitor("SCHEDULER", "diagnostics.scheduler.name", schedulerMonitorConfiguration);

        defineIssue(KEY_PREFIX, HIGH_UTILIZATION_ISSUE_ID, Severity.INFO);
        defineIssue(KEY_PREFIX, SLOW_JOB_ISSUE_ID, Severity.INFO);
    }

    /**
     * Raises an alert for scheduler high utilization.
     *
     * @param timestamp the timestamp to be used on the alert
     * @param runningJobs the running jobs
     * @param workerThreadCount the total number of worker threads
     */
    public void raiseAlertForHighUtilization(@Nonnull final Instant timestamp, @Nonnull final List<RunningJobDiagnostic> runningJobs, final int workerThreadCount) {
        alert(HIGH_UTILIZATION_ISSUE_ID, builder -> builder
                .timestamp(timestamp)
                .details(() -> highUtilizationAlertDetails(runningJobs, workerThreadCount))
        );
    }

    private Map<Object, Object> highUtilizationAlertDetails(final List<RunningJobDiagnostic> runningJobs, final int workerThreadCount) {
        final List<Map<Object, Object>> jobs = runningJobs.stream().map(job -> {
            final long runningTime = System.currentTimeMillis() - job.getRunningJob().getStartTime().getTime();
            final ImmutableMap.Builder<Object, Object> alertBuilder = ImmutableMap.builder()
                    .put("jobRunnerKey", job.getRunningJob().getJobConfig().getJobRunnerKey().toString())
                    .put("jobId", job.getRunningJob().getJobId().toString())
                    .put("jobRunningTimeInMillis", runningTime);

            job.getJobRunner().ifPresent(jobRunner -> {
                alertBuilder.put("jobRunnerClass", jobRunner.getClass().getName());
                bundleFinder.getBundleNameForClass(jobRunner.getClass())
                        .ifPresent(plugin -> alertBuilder.put("plugin", plugin));
            });

            return alertBuilder.build();
        }).collect(Collectors.toList());

        return ImmutableMap.builder()
                .put("workerThreads", workerThreadCount)
                .put("jobs", jobs)
                .build();
    }

    /**
     * Raises an alert for a scheduled job taking longer than it's configured interval.
     *  @param timestamp the timestamp to be used on the alert
     * @param diagnostic the diagnostic details
     */
    public void raiseAlertForSlowJob(@Nonnull final Instant timestamp, @Nonnull final ScheduledJobDiagnostic diagnostic) {
        alert(SLOW_JOB_ISSUE_ID, builder -> { builder
                .trigger(alertTriggerResolver.triggerForBundle(diagnostic.getJobRunner().map(Object::getClass).orElse(null)))
                .timestamp(timestamp)
                .details(() -> slowJobAlertDetails(diagnostic));
        });
    }

    private Map<Object, Object> slowJobAlertDetails(final ScheduledJobDiagnostic diagnostic) {
        final ImmutableMap.Builder<Object, Object> builder = ImmutableMap.builder()
                .put("jobRunnerKey", diagnostic.getJobDetails().getJobRunnerKey().toString())
                .put("jobId", diagnostic.getJobDetails().getJobId().toString());

        diagnostic.getLastRun().ifPresent(lastRun -> builder.put("jobDurationInMillis", diagnostic.getLastRun().get().getDurationInMillis()));

        return builder.build();
    }

}
