package com.atlassian.dragonfly.core;

import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.applinks.api.application.jira.JiraApplicationType;
import com.atlassian.applinks.spi.auth.AuthenticationConfigurationException;
import com.atlassian.applinks.spi.auth.AuthenticationScenario;
import com.atlassian.applinks.spi.link.ApplicationLinkDetails;
import com.atlassian.applinks.spi.link.MutatingApplicationLinkService;
import com.atlassian.applinks.spi.link.ReciprocalActionException;
import com.atlassian.applinks.spi.manifest.ManifestNotFoundException;
import com.atlassian.applinks.spi.util.TypeAccessor;
import com.atlassian.dragonfly.api.ApplicationLinkConfigurator;
import com.atlassian.dragonfly.api.JiraIntegrationConfigurationException;
import com.atlassian.dragonfly.spi.JiraIntegrationSetupHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URI;

public class ApplicationLinkConfiguratorImpl implements ApplicationLinkConfigurator
{
    private static final Logger LOG = LoggerFactory.getLogger(ApplicationLinkConfiguratorImpl.class);

    private final MutatingApplicationLinkService applicationLinkService;
    private final TypeAccessor typeAccessor;
    private final JiraIntegrationSetupHelper jiraIntegrationSetupHelper;

    public ApplicationLinkConfiguratorImpl(MutatingApplicationLinkService applicationLinkService,
                                           TypeAccessor typeAccessor,
                                           JiraIntegrationSetupHelper jiraIntegrationSetupHelper)
    {
        this.applicationLinkService = applicationLinkService;
        this.typeAccessor = typeAccessor;
        this.jiraIntegrationSetupHelper = jiraIntegrationSetupHelper;
    }

    public ApplicationLink configureApplicationLinks(URI jiraUrl, URI localUrl, String username, String password) throws JiraIntegrationConfigurationException
    {
        ApplicationLink applicationLink = null;
        try
        {
            applicationLink = createReciprocatedApplicationLink(jiraUrl, localUrl, username, password);
            authenticateApplicationLink(applicationLink, username, password, localUrl);
        }
        catch (JiraIntegrationConfigurationException jice)
        {
            rollbackApplicationLinkConfiguration(applicationLink);
            throw jice;
        }

        return applicationLink;
    }

    private ApplicationLink createReciprocatedApplicationLink(URI remoteRpcUrl, URI localRpcUrl, String username, String password) throws JiraIntegrationConfigurationException
    {
        try
        {
            // This creates a link from the remoteJira to local
            applicationLinkService.createReciprocalLink(remoteRpcUrl, localRpcUrl, username, password);
        }
        catch (ReciprocalActionException e)
        {
            throw new JiraIntegrationConfigurationException(String.format(
                    "Failed to create application link from JIRA server at %s " +
                            "to this %s server at %s?. Please read the " +
                            "troubleshooting guide.", remoteRpcUrl, jiraIntegrationSetupHelper.getApplicationType().getDisplayName(), localRpcUrl),
                    e);
        }

        final JiraApplicationType jiraType = typeAccessor.getApplicationType(JiraApplicationType.class);
        if (jiraType == null)
        {
            throw new JiraIntegrationConfigurationException("Failed to load the application type: " + JiraApplicationType.class + ". " +
                    "Have you disabled some modules of the Application Links plugin?");
        }

        ApplicationLink applicationLink;

        try
        {
            final ApplicationLinkDetails linkDetails = ApplicationLinkDetails.builder()
                    .rpcUrl(remoteRpcUrl)
                    .displayUrl(remoteRpcUrl) // just set to the RPC URL
                    .isPrimary(true)
                    .name(generateLinkName(remoteRpcUrl))
                    .build();

            applicationLink = applicationLinkService.createApplicationLink(jiraType, linkDetails);
        }
        catch (ManifestNotFoundException e)
        {
            // this should never be thrown by the JiraManifestRetriever implementation, but still
            throw new JiraIntegrationConfigurationException("Failed to retrieve manifest from the remote JIRA server. Is your JIRA server " +
                    "running and accessible from the server that FishEye is installed on?", e);
        }

        return applicationLink;
    }

    protected void authenticateApplicationLink(ApplicationLink applicationLink, String username, String password, URI localRpcUrl) throws JiraIntegrationConfigurationException
    {
        final AuthenticationScenario authenticationScenario = new AuthenticationScenario()
        {
            public boolean isCommonUserBase()
            {
                // we're delegating to JIRA for user management, so userbase is definitely common
                return true;
            }

            public boolean isTrusted()
            {
                // TODO warn the user in the UI that trust is implicit
                return true;
            }
        };

        try
        {
            applicationLinkService.configureAuthenticationForApplicationLink(applicationLink, authenticationScenario, username, password);
        }
        catch (AuthenticationConfigurationException e)
        {
            throw new JiraIntegrationConfigurationException(String.format(
                    "Failed to authenticate application link between JIRA server at %s " +
                            "to this %s server at %s?. Please read the " +
                            "troubleshooting guide.", applicationLink.getRpcUrl(), jiraIntegrationSetupHelper.getApplicationType().getDisplayName(), localRpcUrl),
                    e);
        }
    }

    public void rollbackApplicationLinkConfiguration(ApplicationLink applicationLink)
    {
        // try to clean up local state changes, so the user can retry the connection after manually clearing state
        // from the target JIRA instance (as per the troubleshooting guide)
        try
        {
            if (applicationLink != null)
            {
                applicationLinkService.deleteApplicationLink(applicationLink);
            }
            // TODO: How to delete the reverse application link?
            LOG.info("Rolled back 2-way application link to JIRA.");
        }
        catch (Exception rollbackException)
        {
            // this is possibly what caused the original failure necessitating the rollback, log & ignore
            LOG.error("Failed to rollback local UAL/Crowd configuration", rollbackException);
        }
    }

    private static String generateLinkName(final URI remoteRpcUrl)
    {
        String name = "JIRA";
        if (remoteRpcUrl.getHost() != null)
        {
            name = remoteRpcUrl.getHost() + " " + name;
        }
        return name;
    }
}
