package com.atlassian.dragonfly.core;

import com.atlassian.crowd.model.application.ApplicationType;
import com.atlassian.dragonfly.api.ApplicationNameGenerator;
import com.atlassian.dragonfly.api.CrowdApplicationEntity;
import com.atlassian.dragonfly.api.CrowdIntegrationConfigurator;
import com.atlassian.dragonfly.api.JiraIntegrationConfigurationException;
import com.atlassian.dragonfly.spi.JiraIntegrationSetupHelper;
import com.atlassian.security.random.DefaultSecureTokenGenerator;
import org.apache.commons.httpclient.Credentials;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.ByteArrayRequestEntity;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.lang.StringUtils;


import javax.xml.bind.JAXB;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URI;
import java.text.SimpleDateFormat;
import java.util.Date;

/***
 * CrowdIntegrationConfigurator implementation which uses applink for performing Rest requests (as a part of embedded crowd integration).
 */
public class CrowdIntegrationConfiguratorImpl implements CrowdIntegrationConfigurator
{
    private final JiraIntegrationSetupHelper jiraIntegrationSetupHelper;
    private final ApplicationNameGenerator applicationNameGenerator;

    public CrowdIntegrationConfiguratorImpl(JiraIntegrationSetupHelper jiraIntegrationSetupHelper, String hostName, String id)
    {
        this(jiraIntegrationSetupHelper, new DefaultApplicationNameGenerator(StringUtils.defaultString(hostName), id, jiraIntegrationSetupHelper.getApplicationType().getDisplayName()));
    }

    public CrowdIntegrationConfiguratorImpl(JiraIntegrationSetupHelper jiraIntegrationSetupHelper, ApplicationNameGenerator applicationNameGenerator)
    {
        this.jiraIntegrationSetupHelper = jiraIntegrationSetupHelper;
        this.applicationNameGenerator = applicationNameGenerator;
    }

    public CrowdApplicationEntity configureCrowdAuthentication(URI jiraUrl, String username, String password) throws JiraIntegrationConfigurationException
    {
        try
        {
            CrowdApplicationEntity applicationEntity = createApplicationInCrowd(jiraUrl, username, password);
            jiraIntegrationSetupHelper.switchToCrowdAuthentication(jiraUrl, applicationEntity.getName(), applicationEntity.getPassword());
            return applicationEntity;
        }
        catch (JiraIntegrationConfigurationException jice)
        {
            rollbackCrowdAuthenticationConfiguration();
            throw jice;
        }
    }

    public void rollbackCrowdAuthenticationConfiguration()
    {
        jiraIntegrationSetupHelper.switchToDefaultAuthentication();
    }

    private CrowdApplicationEntity createApplicationInCrowd(final URI jiraUrl, final String username, final String password)
                                                                                                    throws JiraIntegrationConfigurationException
    {
        CrowdApplicationEntity applicationEntity = createApplicationEntity();

        // see https://extranet.atlassian.com/display/INTEGRATION/Application+Management+REST+API#ApplicationManagementRESTAPI-ApplicationResourceforJaaCS
        final HttpClient client = new HttpClient();
        client.getParams().setAuthenticationPreemptive(true);
        final Credentials credentials = new UsernamePasswordCredentials(username, password);
        client.getState().setCredentials(new AuthScope(jiraUrl.getHost(), -1), credentials);

        final PostMethod post = new PostMethod(jiraUrl.toString() + "/rest/appmanagement/1/application?include-request-address=true");
        final ByteArrayOutputStream bs = new ByteArrayOutputStream();
        JAXB.marshal(applicationEntity, bs);
        post.setRequestEntity(new ByteArrayRequestEntity(bs.toByteArray(), "application/xml"));
        post.setRequestHeader("Accept", "application/xml");

        try
        {
            int statusCode = client.executeMethod(post);
            if (!isSuccess(statusCode))
            {
                throw new JiraIntegrationConfigurationException("cannot create application in Jira. Status Code =" + statusCode);
            }
        }
        catch (IOException e)
        {
            throw new JiraIntegrationConfigurationException("cannot create application in Jira", e);
        }
        finally
        {
            post.releaseConnection();
        }

        return applicationEntity;
    }

    /**
     * Return true if the status code is successful (2xx).
     *
     * @param statusCode HTTP status code
     * @return true if the status code is successful (2xx)
     */
    private boolean isSuccess(int statusCode)
    {
        return statusCode >= 200 && statusCode < 300;
    }

    private CrowdApplicationEntity createApplicationEntity()
    {
        final ApplicationType applicationType = jiraIntegrationSetupHelper.getApplicationType();

        // create the application name
        final String appname = applicationNameGenerator.generateApplicationName();

        // create the application password
        final String password = DefaultSecureTokenGenerator.getInstance().generateToken();

        // create the optional description
        final String description = "Automatically created by the setup of " + applicationType.getDisplayName()
                + " on " + SimpleDateFormat.getDateInstance().format(new Date());

        // create the entity
        return new CrowdApplicationEntity(applicationType, appname, password, description, true);
    }
}