package org.apache.lucene.queryParser;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.CachingTokenFilter;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.tokenattributes.CharTermAttribute;
import org.apache.lucene.analysis.tokenattributes.PositionIncrementAttribute;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.MultiPhraseQuery;
import org.apache.lucene.search.PhraseQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.util.Version;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;

/**
 * An extended version of the original Lucene {@link QueryParser} with the following set of additions:
 *
 * <ul>
 *      <li>Allows the host application to reuse the field query creation algorithm and specify an
 *      {@link Analyzer analyzer} when overriding {@link #getFieldQuery(String, String, boolean)}.</li>
 * </ul>
 *
 * @since v3.3.0-atlassian-2
 */
public class ExtendedQueryParser extends QueryParser
{
    public ExtendedQueryParser(Version matchVersion, String f, Analyzer a)
    {
        super(matchVersion, f, a);
    }

    protected ExtendedQueryParser(CharStream stream)
    {
        super(stream);
    }

    protected ExtendedQueryParser(QueryParserTokenManager tm)
    {
        super(tm);
    }

    @Override
    protected Query getFieldQuery(String field, String queryText, boolean quoted) throws ParseException
    {
        return newFieldQuery(analyzer, field, queryText, quoted);
    }

    /**
     * Creates a new field query using the specified {@link Analyzer analyzer} instance for reuse in sub-classes.
     *
     * <p>Contains the same logic specified in the {@code getFieldQuery} method of the {@code LuceneQueryParser} parent
     * class with an additional analyzer parameter so we can specify a different analyzer.</p>
     *
     * <p>
     *     Backported to Lucene 3.3.0, as per the fix to
     *     <a href="https://issues.apache.org/jira/browse/LUCENE-2892">LUCENE-2892</a>.
     * </p>
     * <p>
     *     <strong>IMPORTANT:</strong>
     *     <ul>
     *          <li>Please update this algorithm to match the behaviour of
     *          {@code super.getFieldQuery(String field, String queryText, boolean quoted)} when updating the lucene
     *          version targeted by this library.
     *          </li>
     *          <li>This is part of lucene core since 4.0-ALPHA, so it is only needed for versions before that.</li>
     *     </ul>
     * </p>
     */
    protected Query newFieldQuery(Analyzer analyzer, String field, String queryText, boolean quoted)  throws ParseException
    {
        // Use the analyzer to get all the tokens, and then build a TermQuery,
        // PhraseQuery, or nothing based on the term count

        TokenStream source;
        try
        {
            source = analyzer.reusableTokenStream(field, new StringReader(queryText));
            source.reset();
        } catch (IOException e)
        {
            source = analyzer.tokenStream(field, new StringReader(queryText));
        }
        CachingTokenFilter buffer = new CachingTokenFilter(source);
        CharTermAttribute termAtt = null;
        PositionIncrementAttribute posIncrAtt = null;
        int numTokens = 0;

        boolean success = false;
        try
        {
            buffer.reset();
            success = true;
        } catch (IOException e)
        {
            // success==false if we hit an exception
        }
        if (success)
        {
            if (buffer.hasAttribute(CharTermAttribute.class))
            {
                termAtt = buffer.getAttribute(CharTermAttribute.class);
            }
            if (buffer.hasAttribute(PositionIncrementAttribute.class))
            {
                posIncrAtt = buffer.getAttribute(PositionIncrementAttribute.class);
            }
        }

        int positionCount = 0;
        boolean severalTokensAtSamePosition = false;

        boolean hasMoreTokens = false;
        if (termAtt != null)
        {
            try
            {
                hasMoreTokens = buffer.incrementToken();
                while (hasMoreTokens)
                {
                    numTokens++;
                    int positionIncrement = (posIncrAtt != null) ? posIncrAtt.getPositionIncrement() : 1;
                    if (positionIncrement != 0)
                    {
                        positionCount += positionIncrement;
                    } else
                    {
                        severalTokensAtSamePosition = true;
                    }
                    hasMoreTokens = buffer.incrementToken();
                }
            } catch (IOException e)
            {
                // ignore
            }
        }
        try
        {
            // rewind the buffer stream
            buffer.reset();

            // close original stream - all tokens buffered
            source.close();
        } catch (IOException e)
        {
            // ignore
        }

        if (numTokens == 0)
            return null;
        else if (numTokens == 1)
        {
            String term = null;
            try
            {
                boolean hasNext = buffer.incrementToken();
                assert hasNext == true;
                term = termAtt.toString();
            } catch (IOException e)
            {
                // safe to ignore, because we know the number of tokens
            }
            return newTermQuery(new Term(field, term));
        } else
        {
            if (severalTokensAtSamePosition || (!quoted && !getAutoGeneratePhraseQueries()))
            {
                if (positionCount == 1 || (!quoted && !getAutoGeneratePhraseQueries()))
                {
                    // no phrase query:
                    BooleanQuery q = newBooleanQuery(positionCount == 1);

                    BooleanClause.Occur occur = positionCount > 1 && getDefaultOperator() == AND_OPERATOR ?
                            BooleanClause.Occur.MUST : BooleanClause.Occur.SHOULD;

                    for (int i = 0; i < numTokens; i++)
                    {
                        String term = null;
                        try
                        {
                            boolean hasNext = buffer.incrementToken();
                            assert hasNext == true;
                            term = termAtt.toString();
                        } catch (IOException e)
                        {
                            // safe to ignore, because we know the number of tokens
                        }

                        Query currentQuery = newTermQuery(
                                new Term(field, term));
                        q.add(currentQuery, occur);
                    }
                    return q;
                } else
                {
                    // phrase query:
                    MultiPhraseQuery mpq = newMultiPhraseQuery();
                    mpq.setSlop(phraseSlop);
                    List<Term> multiTerms = new ArrayList<Term>();
                    int position = -1;
                    for (int i = 0; i < numTokens; i++)
                    {
                        String term = null;
                        int positionIncrement = 1;
                        try
                        {
                            boolean hasNext = buffer.incrementToken();
                            assert hasNext == true;
                            term = termAtt.toString();
                            if (posIncrAtt != null)
                            {
                                positionIncrement = posIncrAtt.getPositionIncrement();
                            }
                        } catch (IOException e)
                        {
                            // safe to ignore, because we know the number of tokens
                        }

                        if (positionIncrement > 0 && multiTerms.size() > 0)
                        {
                            if (enablePositionIncrements)
                            {
                                mpq.add(multiTerms.toArray(new Term[0]), position);
                            } else
                            {
                                mpq.add(multiTerms.toArray(new Term[0]));
                            }
                            multiTerms.clear();
                        }
                        position += positionIncrement;
                        multiTerms.add(new Term(field, term));
                    }
                    if (enablePositionIncrements)
                    {
                        mpq.add(multiTerms.toArray(new Term[0]), position);
                    } else
                    {
                        mpq.add(multiTerms.toArray(new Term[0]));
                    }
                    return mpq;
                }
            } else
            {
                PhraseQuery pq = newPhraseQuery();
                pq.setSlop(phraseSlop);
                int position = -1;


                for (int i = 0; i < numTokens; i++)
                {
                    String term = null;
                    int positionIncrement = 1;

                    try
                    {
                        boolean hasNext = buffer.incrementToken();
                        assert hasNext == true;
                        term = termAtt.toString();
                        if (posIncrAtt != null)
                        {
                            positionIncrement = posIncrAtt.getPositionIncrement();
                        }
                    } catch (IOException e)
                    {
                        // safe to ignore, because we know the number of tokens
                    }

                    if (enablePositionIncrements)
                    {
                        position += positionIncrement;
                        pq.add(new Term(field, term), position);
                    } else
                    {
                        pq.add(new Term(field, term));
                    }
                }
                return pq;
            }
        }
    }
}