package com.atlassian.sal.api.features;

import com.atlassian.fugue.Option;

import java.util.Set;

/**
 * Provides a cross-product method for determining whether a dark feature is enabled.
 *
 * Implementing products can back these checks with their own internal dark feature management system, but must follow
 * the enable and disable dark features on startup based on system properties and the contents of an optional properties file.
 *
 * Dark feature keys must begin with atlassian.darkfeature. Values must be either true or false.
 * Location of dark features property file can be overridden with darkfeatures.properties.file system property.
 *
 * See SystemDarkFeatureInitializer in sal-core for implementation.
 */
public interface DarkFeatureManager
{
    /**
     * Checks if a dark feature is enabled
     * @param darkFeatureSystemPropertyKey key of the feature to be checked
     * @return true if enabled, false otherwise
     */
    public boolean isSystemFeatureEnabled(String darkFeatureSystemPropertyKey);

    /**
     * Checks if a dark feature is enabled for the current user (must be called within the context of a request)
     * @param darkFeatureSystemPropertyKey key of the feature to be checked
     * @return true if enabled, false otherwise
     */
    public boolean isUserFeatureEnabled(String darkFeatureSystemPropertyKey);

    /**
     * Checks if a dark feature is enabled for the user with name username
     *
     * @param username the name of the user being queried
     * @param darkFeatureSystemPropertyKey key of the feature to be checked
     * @return an Option.Some containing Boolean.TRUE if enabled for the user, containing Boolean.FALSE otherwise. If no user
     * exists with name username then returns Option.None
     */
    public Option<Boolean> isUserFeatureEnabledForUser(String username, String darkFeatureSystemPropertyKey);

    /**
     * Gets all enabled dark features for the current user (must be called within the context of a request)
     *
     * @return a Set containing the keys for all dark features applicable for the current user (a union of all system
     * dark features and user dark features enabled for current user).
     */
    public Set<String> getAllEnabledFeaturesForCurrentUser();

    /**
     * Gets all enabled dark features for a given user
     *
     * @param username the name of the user being queried
     * @return a Set containing the keys for all dark features applicable for the given user (a union of all system
     * dark features and user dark features enabled for given user).
     */
    public Option<Set<String>> getAllEnanbedFeaturesForUser(String username);
}
