package com.atlassian.sal.api.pluginsettings;

/**
 * Factory for creating {@link PluginSettings} instances, each scoped to a namespace.
 * <p>
 * A namespace is a scope or container that isolates related settings. Within each namespace, individual settings are
 * stored using keys. Some products may restrict valid namespaces to product-specific entity identifiers.
 *
 * @since 2.0
 * @see PluginSettings
 */
public interface PluginSettingsFactory {

    /**
     * @deprecated since 8.0. The method name "createSettingsForKey" is misleading because the parameter represents a
     * namespace (scope), not a key. The returned {@link PluginSettings} instance defines methods like
     * {@link PluginSettings#get(String key)} where "key" refers to individual settings within that namespace. Use
     * {@link #createSettingsForNamespace(String)} instead, which clarifies this namespace vs. key distinction.
     */
    @Deprecated(forRemoval = true)
    default PluginSettings createSettingsForKey(String namespace) {
        return createSettingsForNamespace(namespace);
    }

    /**
     * Creates a {@link PluginSettings} instance scoped to a specific namespace.
     * <p>
     * A namespace acts as a scope or container for related settings. Within each namespace, you can store individual
     * settings using keys via methods like {@link PluginSettings#get(String)},
     * {@link PluginSettings#put(String, Object)}, etc.
     * <p>
     * <strong>Important:</strong> Products often correlate namespaces to product-specific entities. For example,
     * Confluence may use space keys as namespaces for space-related settings, and Jira may use project keys
     * for project-related settings. When choosing a namespace strategy, consider the following:
     * <ul>
     * <li><strong>Using a shared namespace</strong> (e.g., a space or project key): Use specific, unique keys
     * to avoid conflicts with other plugins that may store unrelated settings in the same namespace.
     * Consider prefixing your keys with your plugin key.</li>
     * <li><strong>Using a dedicated namespace</strong>: If you need exclusive control over all keys within
     * a namespace, use a unique namespace identifier, such as your plugin key.</li>
     * </ul>
     *
     * @param namespace the namespace scope for the settings. Can be {@code null} to retrieve the "global" namespace
     *                  (equivalent to {@link #createGlobalSettings()})
     * @return a {@link PluginSettings} instance scoped to the specified namespace
     * @throws IllegalArgumentException if the product does not support the specified namespace
     * @since 8.0
     */
    default PluginSettings createSettingsForNamespace(String namespace) {
        return createSettingsForKey(namespace);
    }

    /**
     * Creates a {@link PluginSettings} instance scoped to the global namespace.
     * <p>
     * The global namespace is shared across all components of the product. Because this namespace is accessible to all
     * plugins and product components, key collisions are a significant risk. Always use specific, unique keys to avoid
     * conflicts—prefixing your keys with your plugin key is strongly recommended.
     * <p>
     * For settings that require isolation or exclusive control over keys, consider using
     * {@link #createSettingsForNamespace(String)} with a unique namespace identifier instead.
     *
     * @return a {@link PluginSettings} instance scoped to the global namespace
     * @see #createSettingsForNamespace(String) for guidance on choosing between shared and dedicated namespaces
     */
    PluginSettings createGlobalSettings();
}
