package com.atlassian.sal.core.net;

import com.atlassian.sal.api.net.NonMarshallingRequestFactory;
import com.atlassian.sal.api.net.Request.MethodType;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.sal.core.trusted.CertificateFactory;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.params.HttpConnectionManagerParams;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Does NOT support json/xml oject marshalling. Use the atlassian-rest implementation of
 * {@link com.atlassian.sal.api.net.RequestFactory} instead.
 */
public class HttpClientRequestFactory implements NonMarshallingRequestFactory<HttpClientRequest>
{
    private static final Logger log = LoggerFactory.getLogger(HttpClientRequestFactory.class);

    private final CertificateFactory certificateFactory;
    private final UserManager userManager;

    public HttpClientRequestFactory(final CertificateFactory certificateFactory, final UserManager userManager)
    {
        this.certificateFactory = certificateFactory;
        this.userManager = userManager;
    }

    /**
     * The default time to wait without retrieving data from the remote connection
     */
    public static final int DEFAULT_SOCKET_TIMEOUT=Integer.parseInt(System.getProperty("http.socketTimeout", "10000"));

    /**
     * The default time allowed for establishing a connection
     */
    public static final int DEFAULT_CONNECTION_TIMEOUT=Integer.parseInt(System.getProperty("http.connectionTimeout", "10000"));

    /* (non-Javadoc)
     * @see com.atlassian.sal.api.net.RequestFactory#createMethod(com.atlassian.sal.api.net.Request.MethodType, java.lang.String)
     */
    public HttpClientRequest createRequest(final MethodType methodType, final String url)
    {
        final HttpClient httpClient = getHttpClient(url);
        return new HttpClientRequest(httpClient, methodType, url, certificateFactory, userManager);
    }

    /**
     * @param url The URL
     * @return The HTTP client
     */
    protected HttpClient getHttpClient(final String url)
    {
        final HttpClient httpClient = new HttpClient();
        configureProtocol(httpClient, url);
        configureProxy(httpClient, url);
        configureConnectionParameters(httpClient);
        return httpClient;
    }

    /**
     * Applies a set of parameters to a client
     *
     * @param httpClient the client to which parameters are applied
     */
    protected void configureConnectionParameters(final HttpClient httpClient)
    {
        final HttpConnectionManagerParams params = httpClient.getHttpConnectionManager().getParams();
        params.setSoTimeout(DEFAULT_SOCKET_TIMEOUT);
        params.setConnectionTimeout(DEFAULT_CONNECTION_TIMEOUT);
    }

    /**
     * @param client The client to configure the proxy of
     * @param remoteUrl The remote URL
     */
    protected void configureProxy(final HttpClient client, final String remoteUrl)
    {
        new HttpClientProxyConfig().configureProxy(client, remoteUrl);
    }

    protected void configureProtocol(final HttpClient httpClient, final String remoteUrl)
    {
        HttpClientProtocolConfig.configureProtocol(httpClient, remoteUrl);
    }

    public boolean supportsHeader()
    {
        return true;
    }
}
