package com.atlassian.sal.core.scheduling;

import com.atlassian.sal.api.component.ComponentLocator;
import com.atlassian.sal.api.scheduling.PluginJob;
import com.atlassian.sal.api.scheduling.PluginScheduler;
import org.quartz.Job;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SimpleTrigger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Arrays;
import java.util.Date;
import java.util.Map;

/**
 * Plugin scheduler that uses Quartz's Scheduler.
 */
public class QuartzPluginScheduler implements PluginScheduler
{
    private static final String JOB_CLASS_KEY = "pluginJobClass";
    private static final String JOB_DATA_MAP_KEY = "pluginJobDataMap";
    private static final Logger log = LoggerFactory.getLogger(QuartzPluginScheduler.class);
    private static final String DEFAULT_GROUP = "pluginSchedulerJobGroup";

    public void scheduleJob(final String name, final Class<? extends PluginJob> job, final Map<String, Object> jobDataMap, final Date startTime, final long repeatInterval)
    {
        // Create a new job detail
        final JobDetail jobDetail = new JobDetail();
        jobDetail.setGroup(DEFAULT_GROUP);
        jobDetail.setName(name);
        jobDetail.setJobClass(QuartzPluginJob.class);
        jobDetail.setJobDataMap(getJobMap(job, jobDataMap));

        // Create a new trigger
        final SimpleTrigger trigger = new SimpleTrigger();
        trigger.setGroup("pluginSchedulerTriggerGroup");
        trigger.setName(name + "Trigger");
        if (startTime != null)
        {
            trigger.setStartTime(startTime);
        }
        if (repeatInterval == 0)
        {
            trigger.setRepeatCount(0);
        }
        else
        {
            trigger.setRepeatInterval(repeatInterval);
            trigger.setRepeatCount(SimpleTrigger.REPEAT_INDEFINITELY);
        }

        // Schedule job
        try
        {
            final Scheduler scheduler = getScheduler();
            scheduler.scheduleJob(jobDetail, trigger);
        }
        catch (final SchedulerException se)
        {
            log.error("Error scheduling job " +name, se);
        }
    }

    /**
     * A Quartz job that executes a PluginJob
     */
    public static class QuartzPluginJob implements Job
    {
        @SuppressWarnings("unchecked")
        public void execute(final JobExecutionContext jobExecutionContext) throws JobExecutionException
        {
            final JobDataMap map = jobExecutionContext.getJobDetail().getJobDataMap();
            final Class<? extends PluginJob> jobClass = (Class<? extends PluginJob>) map.get(JOB_CLASS_KEY);
            final Map<String, Object> pluginJobMap = (Map<String, Object>) map.get(JOB_DATA_MAP_KEY);
            // Instantiate the job
            PluginJob job;
            try
            {
                job = jobClass.newInstance();
            }
            catch (final InstantiationException ie)
            {
                throw new JobExecutionException("Error instantiating job", ie, false);
            }
            catch (final IllegalAccessException iae)
            {
                throw new JobExecutionException("Cannot access job class", iae, false);
            }
            job.execute(pluginJobMap);
        }
    }

    public void unscheduleJob(final String name)
    {
        try
        {
            final Scheduler scheduler = getScheduler();
            if (!Arrays.asList(scheduler.getJobNames(DEFAULT_GROUP)).contains(name))
            {
                throw new IllegalArgumentException("Error unscheduling job. Job '" + name + "' is not scheduled ");
            }
            scheduler.deleteJob(name, DEFAULT_GROUP);
        } catch (final SchedulerException e)
        {
            throw new IllegalArgumentException("Error unscheduling job " + name, e);
        }
    }

    protected JobDataMap getJobMap(Class<? extends PluginJob> job, Map<String, Object> jobDataMap) {
        final JobDataMap jobDetailMap = new JobDataMap();
        jobDetailMap.put(JOB_CLASS_KEY, job);
        jobDetailMap.put(JOB_DATA_MAP_KEY, jobDataMap);
        return jobDetailMap;
    }

    protected Scheduler getScheduler() {
        return ComponentLocator.getComponent(Scheduler.class);
    }

}
