package com.atlassian.user;

import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.security.password.Credential;
import com.atlassian.user.security.password.PasswordEncryptor;
import com.atlassian.user.impl.DuplicateEntityException;

public interface UserManager extends EntityManager
{
    /**
     * @return a {@link Pager} holding all users being managed.
     * @throws EntityException
     */
    Pager<User> getUsers()  throws EntityException;

    /**
     * @return a {@link Pager} holding the names of all users being managed.
     * @throws EntityException
     */
    Pager<String> getUserNames() throws EntityException;

    /**
     * @return a <code>null</code> or a {@link User} if one could be found.
     */
    User getUser(String username) throws EntityException;

    /**
     * Creates a new user with the username provided. Returns the newly created user.
     *
     * @return a {@link User} object representing the new user.
     * @throws DuplicateEntityException if a user with the username already exists.
     * @throws UnsupportedOperationException - if {@link UserManager#isCreative()} returns <code>false</code>.
     * @deprecated since 3.0 use {@link #createUser(User, Credential)} because this is an inconvenient way to create users now that the
     * setFullName() and setEmail() methods are no longer on the User interface
     */
    User createUser(String username) throws EntityException;

    /**
     * Creates a new user with the provided user details and encrypted password. Returns the newly created user.
     *
     * @param userTemplate the user template to use, which must have a non-null user name
     * @param credential the user's password
     * @return the newly created user, which should be used for subsequent operations on that user
     * @throws DuplicateEntityException if a user with the same name already exists
     * @throws RepositoryException if there is a problem communicating with the underlying storage mechanism
     * @throws UnsupportedOperationException if {@link #isCreative()} returns {@code false}
     * @throws IllegalArgumentException if the user, user name or credential is null
     */
    User createUser(User userTemplate, Credential credential)
        throws EntityException, UnsupportedOperationException, IllegalArgumentException;

    /**
     * Encrypts the plain password, sets it on the user, and saves the user.
     * Implementations supporting this will usually have an internal {@link PasswordEncryptor}.
     *
     * @throws UnsupportedOperationException - if {@link UserManager#isCreative()} returns <code>false</code>.
     */
    void alterPassword(User user, String plainTextPass) throws EntityException;

    /**
     * Renames the user to the new name.
     */
    User renameUser(final User user, final String userName) throws EntityException;

    /**
     * Notifies the manager of a user rename done externally
     */
    User onExternalUserRename(final String oldName, final String newName) throws EntityException;
    /**
     * Persists any changes made to the user to the storage system used by this user manager.
     * <p/>
     * To ensure consistent behaviour across all repository types, clients must call this method after
     * changing any properties on a {@link User}.
     *
     * @throws UnsupportedOperationException - if {@link UserManager#isCreative()} returns {@code false}
     * @throws IllegalArgumentException if the user is null or not managed by this repository
     * @return the saved user, which may be a different object to the one passed in
     */
    User saveUser(User user) throws EntityException, IllegalArgumentException;

    /**
     * Removes the specified user from the repository.
     *
     * @throws UnsupportedOperationException if {@link UserManager#isReadOnly(User)} returns <code>true</code>.
     * @throws IllegalArgumentException if the user is null or not managed by this repository
     */
    void removeUser(User user) throws EntityException, IllegalArgumentException;

    /**
     * @return true indicates that information on the user object cannot be altered in the storage system
     * (see {@link com.atlassian.user.repository.RepositoryIdentifier}),
     * false indicates that the storage system will save changes or that this {@link UserManager} does not 
     * know about the {@link User}.
     */
    boolean isReadOnly(User user) throws EntityException;
}

