package com.atlassian.user.search.page;

/**
 * Will page through results in a database, LDAP system, etc., as required by the iterator it hands out.
 *
 * The purpose of the pager is to preload a {@link List} of results, hand out on a single iterator which
 * can walk the results and transform each one, and transparently reconnect to the
 * remote system when more data is required.
 *
 * Clients can use the call to iterator directly or can rely upon reading the current page and calling nextPage().
 */

import java.util.Iterator;
import java.util.List;

public interface Pager<T> extends Iterable<T>
{
//    final Pager EMPTY_PAGER = DefaultPager.emptyPager();

    /**
     * Maximum number of elements to hold in memory at a given moment.
     */
    final int PRELOAD_LIMIT = 100;
    final int NO_POSITION = -1;

    boolean isEmpty();

    /**
     * Use this if you want a typical iterator over the entire data.
     *
     * @return an {@link Iterator} for the entire result set.
     */
    Iterator<T> iterator();

    /**
     * @return a single, preloaded page.
     */
    List<T> getCurrentPage();

    void nextPage();

    boolean onLastPage();

    /**
     * Will run the index up to this point. Calling {@link Pager#getCurrentPage()} will
     * then return a page holding this index.
     *
     * @param index the zero-based index of the item to skip to.
     * @throws PagerException - if the number of items in the backing data is exceeded by the index.
     */
    void skipTo(int index) throws PagerException;

    /**
     * @return the current index position of the pager
     */
    int getIndex();

    /**
     * @return the index of the first item in the current page, relative to the start of the set
     */
    int getIndexOfFirstItemInCurrentPage();
}