package com.atlassian.user.impl.cache;

import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheFactory;
import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.User;
import com.atlassian.user.UserManager;
import com.atlassian.user.impl.DefaultUser;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.security.password.Credential;
import org.apache.log4j.Logger;

public class CachingUserManager implements UserManager
{
    private static final Logger log = Logger.getLogger(CachingUserManager.class);

    private final UserManager underlyingUserManager;
    private final CacheFactory cacheFactory;

    private String userCacheName = null;
    private String userROCacheName = null;
    private String repositoryCacheName = null;

    private static final String CACHE_SUFFIX_USERS = "users";
    private static final String CACHE_SUFFIX_USERS_RO = "users_ro";
    private static final String CACHE_SUFFIX_REPOSITORIES = "repository";

    protected static User NULL_USER = new DefaultUser()
    {
        public String toString()
        {
            return "NULL USER";
        }
    };

    public CachingUserManager(UserManager underlyingUserManager, CacheFactory cacheFactory)
    {
        this.underlyingUserManager = underlyingUserManager;
        this.cacheFactory = cacheFactory;
    }

    public Pager<User> getUsers() throws EntityException
    {
        return underlyingUserManager.getUsers();
    }

    public Pager<String> getUserNames() throws EntityException
    {
        return underlyingUserManager.getUserNames();
    }

    /**
     * Caches users retrieved.<br>
     * Will also cache the fact that a user could not be found (so that we don't incur the expense of another search when have previously determined that a user doesn't exist)
     * Hence this method will also cache null results.
     *
     * @return - an {@link com.atlassian.user.User} if one could be found, otherwise null.
     * @throws com.atlassian.user.EntityException
     *          - representing the exception which prohibited looking for or
     *          retrieving the user.
     */
    public User getUser(String username) throws EntityException
    {
        User cachedUser = (User) getUserCache().get(username);
        if (cachedUser != null)
        {
            return NULL_USER.equals(cachedUser) ? null : cachedUser;
        }
        else
        {
            User user = underlyingUserManager.getUser(username);
            cacheUser(username, user);
            return user;
        }
    }

    private void cacheUser(String username, User user)
    {
        getUserCache().put(username, user == null ? NULL_USER : user);
    }

    private void cacheRepository(String username, RepositoryIdentifier repository)
    {
        getRepositoryCache().put(username, repository);
    }

    private void cacheUserROFlag(User user, boolean ro)
    {
        getUserROFlagCache().put(user.getName(), ro);
    }

    private Cache getUserCache()
    {
        synchronized(this)
        {
            if (userCacheName == null)
                userCacheName = getCacheKey(CACHE_SUFFIX_USERS);
        }
        return cacheFactory.getCache(userCacheName);
    }

    private Cache getUserROFlagCache()
    {
        synchronized(this)
        {
            if (userROCacheName == null)
                userROCacheName = getCacheKey(CACHE_SUFFIX_USERS_RO);
        }
        return cacheFactory.getCache(userROCacheName);
    }

    private Cache getRepositoryCache()
    {
        synchronized(this)
        {
            if (repositoryCacheName == null)
                repositoryCacheName = getCacheKey(CACHE_SUFFIX_REPOSITORIES);
        }
        return cacheFactory.getCache(repositoryCacheName);
    }

    public User createUser(String username) throws EntityException
    {
        User user = underlyingUserManager.createUser(username);

        if (user != null)
            cacheUser(user.getName(), user);

        return user;
    }

    public User createUser(User userTemplate, Credential credential) throws EntityException
    {
        User user = underlyingUserManager.createUser(userTemplate, credential);

        if (user != null)
            cacheUser(user.getName(), user);

        return user;
    }

    /**
     * Encrypts the plain password, sets it on the user, and saves the user.
     */
    public void alterPassword(User user, String plainTextPass) throws EntityException
    {
        underlyingUserManager.alterPassword(user, plainTextPass);
        if (user != null)
            cacheUser(user.getName(), underlyingUserManager.getUser(user.getName()));
    }

    @Override
    public User renameUser(final User user, final String userName) throws EntityException
    {
        final User renamedUser = underlyingUserManager.renameUser(user, userName);

        updateCacheAfterRename(user.getName(), renamedUser);
        return renamedUser;
    }

    @Override
    public User onExternalUserRename(final String oldName, final String newName) throws EntityException
    {
        final User renamedUser = underlyingUserManager.onExternalUserRename(oldName, newName);

        updateCacheAfterRename(oldName, renamedUser);
        return renamedUser;
    }

    private void updateCacheAfterRename(final String oldName, final User renamedUser)
    {
        getUserCache().remove(oldName);
        if (renamedUser != null)
        {
            cacheUser(renamedUser.getName(), renamedUser);
        }
    }

    public User saveUser(User user) throws EntityException
    {
        final User savedUser = underlyingUserManager.saveUser(user);

        if (savedUser != null)
            cacheUser(savedUser.getName(), savedUser);

        return savedUser;
    }

    /**
     * Removes the specified group, if it is present.
     */
    public void removeUser(User user) throws EntityException
    {
        if (log.isDebugEnabled())
            log.debug("removing user: " + user.getName());
        underlyingUserManager.removeUser(user);
        if (log.isDebugEnabled())
            log.debug("user " + user.getName() + " removed from underlying user manager " + underlyingUserManager.getIdentifier().getName());

        try
        {
            if (log.isDebugEnabled())
                log.debug("removing user from cache: " + user.getName());
            removeUserFromCache(user);
            if (log.isDebugEnabled())
            {
                log.debug("removed user from cache: " + user.getName());
                // Lets be paranoid
                // TODO delete it
                if (getUserCache().get(user.getName()) != null)
                {
                    log.error("WTF???");    
                }
            }
        }
        catch (Exception e)
        {
            throw new EntityException("User removed in underlying repository but could not remove from cache");
        }
    }

    private void removeUserFromCache(User user)
    {
        if (user != null)
            getUserCache().remove(user.getName());
    }

    /**
     * @return true indicates that information on the user object cannot be altered in the storage system
     *         (see {@link com.atlassian.user.repository.RepositoryIdentifier}),
     *         false indicates that the storage system will save changes or that this {@link com.atlassian.user.UserManager} does not
     *         know about the {@link com.atlassian.user.User}.
     *
     */
    public boolean isReadOnly(User user) throws EntityException
    {
        Boolean cachedROFlag = (Boolean) getUserROFlagCache().get(user.getName());

        if (cachedROFlag == null)
        {
            boolean ro = underlyingUserManager.isReadOnly(user);
            cacheUserROFlag(user, ro);
            return ro;
        }
        else
        {
            return cachedROFlag;
        }
    }

    public RepositoryIdentifier getIdentifier()
    {
        return underlyingUserManager.getIdentifier();
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} in which the entity is stored, otherwise null.
     */
    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        final RepositoryIdentifier cachedRepository = (RepositoryIdentifier) getRepositoryCache().get(entity.getName());
        if (cachedRepository != null)
            return cachedRepository;

        // do not cache null RepositoryIdentifier - it means that underlying UserManager is not handling this entity
        final RepositoryIdentifier repository = underlyingUserManager.getRepository(entity);
        if (repository != null)
        {
            cacheRepository(entity.getName(), repository);
        }
        else if (log.isDebugEnabled())
        {
            log.debug(String.format("UserManager %s does not contain entity %s",
                    underlyingUserManager.getIdentifier(), entity.getName()));
        }
        return repository;
    }

    /**
     * Used to determine whether an entity can be created (eg, can call {@link com.atlassian.user.UserManager#createUser(String)} or
     * {@link com.atlassian.user.GroupManager#createGroup(String)}
     *
     * @return true to indicate that {@link com.atlassian.user.Entity} objects can be created by this manager, or false to indicate
     *         not.
     */
    public boolean isCreative()
    {
        return underlyingUserManager.isCreative();
    }

    /**
     * Generates a unique cache key. This cache key should not be shared by other instances of CachingUserManager that delegate to different underlying user managers.
     * @return cache key
     */
    private String getCacheKey(String cacheName)
    {
        String className = underlyingUserManager.getClass().getName();
        String repositoryKey = underlyingUserManager.getIdentifier().getKey(); // use the repository key to compose the cache key, so we can have a cache per (repository + userManager) combination.
        return className + "." + repositoryKey + "." + cacheName;
    }

}
