package com.atlassian.user.impl.delegation;

import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.Group;
import com.atlassian.user.GroupManager;
import com.atlassian.user.User;
import com.atlassian.user.UserManager;
import com.atlassian.user.impl.delegation.repository.DelegatingRepository;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.page.PagerFactory;
import com.atlassian.user.util.Assert;
import org.apache.log4j.Logger;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

/*
 * Group manager with a list of other group managers to delegate to.
 * The group managers are tried in the iteration order of the list.
 */
public class DelegatingGroupManager implements GroupManager
{
    private static final Logger log = Logger.getLogger(DelegatingGroupManager.class);

    private final List<GroupManager> groupManagers = new ArrayList<GroupManager>();

    public DelegatingGroupManager(List<GroupManager> groupManagers)
    {
        this.groupManagers.addAll(groupManagers);
    }

    /**
     * @return an unmodifiable list of {@link GroupManager}s which this manager delegates to,
     *         in the order of delegation.
     */
    public List<GroupManager> getGroupManagers()
    {
        return Collections.unmodifiableList(groupManagers);
    }

    /**
     * Returns the first group manager in iteration order which returns a non-null value from
     * {@link GroupManager#getGroup(String)}, or the last group manager in the iteration
     * order if no group manager meets this criteria.
     *
     * @param groupName the name of the group to match against
     * @throws EntityException      if any of the delegating managers throws an exception
     *                              in {@link GroupManager#getGroup(String)}.
     * @throws NullPointerException if the group is null
     */
    private GroupManager getMatchingGroupManager(String groupName) throws EntityException
    {
        GroupManager lastFoundGroupManager = null;

        for (GroupManager groupManager : groupManagers)
        {
            lastFoundGroupManager = groupManager;

            if (groupManager.getGroup(groupName) != null)
            {
                return groupManager;
            }
        }

        // returns the last groupManager if none matched
        return lastFoundGroupManager;
    }

    public Pager<Group> getGroups() throws EntityException
    {
        List<Pager<Group>> groups = new ArrayList<Pager<Group>>();

        for (GroupManager groupManager : groupManagers)
        {
            try
            {
                groups.add(groupManager.getGroups());
            }
            catch (EntityException e)
            {
                log.error("Failed to retrieve groups from group manager in delegation: " +
                        groupManager.getClass().toString() + ". Continuing with remaining managers.", e);
            }
        }

        return PagerFactory.getPager(groups);
    }

    public Pager<Group> getGroups(User user) throws EntityException
    {
        Assert.notNull(user, "User must not be null");
        List<Pager<Group>> pagers = new LinkedList<Pager<Group>>();

        for (GroupManager groupManager : groupManagers)
        {
            try
            {
                Pager<Group> groupPager = groupManager.getGroups(user);
                pagers.add(groupPager);
            }
            catch (EntityException e)
            {
                log.error("Failed to retrieve groups for user [" + user + "] from group manager: " +
                        groupManager.getClass().toString() + ". Continuing with remaining managers.", e);
            }
        }

        return PagerFactory.getPager(pagers);
    }

    public List<Group> getWritableGroups()
    {
        List<Group> groups = new ArrayList<Group>();

        for (GroupManager groupManager : groupManagers)
        {
            groups.addAll(groupManager.getWritableGroups());
        }

        return groups;
    }

    public Pager<String> getMemberNames(Group group) throws EntityException
    {
        GroupManager groupManager = getMatchingGroupManager(group.getName());

        if (groupManager == null)
        {
            return DefaultPager.emptyPager();
        }

        return groupManager.getMemberNames(group);
    }

    public Pager<String> getLocalMemberNames(Group group) throws EntityException
    {
        GroupManager groupManager = getMatchingGroupManager(group.getName());

        if (groupManager == null)
        {
            return DefaultPager.emptyPager();
        }

        return groupManager.getLocalMemberNames(group);
    }

    public Pager<String> getExternalMemberNames(Group group) throws EntityException
    {
        GroupManager groupManager = getMatchingGroupManager(group.getName());

        if (groupManager == null)
        {
            return DefaultPager.emptyPager();
        }

        return groupManager.getExternalMemberNames(group);
    }

    public Group getGroup(String groupName) throws EntityException
    {
        Iterator iter = groupManagers.iterator();
        Group foundGroup;

        while (iter.hasNext())
        {
            GroupManager groupManager = (GroupManager) iter.next();

            foundGroup = groupManager.getGroup(groupName);

            if (foundGroup != null)
            {
                return foundGroup;
            }
        }

        return null;
    }

    public Group createGroup(String groupName) throws EntityException
    {
        Iterator iter = groupManagers.iterator();
        Group createdGroup = null;

        while (iter.hasNext())
        {
            GroupManager groupManager = (GroupManager) iter.next();

            if (groupManager.isCreative())
            {
                createdGroup = groupManager.createGroup(groupName);
            }

            if (createdGroup != null)
            {
                return createdGroup;
            }
        }

        return null;
    }

    /**
     * Removes the specified group, if it is present.
     *
     * @throws EntityException - representing the exception which prohibited removal
     */
    public void removeGroup(Group group) throws EntityException
    {
        for (GroupManager groupManager : groupManagers)
        {
            if (groupManager.getGroup(group.getName()) == null)
            {
                continue;
            }

            if (groupManager.isReadOnly(group))
            {
                continue;
            }

            groupManager.removeGroup(group);
            break;
        }
    }

    public void addMembership(Group group, User user) throws EntityException
    {
        if (group == null)
        {
            throw new IllegalArgumentException("Can't add membership for null group");
        }
        GroupManager groupManager = getMatchingGroupManager(group.getName());
        groupManager.addMembership(group, user);
    }

    /**
     * Check whether this manager has a record of membership between the
     * argued user and group.
     *
     * @throws com.atlassian.user.impl.RepositoryException
     *
     */
    public boolean hasMembership(Group group, User user) throws EntityException
    {
        if (group == null)
        {
            return false;
        }
        GroupManager groupManager = getMatchingGroupManager(group.getName());
        return groupManager.hasMembership(group, user);
    }

    public void removeMembership(Group group, User user) throws EntityException
    {
        if (group == null)
        {
            throw new IllegalArgumentException("Can't remove membership for null group");
        }
        GroupManager groupManager = getMatchingGroupManager(group.getName());
        groupManager.removeMembership(group, user);
    }

    /**
     * @return true if users from other systems can be granted membership in the underlying {@link com.atlassian.user.repository.RepositoryIdentifier},
     *         otherwise false.
     * @throws EntityException
     */
    public boolean supportsExternalMembership() throws EntityException
    {
        for (GroupManager groupManager : groupManagers)
        {
            if (groupManager.supportsExternalMembership())
            {
                return true;
            }
        }

        return false;
    }

    public boolean isReadOnly(Group group) throws EntityException
    {
        for (GroupManager groupManager : groupManagers)
        {
            if (groupManager.getGroup(group.getName()) != null)
            {
                return groupManager.isReadOnly(group);
            }
        }

        return false;
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        ArrayList<RepositoryIdentifier> repositories = new ArrayList<RepositoryIdentifier>();

        for (GroupManager groupManager : groupManagers)
        {
            repositories.add(groupManager.getIdentifier());
        }

        return new DelegatingRepository(repositories);
    }

    /**
     * Don't use for now. This is very expensive for large LDAP instances. todo: get user/group entities to store which repository they came from
     *
     * @return the {@link RepositoryIdentifier} in which the entity is stored, otherwise null.
     * @throws EntityException
     */
    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        if (!(entity instanceof Group))
        {
            return null;
        }

        GroupManager groupManager = getMatchingGroupManager(entity.getName());

        return groupManager.getIdentifier();
    }

    /**
     * Used to detemine whether an entity can be created (eg, can call {@link UserManager#createUser(String)} or
     * {@link GroupManager#createGroup(String)}
     *
     * @return true to indicate that {@link Entity} objects can be created by this manager, or false to indicate
     *         not.
     */
    public boolean isCreative()
    {
        for (GroupManager groupManager : groupManagers)
        {
            if (groupManager.isCreative())
            {
                return true;
            }
        }
        return false;
    }
}
