package com.atlassian.user.impl.delegation;

import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.User;
import com.atlassian.user.UserManager;
import com.atlassian.user.impl.DefaultUser;
import com.atlassian.user.impl.DuplicateEntityException;
import com.atlassian.user.impl.delegation.repository.DelegatingRepository;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.page.PagerFactory;
import com.atlassian.user.security.password.Credential;
import com.atlassian.user.util.Assert;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/*
 * User manager with a list of other user managers to delegate to.
 * The user managers are tried in the iteration order of the list.
 */
public class DelegatingUserManager implements UserManager
{
    private final List<UserManager> userManagers;

    public DelegatingUserManager(List<UserManager> userManagers)
    {
        this.userManagers = userManagers;
    }

    public Pager<User> getUsers() throws EntityException
    {
        ArrayList<Pager<User>> pagers = new ArrayList<Pager<User>>();

        for (UserManager userManager : userManagers)
        {
            pagers.add(userManager.getUsers());
        }

        return PagerFactory.getPager(pagers);
    }

    public Pager<String> getUserNames() throws EntityException
    {
        ArrayList<Pager<String>> pagers = new ArrayList<Pager<String>>();

        for (UserManager userManager : userManagers)
        {
            pagers.add(userManager.getUserNames());
        }

        return PagerFactory.getPager(pagers);
    }

    /**
     * @return - an {@link com.atlassian.user.User} if one could be found, otherwise null.
     * @throws com.atlassian.user.EntityException
     *          - representing the exception which prohibited looking for or
     *          retrieving the user.
     */
    public User getUser(String username) throws EntityException
    {
        User foundUser = null;

        for (UserManager userManager : userManagers)
        {
            foundUser = userManager.getUser(username);

            if (foundUser != null)
                break;
        }

        return foundUser;
    }

    /**
     * @return the created user
     * @throws com.atlassian.user.EntityException If user cannot be created in any user managers.
     */
    public User createUser(String username) throws EntityException
    {
        User preexistingUser;
        try
        {
            preexistingUser = getUser(username);
        }
        catch (EntityException e)
        {
            throw new EntityException("Couldn't check whether user already exists", e);
        }

        if (preexistingUser != null)
        {
            UserManager manager = getMatchingUserManager(preexistingUser);
            if (manager == null)
            {
                // This could just mean the user has just been deleted since the first call to getUser,
                // but it may be a bug in in the getMatchingUserManager logic
                throw new DuplicateEntityException("User [" + username + "] reported to exist by unknown manager");
            }
            RepositoryIdentifier repository = manager.getRepository(preexistingUser);
            if (repository == null)
            {
                // This most likely indicates a bug in the manager.  E.g. its cache contains the user but its repo doesn't.
                // Or again it could have been removed since the call to getMatchingUserManager
                throw new DuplicateEntityException("User [" + username + "] reported to exist in unknown repository by manager: " + manager);
            }
            else
            {
                throw new DuplicateEntityException("User [" + username + "] already exists in: " + repository.getName());
            }
        }


        User createdUser = null;

        for (UserManager userManager : userManagers)
        {

            if (userManager.isCreative())
                createdUser = userManager.createUser(username);

            if (createdUser != null)
                break;
        }

        if (createdUser == null)
            throw new EntityException("Could not create user: " + username + ". " +
                "Ensure you have a read-write repository configured.");

        return createdUser;
    }


    public User createUser(User userTemplate, Credential credential) throws EntityException, UnsupportedOperationException
    {
        String username = userTemplate.getName();
        User preexistingUser;
        try
        {
            preexistingUser = getUser(username);
        }
        catch (EntityException e)
        {
            throw new EntityException("Couldn't check whether user already exists", e);
        }

        if (preexistingUser != null)
        {
            UserManager manager = getMatchingUserManager(preexistingUser);
            if (manager == null)
            {
                // This could just mean the user has just been deleted since the first call to getUser,
                // but it may be a bug in in the getMatchingUserManager logic
                throw new DuplicateEntityException("User [" + username + "] reported to exist by unknown manager");
            }
            RepositoryIdentifier repository = manager.getRepository(preexistingUser);
            if (repository == null)
            {
                // This most likely indicates a bug in the manager.  E.g. its cache contains the user but its repo doesn't.
                // Or again it could have been removed since the call to getMatchingUserManager
                throw new DuplicateEntityException("User [" + username + "] reported to exist in unknown repository by manager: " + manager);
            }
            else
            {
                throw new DuplicateEntityException("User [" + username + "] already exists in: " + repository.getName());
            }
        }


        User createdUser = null;

        for (UserManager userManager : userManagers)
        {

            if (userManager.isCreative())
                createdUser = userManager.createUser(userTemplate, credential);

            if (createdUser != null)
                break;
        }

        if (createdUser == null)
            throw new EntityException("Could not create user: " + username + ". " +
                "Ensure you have a read-write repository configured.");

        return createdUser;
    }

    /**
     * Encrypts the plain password, sets it on the user, and saves the user.
     */
    public void alterPassword(User user, String plainTextPass) throws EntityException
    {
        UserManager userManager = getMatchingUserManager(user);

        if (userManager != null)
            userManager.alterPassword(user, plainTextPass);
        else
            throw new EntityException("Cannot find a userManager responsible for user [" + user.getName() + "]");
    }

    @Override
    public User renameUser(final User user, final String userName) throws EntityException
    {
        return getNonNullUserManager(user).renameUser(user, userName);
    }

    @Override
    public User onExternalUserRename(final String oldName, final String newName) throws EntityException
    {
        UserManager userManager = getMatchingUserManager(new DefaultUser(oldName));
        if (userManager==null)
        {
            userManager = getNonNullUserManager(new DefaultUser(newName));
        }
        return userManager.onExternalUserRename(oldName, newName);
    }

    public User saveUser(User user) throws EntityException
    {
        UserManager userManager = getMatchingUserManager(user);

        if (userManager != null)
            return userManager.saveUser(user);
        else
            throw new EntityException("Cannot find a userManager responsible for user [" + user.getName() + "]");
    }

    public void removeUser(User user) throws EntityException
    {
        UserManager userManager = getMatchingUserManager(user);

        if (userManager != null)
            userManager.removeUser(user);
        else
            throw new IllegalArgumentException("Cannot find a userManager responsible for user [" + user.getName() + "]");
    }

    /**
     * @return true indicates that information on the user object cannot be altered in the storage system
     *         (see {@link com.atlassian.user.repository.RepositoryIdentifier}),
     *         false indicates that the storage system will save changes or that this {@link com.atlassian.user.UserManager} does not
     *         know about the {@link com.atlassian.user.User}.
     * @throws com.atlassian.user.EntityException
     *
     */
    public boolean isReadOnly(User user) throws EntityException
    {
        UserManager userManager = getMatchingUserManager(user);

        if (userManager != null)
            return userManager.isReadOnly(user);
        else
            throw new EntityException("Cannot find a userManager responsible for user [" + user.getName() + "]");
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        Iterator iter = userManagers.iterator();
        ArrayList<RepositoryIdentifier> repositories = new ArrayList<RepositoryIdentifier>();

        while (iter.hasNext())
        {
            UserManager userManager = (UserManager) iter.next();
            repositories.add(userManager.getIdentifier());
        }

        return new DelegatingRepository(repositories);
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} in which the entity is stored, otherwise null.
     * @throws com.atlassian.user.EntityException
     *
     */
    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {

        for (UserManager userManager : userManagers)
        {
            RepositoryIdentifier repo = userManager.getRepository(entity);

            if (repo != null)
                return repo;
        }

        return null;
    }

    /**
     * Used to detemine whether an entity can be created (eg, can call {@link com.atlassian.user.UserManager#createUser(String)} or
     * {@link com.atlassian.user.GroupManager#createGroup(String)}
     *
     * @return true to indicate that {@link com.atlassian.user.Entity} objects can be created by this manager, or false to indicate
     *         not.
     */
    public boolean isCreative()
    {
        for (UserManager userManager : userManagers)
        {
            if (userManager.isCreative())
                return true;
        }

        return false;
    }

    /**
     * Helper method to locate the first userManager responsible for the given user, in the delegation.
     */
    protected UserManager getMatchingUserManager(User user) throws EntityException
    {
        Assert.notNull(user, "User must not be null");

        for (UserManager userManager : userManagers)
        {
            User foundUser = userManager.getUser(user.getName());

            if (foundUser != null)
                return userManager;
        }

        return null;
    }

    private UserManager getNonNullUserManager(User user) throws EntityException
    {
        final UserManager userManager = getMatchingUserManager(user);

        if (userManager == null)
        {
            throw new EntityException("Cannot find a userManager responsible for user [" + user.getName() + "]");
        }
        return userManager;
    }

    /**
     * @return an unmodifiable list of {@link UserManager}s which this manager delegates to,
     * in the order of delegation.
     */
    public List getUserManagers()
    {
        return Collections.unmodifiableList(userManagers);
    }
}
