package com.atlassian.user.repository;

/**
 * Default identifier class for a repository.
 * <p/>
 * Comparisons between different repositories are done with the key.
 *
 * @see #equals(Object)
 */
public class DefaultRepositoryIdentifier implements RepositoryIdentifier
{
    private final String key;
    private final String name;

    /**
     * Create a repository with the specified key.
     *
     * @param key the repository key, which must be unique and not <code>null</code>.
     * @param name the name of the repository, which must not be <code>null</code>.
     * @throws IllegalArgumentException if either key or name is <code>null</code>.
     */
    public DefaultRepositoryIdentifier(String key, String name)
    {
        if (key == null)
            throw new IllegalArgumentException("Repository key cannot be null");
        if (name == null)
            throw new IllegalArgumentException("Repository name cannot be null");
        this.key = key;
        this.name = name;
    }

    public String getName()
    {
        return name;
    }

    public String getKey()
    {
        return key;
    }

    /**
     * Compares this repository with another repository, using the keys of both.
     *
     * @return true if the other object is a {@link RepositoryIdentifier} with an equal key, otherwise false.
     */
    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (!(o instanceof RepositoryIdentifier)) return false;

        final RepositoryIdentifier repo = (RepositoryIdentifier) o;
        return key.equals(repo.getKey());
    }

    public int hashCode()
    {
        return key.hashCode();
    }

    public String toString()
    {
        return name;
    }
}
