package com.atlassian.user.search.page;

import org.apache.log4j.Category;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

public class MergedListPager<T> implements Pager<T>
{
    private static final Category log = Category.getInstance(MergedListPager.class);
    private List<? extends Pager<T>> pagers;
    private int combinedIndex;
    public Pager<? extends T> currentPager;
    private List<T> currentPage;
    private int indexOfFirstItemInCurrentPage;
    private boolean onLastPage = false;

    MergedListPager(List<? extends Pager<T>> pagers)
    {
        this.pagers = pagers;
    }

    public boolean isEmpty()
    {
        for (Pager<T> pager : pagers)
        {
            if (!pager.isEmpty())
                return false;
        }

        return true;
    }

    /**
     * @return an iterator over all results. Note: calling this multiple times does not reset the iterator back to the first element, as DefaultPager itself does not do this.
     */
    public Iterator<T> iterator()
    {
        if (pagers == null)
        {
            return Collections.<T>emptyList().iterator();
        }
        else
        {
            return new MergedListIterator(pagers);
        }
    }


    /**
     * returns a list of results (up to the limit defined in PRELOAD_LIMIT).<br/>
     * if there is insufficient preloaded results in the first pager, it will go to the next pager and so on, until the
     * preload limit is reached or all pagers have been checked.
     *
     * Calling this method will move the position of the iterator! (like skipTo() does)
     *
     * Note: this will exhaust all the results in a pager before going to the next one. It does <b>not<b/> loop through the preloaded results of each each pager passed in.
     * Note: the position in which we start pulling elements from in the iterator is set by skipTo()
     *
     */
    public List<T> getCurrentPage()
    {
        if (currentPage == null)
        {
            currentPage = new ArrayList<T>();
            indexOfFirstItemInCurrentPage = combinedIndex;

            for (Iterator<T> iterator = iterator(); iterator.hasNext();)
            {
                if (currentPage.size() < PRELOAD_LIMIT)
                {
                    currentPage.add(iterator.next());
                    combinedIndex++;
                }
                else
                    break;
            }

            // could not fill up to preload limit. Must have exhausted pagers.
            if (currentPage.size() < PRELOAD_LIMIT || !iterator().hasNext())
                onLastPage = true;
        }

        return currentPage;
    }

    public void nextPage()
    {
        try
        {
            // if current page has been loaded, then we have already increased the value of the pointer by PRELOAD_LIMIT
            skipTo(combinedIndex + (currentPage == null ? PRELOAD_LIMIT : 0));
        }
        catch (PagerException e)
        {
            log.error("Erroring calling nextPage()", e);
        }
    }

    public boolean onLastPage()
    {
        return onLastPage;
    }

    /**
     * Will run the index up to this point. Calling {@link Pager#getCurrentPage()} will
     * then return a page holding this index.
     *
     * @param index the zero-based index in the result set to skip to
     * @throws PagerException - if the number of items in the backing data is exceeded by the index.
     */
    public void skipTo(int index) throws PagerException
    {
        if (index < combinedIndex)
            throw new PagerException("Cannot run the index back to [" + index + "] from [" + combinedIndex + "]");

        while (combinedIndex < index)
        {
            iterator().next();
            combinedIndex++;
        }

        // reset currentPage and put in next batch of results loaded from position 'index')
        currentPage = null;
        getCurrentPage();
    }

    /**
     * @return the current index position of the pager
     */
    public int getIndex()
    {
        return combinedIndex;
    }

    public int getIndexOfFirstItemInCurrentPage()
    {
        getCurrentPage(); // this call is necessary to initialise this value
        return indexOfFirstItemInCurrentPage;
    }

    private class MergedListIterator implements Iterator<T>
    {
        private List<Iterator<T>> iterators = new ArrayList<Iterator<T>>();

        public MergedListIterator(List<? extends Pager<T>> listOfPagers)
        {
            for (Pager<T> listOfPager : listOfPagers)
            {
                this.iterators.add(listOfPager.iterator());
            }
        }

        private Iterator<T> getCurrentIterator()
        {
            for (Iterator<T> iterator : iterators)
            {
                if (iterator.hasNext())
                    return iterator;
            }

            return Collections.<T>emptyList().iterator();
        }


        public void remove()
        {
            throw new UnsupportedOperationException("This iterator does not support removal");
        }

        public boolean hasNext()
        {
            return getCurrentIterator().hasNext();
        }

        public T next()
        {
            T nextElement = getCurrentIterator().next();

            if (!hasNext())
                onLastPage = true;

            return nextElement;
        }
    }

}
