package com.atlassian.user.impl.cache.properties;

import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.properties.PropertySetFactory;
import com.atlassian.cache.CacheFactory;
import com.atlassian.cache.Cache;
import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.module.propertyset.PropertyException;
import com.opensymphony.module.propertyset.PropertySetSchema;

import java.util.HashMap;
import java.util.Map;
import java.util.Date;
import java.util.Collection;
import java.util.Properties;

import org.w3c.dom.Document;

/**
 * A PropertySetFactory that wraps another PropertySetFactory and maintains a cache of its {@link PropertySet}s.  This
 * prevents the need for PropertySets to be looked up repeatedly from the wrapped PropertySetFactory.
 * Retrieved PropertySets are also wrapped in a CachedPropertySet so that individual property values are cached
 * instead of repeatedly retrieved.  Both levels of caching rewrite to the cache whenever a setter is
 * called on a {@link PropertySet}.
 *
 * <p>The cache is keyed purely on entity name, so it assumes all entities
 * used have unique names, even across entity types.<p>
 *
 * <p>Since the decorated PropertySets retrieved from the <code>underlyingPropertySetFactory</code> are cached
 * they must be {@link java.io.Serializable}.</p>
 */
public class CachingPropertySetFactory implements PropertySetFactory
{
    private final PropertySetFactory underlyingPropertySetFactory;
    private final CacheFactory cacheFactory;

    public CachingPropertySetFactory(PropertySetFactory underlyingPropertySetFactory, CacheFactory cacheFactory)
    {
        this.underlyingPropertySetFactory = underlyingPropertySetFactory;
        this.cacheFactory = cacheFactory;
    }

    public PropertySet getPropertySet(Entity entity) throws EntityException
    {
        Cache cache = getCache();
        CachedPropertySet cachedPropertySet = (CachedPropertySet) cache.get(entity.getName());
        if (cachedPropertySet == null)
        {
            PropertySet underlyingPropertySet = underlyingPropertySetFactory.getPropertySet(entity);
            HashMap<String, PropertySet> args = new HashMap<String, PropertySet>();
            args.put("PropertySet", underlyingPropertySet);
            cachedPropertySet = new CachedPropertySet();
            cachedPropertySet.init(args, args);

            cache.put(entity.getName(), cachedPropertySet);
        }
        return new RecacheOnWritePropertySet(cachedPropertySet, entity.getName());
    }

    private Cache getCache()
    {
        String cacheName = underlyingPropertySetFactory.getClass().getName() + ".propertysets";
        return cacheFactory.getCache(cacheName);
    }

    /**
     * PropertySet Wrapper that propagates any changes in the wrapped PropertySet into the cache.
     * This is necessary because some caches do not update correctly when a cached object is modified.
     */
    private class RecacheOnWritePropertySet implements PropertySet
    {
        // Every {@link PropertySet} setter in this object must recache the wrapped delegate
        // into the cache.

        private final PropertySet delegate;
        private String cacheKey;

        public RecacheOnWritePropertySet(PropertySet delegate, String cacheKey)
        {
            this.delegate = delegate;
            this.cacheKey = cacheKey;
        }

        public void init(Map config, Map args)
        {
            delegate.init(config, args);
            recache();
        }

        public void setAsActualType(String key, Object value) throws PropertyException
        {
            delegate.setAsActualType(key, value);
            recache();
        }

        public Object getAsActualType(String key) throws PropertyException
        {
            return delegate.getAsActualType(key);
        }

        public void setBoolean(String key, boolean value) throws PropertyException
        {
            delegate.setBoolean(key, value);
            recache();
        }

        public boolean getBoolean(String key) throws PropertyException
        {
            return delegate.getBoolean(key);
        }

        public void setData(String key, byte[] value) throws PropertyException
        {
            delegate.setData(key, value);
            recache();
        }

        public byte[] getData(String key) throws PropertyException
        {
            return delegate.getData(key);
        }

        public void setDate(String key, Date value) throws PropertyException
        {
            delegate.setDate(key, value);
            recache();
        }

        public Date getDate(String key) throws PropertyException
        {
            return delegate.getDate(key);
        }

        public void setDouble(String key, double value) throws PropertyException
        {
            delegate.setDouble(key, value);
            recache();
        }

        public double getDouble(String key) throws PropertyException
        {
            return delegate.getDouble(key);
        }

        public void setInt(String key, int value) throws PropertyException
        {
            delegate.setInt(key, value);
            recache();
        }

        public int getInt(String key) throws PropertyException
        {
            return delegate.getInt(key);
        }

        public Collection getKeys() throws PropertyException
        {
            return delegate.getKeys();
        }

        public Collection getKeys(int type) throws PropertyException
        {
            return delegate.getKeys(type);
        }

        public Collection getKeys(String prefix) throws PropertyException
        {
            return delegate.getKeys(prefix);
        }

        public Collection getKeys(String prefix, int type) throws PropertyException
        {
            return delegate.getKeys(prefix, type);
        }

        public void setLong(String key, long value) throws PropertyException
        {
            delegate.setLong(key, value);
            recache();
        }

        public long getLong(String key) throws PropertyException
        {
            return delegate.getLong(key);
        }

        public void setObject(String key, Object value) throws PropertyException
        {
            delegate.setObject(key, value);
            recache();
        }

        public Object getObject(String key) throws PropertyException
        {
            return delegate.getObject(key);
        }

        public void setProperties(String key, Properties value) throws PropertyException
        {
            delegate.setProperties(key, value);
            recache();
        }

        public Properties getProperties(String key) throws PropertyException
        {
            return delegate.getProperties(key);
        }

        public void setSchema(PropertySetSchema schema) throws PropertyException
        {
            delegate.setSchema(schema);
            recache();
        }

        public PropertySetSchema getSchema() throws PropertyException
        {
            return delegate.getSchema();
        }

        public boolean isSettable(String property)
        {
            return delegate.isSettable(property);
        }

        public void setString(String key, String value) throws PropertyException
        {
            delegate.setString(key, value);
            recache();
        }

        public String getString(String key) throws PropertyException
        {
            return delegate.getString(key);
        }

        public void setText(String key, String value) throws PropertyException
        {
            delegate.setText(key, value);
            recache();
        }

        public String getText(String key) throws PropertyException
        {
            return delegate.getText(key);
        }

        public int getType(String key) throws PropertyException
        {
            return delegate.getType(key);
        }

        public void setXML(String key, Document value) throws PropertyException
        {
            delegate.setXML(key, value);
            recache();
        }

        public Document getXML(String key) throws PropertyException
        {
            return delegate.getXML(key);
        }

        public boolean exists(String key) throws PropertyException
        {
            return delegate.exists(key);
        }

        public void remove(String key) throws PropertyException
        {
            delegate.remove(key);
            recache();
        }

        public boolean supportsType(int type)
        {
            return delegate.supportsType(type);
        }

        public boolean supportsTypes()
        {
            return delegate.supportsTypes();
        }

        private void recache()
        {
            getCache().put(cacheKey, delegate);
        }
    }
}

