package com.atlassian.user.impl.hibernate.search.query;

import com.atlassian.user.EntityException;
import com.atlassian.user.Group;
import com.atlassian.user.User;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.hibernate.DefaultHibernateGroup;
import com.atlassian.user.impl.hibernate.DefaultHibernateUser;
import com.atlassian.user.impl.hibernate.repository.HibernateRepository;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.DefaultSearchResult;
import com.atlassian.user.search.SearchResult;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.query.*;
import net.sf.hibernate.Criteria;
import net.sf.hibernate.HibernateException;
import net.sf.hibernate.Session;
import net.sf.hibernate.expression.*;
import org.springframework.orm.hibernate.SessionFactoryUtils;

import java.util.*;

/**
 * Handles {@link Query} objects on a {@link HibernateRepository}.
 */
public final class HibernateEntityQueryParser implements EntityQueryParser
{
    private final RepositoryIdentifier identifier;
    private final HibernateRepository repository;
    private final QueryValidator queryValidator = new QueryValidator();

    public HibernateEntityQueryParser(RepositoryIdentifier identifier, HibernateRepository repository)
    {
        this.identifier = identifier;
        this.repository = repository;
    }

    private Session getSession()
    {
        return SessionFactoryUtils.getSession(repository.getSessionFactory(), true);
    }

    public SearchResult<User> findUsers(Query<User> query) throws EntityException
    {
        queryValidator.assertValid(query);
        Criteria criteria = getSession().createCriteria(DefaultHibernateUser.class);
        criteria = identifyAndAddSearchCriteria(query, criteria);
        criteria.addOrder(Order.asc("name")); // sort results explicitly by username rather than leaving this up to the mood of the database
        List<User> results;
        try
        {
            //noinspection unchecked
            results = criteria.list();
        }
        catch (HibernateException e)
        {
            throw new RepositoryException(e);
        }
        return new DefaultSearchResult<User>(new DefaultPager<User>(results), identifier.getKey());
    }

    public SearchResult<Group> findGroups(Query<Group> query) throws EntityException
    {
        queryValidator.assertValid(query);
        Criteria criteria = getSession().createCriteria(DefaultHibernateGroup.class);
        criteria = identifyAndAddSearchCriteria(query, criteria);
        criteria.addOrder(Order.asc("name")); // sort results explicitly by username rather than leaving this up to the mood of the database
        List<Group> results;
        try
        {
            //noinspection unchecked
            results = criteria.list();
        }
        catch (HibernateException e)
        {
            throw new RepositoryException(e);
        }
        return new DefaultSearchResult<Group>(new DefaultPager<Group>(results), identifier.getKey());
    }

    public SearchResult<User> findUsers(Query<User> query, QueryContext context) throws EntityException
    {
        if (!context.contains(identifier))
            return null;

        return findUsers(query);
    }

    public SearchResult<Group> findGroups(Query<Group> query, QueryContext context) throws EntityException
    {
        if (!context.contains(identifier))
            return null;

        return findGroups(query);
    }

    private static MatchMode getMatchMode(String matchingRule)
    {
        if (matchingRule.equals(TermQuery.SUBSTRING_CONTAINS))
        {
            return MatchMode.ANYWHERE;
        }
        if (matchingRule.equals(TermQuery.SUBSTRING_ENDS_WITH))
        {
            return MatchMode.END;
        }
        if (matchingRule.equals(TermQuery.SUBSTRING_STARTS_WITH))
        {
            return MatchMode.START;
        }
        return MatchMode.EXACT;
    }

    private static String identifyProperty(TermQuery q)
    {
        if (q instanceof UserNameTermQuery)
            return "name";
        else if (q instanceof EmailTermQuery)
            return "email";
        else if (q instanceof FullNameTermQuery)
            return "fullName";
        else if (q instanceof GroupNameTermQuery)
            return "name";
        return null;
    }

    private static Criteria identifyAndAddSearchCriteria(Query q, Criteria baseCriteria)
        throws EntityQueryException
    {
        if (q instanceof BooleanQuery)
            return addSearchCriteria((BooleanQuery) q, baseCriteria);
        else
        {
            return addSearchCriteria((TermQuery) q, baseCriteria);
        }
    }

    private static Criteria addSearchCriteria(BooleanQuery<?> booleanQuery, Criteria baseCriteria)
        throws EntityQueryException
    {
        Junction junction = booleanQuery.isAND() ? Expression.conjunction() : Expression.disjunction();
        baseCriteria.add(junction);

        for (Query nestedQuery : booleanQuery.getQueries())
        {
            if (nestedQuery instanceof BooleanQuery)
            {
                addSearchCriteria((BooleanQuery) nestedQuery, baseCriteria);
            }
            else if (nestedQuery instanceof TermQuery)
            {
                junction.add(getQueryExpression((TermQuery) nestedQuery));
            }
            else
                throw new EntityQueryException("Unknown query type: [" + nestedQuery.getClass().getName() + "]");
        }

        return baseCriteria;
    }

    private static Criteria addSearchCriteria(TermQuery q, Criteria baseCriteria)
    {
        Criterion expression = getQueryExpression(q);
        baseCriteria.add(expression);
        return baseCriteria;
    }

    private static Criterion getQueryExpression(TermQuery termQuery)
    {
        String hqlField = identifyProperty(termQuery);

        if (termQuery.isMatchingSubstring())
        {
            MatchMode matchMode = getMatchMode(termQuery.getMatchingRule());
            return Expression.ilike(hqlField, termQuery.getTerm(), matchMode);
        }
        else
        {
            return new EqExpression(hqlField, termQuery.getTerm(), true);
        }
    }
}
