package com.atlassian.user.impl.ldap.properties;

import javax.naming.directory.SearchControls;

/**
 * Properties which map an LDAP data structure to users and groups for searching an LDAP repository.
 */
public interface LdapSearchProperties
{
    /**
     * @return base DN of user tree in LDAP server, for example 'ou=users,dc=atlassian,dc=com'
     */
    String getBaseUserNamespace();

    /**
     * @return LDAP search filter for users, for example '(objectClass=inetOrgPerson)'
     */
    String getUserFilter();

    /**
     * Returns <code>true</code> if user searches will scan the entire LDAP tree beneath the DN specified by
     * {@link #getBaseUserNamespace()} (sub-tree scope), or <code>false</code> if only direct children of that
     * DN should be scanned (one-level scope).
     *
     * @see SearchControls#SUBTREE_SCOPE
     * @see SearchControls#ONELEVEL_SCOPE
     */
    boolean isUserSearchScopeAllDepths();

    /**
     * @return attribute name of the user identifier attribute on an LDAP user record, for example 'sAMAccountName'
     */
    String getUsernameAttribute();

    /**
     * @return attribute name of the first name attribute on an LDAP user record, for example 'givenName'
     */
    String getFirstnameAttribute();

    /**
     * @return attribute name of the surname attribute on an LDAP user record, for example 'sn'
     */
    String getSurnameAttribute();

    /**
     * @return attribute name of the email attribute on an LDAP user record, for example 'mail'
     */
    String getEmailAttribute();

    /**
     * @return base DN of group tree in LDAP server, for example 'ou=groups,dc=atlassian,dc=com'
     */
    String getBaseGroupNamespace();

    /**
     * @return LDAP search filter for group, for example '(objectClass=groupOfNames)'
     */
    String getGroupFilter();

    /**
     * Returns <code>true</code> if group searches will scan the entire LDAP tree beneath the DN specified by
     * {@link #getBaseGroupNamespace()} (sub-tree scope), or <code>false</code> if only direct children of that
     * DN should be scanned (one-level scope).
     *
     * @see SearchControls#SUBTREE_SCOPE
     * @see SearchControls#ONELEVEL_SCOPE
     */
    boolean isGroupSearchScopeAllDepths();

    /**
     * @return attribute name of the group identifier attribute on an LDAP group record, for example 'cn'
     */
    String getGroupnameAttribute();

    /**
     * @return maximum time LDAP connection will wait for search results to return before aborting the search.
     * Measured in milliseconds, where 0 means wait indefinitely.
     * @see SearchControls#getTimeLimit()
     */
    int getTimeLimitMillis();

}
