package com.atlassian.user.impl.osuser.security.authentication;

import com.atlassian.user.EntityException;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.security.authentication.Authenticator;
import com.atlassian.user.security.authentication.EntityAuthenticationException;
import com.opensymphony.user.authenticator.AuthenticationException;
import com.opensymphony.user.provider.CredentialsProvider;

/**
 * OSUser has two classes which handle authentication - the {@link CredentialsProvider} and
 * the {@link com.opensymphony.user.authenticator.Authenticator}. This Authenticator impl.
 * has the capability to wrap one of either.
 */
public class OSUAuthenticator implements Authenticator
{
    private final RepositoryIdentifier repository;
    private final CredentialsProvider credentialsProvider;
    private final com.opensymphony.user.authenticator.Authenticator osuserAuthenticator;

    public OSUAuthenticator(RepositoryIdentifier repository, CredentialsProvider credentialsProvider)
    {
        this(repository, credentialsProvider, null);
    }

    public OSUAuthenticator(RepositoryIdentifier repository, com.opensymphony.user.authenticator.Authenticator osuserAuthenticator)
    {
        this(repository, null, osuserAuthenticator);
    }

    private OSUAuthenticator(RepositoryIdentifier repository, CredentialsProvider credentialsProvider,
        com.opensymphony.user.authenticator.Authenticator osuserAuthenticator)
    {
        this.repository = repository;
        this.credentialsProvider = credentialsProvider;
        this.osuserAuthenticator = osuserAuthenticator;
    }

    public boolean authenticate(String username, String password) throws EntityException
    {
        if (isWrappingCredentialProvider())
        {
            return credentialsProvider.authenticate(username, password);
        }

        try
        {
            return osuserAuthenticator.login(username, password);
        }
        catch (AuthenticationException e)
        {
            throw new EntityAuthenticationException(e);
        }
    }

    public RepositoryIdentifier getRepository()
    {
        return repository;
    }

    /**
     * @return true if an instance of {@link CredentialsProvider} has been wrapped, false otherwise.
     */
    public boolean isWrappingCredentialProvider()
    {
        return credentialsProvider != null;
    }
}
