// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.security.keyvault.keys;

import com.azure.core.annotation.ReturnType;
import com.azure.core.annotation.ServiceMethod;
import com.azure.core.exception.HttpResponseException;
import com.azure.core.exception.ResourceModifiedException;
import com.azure.core.exception.ResourceNotFoundException;
import com.azure.core.http.rest.PagedIterable;
import com.azure.core.http.rest.Response;
import com.azure.core.annotation.ServiceClient;
import com.azure.core.util.Context;
import com.azure.core.util.polling.SyncPoller;
import com.azure.security.keyvault.keys.models.DeletedKey;
import com.azure.security.keyvault.keys.models.CreateEcKeyOptions;
import com.azure.security.keyvault.keys.models.KeyVaultKey;
import com.azure.security.keyvault.keys.models.KeyProperties;
import com.azure.security.keyvault.keys.models.CreateKeyOptions;
import com.azure.security.keyvault.keys.models.ImportKeyOptions;
import com.azure.security.keyvault.keys.models.CreateRsaKeyOptions;
import com.azure.security.keyvault.keys.models.JsonWebKey;
import com.azure.security.keyvault.keys.models.KeyCurveName;
import com.azure.security.keyvault.keys.models.KeyOperation;
import com.azure.security.keyvault.keys.models.KeyType;

/**
 * The KeyClient provides synchronous methods to manage {@link KeyVaultKey keys} in the Azure Key Vault. The client supports
 * creating, retrieving, updating, deleting, purging, backing up, restoring and listing the {@link KeyVaultKey keys}. The client
 * also supports listing {@link DeletedKey deleted keys} for a soft-delete enabled Azure Key Vault.
 *
 * <p><strong>Samples to construct the sync client</strong></p>
 * <pre>
 * KeyClient keyClient = new KeyClientBuilder&#40;&#41;
 *     .vaultUrl&#40;&quot;https:&#47;&#47;myvault.azure.net&#47;&quot;&#41;
 *     .credential&#40;new DefaultAzureCredentialBuilder&#40;&#41;.build&#40;&#41;&#41;
 *     .buildClient&#40;&#41;;
 * </pre>
 *
 * @see KeyClientBuilder
 * @see PagedIterable
 */
@ServiceClient(builder = KeyClientBuilder.class, serviceInterfaces = KeyService.class)
public final class KeyClient {
    private final KeyAsyncClient client;

    /**
     * Creates a KeyClient that uses {@code pipeline} to service requests
     *
     * @param client The {@link KeyAsyncClient} that the client routes its request through.
     */
    KeyClient(KeyAsyncClient client) {
        this.client = client;
    }

    /**
     * Get the vault endpoint url
     * @return the vault endpoint url
     */
    public String getVaultUrl() {
        return client.getVaultUrl();
    }

    /**
     * Creates a new key and stores it in the key vault. The create key operation can be used to create any key type in
     * key vault. If the named key already exists, Azure Key Vault creates a new version of the key. It requires the
     * {@code keys/create} permission.
     *
     * <p>The {@link KeyType keyType} indicates the type of key to create. Possible values include: {@link KeyType#EC
     * EC}, {@link KeyType#EC_HSM EC-HSM}, {@link KeyType#RSA RSA}, {@link KeyType#RSA_HSM RSA-HSM} and {@link
     * KeyType#OCT OCT}.</p>
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Creates a new EC key. Prints out the details of the created key.</p>
     * <pre>
     * KeyVaultKey key = keyClient.createKey&#40;&quot;keyName&quot;, KeyType.EC&#41;;
     * System.out.printf&#40;&quot;Key is created with name %s and id %s %n&quot;, key.getName&#40;&#41;, key.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param name The name of the key being created.
     * @param keyType The type of key to create. For valid values, see {@link KeyType KeyType}.
     * @return The {@link KeyVaultKey created key}.
     * @throws ResourceModifiedException if {@code name} or {@code keyType} is null.
     * @throws HttpResponseException if {@code name} is empty string.
     */
    public KeyVaultKey createKey(String name, KeyType keyType) {
        return createKeyWithResponse(new CreateKeyOptions(name, keyType), Context.NONE).getValue();
    }

    /**
     * Creates a new key and stores it in the key vault. The create key operation can be used to create any key type in
     * key vault. If the named key already exists, Azure Key Vault creates a new version of the key. It requires the
     * {@code keys/create} permission.
     *
     * <p>The {@link CreateKeyOptions} is required. The {@link CreateKeyOptions#getExpiresOn() expires} and {@link
     * CreateKeyOptions#getNotBefore() notBefore} values are optional. The {@link CreateKeyOptions#isEnabled()} enabled} field
     * is set to true by Azure Key Vault, if not specified.</p>
     *
     * <p>The {@link CreateKeyOptions#getKeyType() keyType} indicates the type of key to create. Possible values include:
     * {@link KeyType#EC EC}, {@link KeyType#EC_HSM EC-HSM}, {@link KeyType#RSA RSA}, {@link KeyType#RSA_HSM RSA-HSM}
     * and {@link KeyType#OCT OCT}.</p>
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Creates a new RSA key which activates in one day and expires in one year. Prints out the details of the
     * <pre>
     * CreateKeyOptions createKeyOptions = new CreateKeyOptions&#40;&quot;keyName&quot;, KeyType.RSA&#41;
     *     .setNotBefore&#40;OffsetDateTime.now&#40;&#41;.plusDays&#40;1&#41;&#41;
     *     .setExpiresOn&#40;OffsetDateTime.now&#40;&#41;.plusYears&#40;1&#41;&#41;;
     * KeyVaultKey optionsKey = keyClient.createKey&#40;createKeyOptions&#41;;
     * System.out.printf&#40;&quot;Key is created with name %s and id %s %n&quot;, optionsKey.getName&#40;&#41;, optionsKey.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param createKeyOptions The key options object containing information about the key being created.
     * @return The {@link KeyVaultKey created key}.
     * @throws NullPointerException if {@code keyCreateOptions} is {@code null}.
     * @throws ResourceModifiedException if {@code keyCreateOptions} is malformed.
     */
    public KeyVaultKey createKey(CreateKeyOptions createKeyOptions) {
        return createKeyWithResponse(createKeyOptions, Context.NONE).getValue();
    }

    /**
     * Creates a new key and stores it in the key vault. The create key operation can be used to create any key type in
     * key vault. If the named key already exists, Azure Key Vault creates a new version of the key. It requires the
     * {@code keys/create} permission.
     *
     * <p>The {@link CreateKeyOptions} is required. The {@link CreateKeyOptions#getExpiresOn() expires} and {@link
     * CreateKeyOptions#getNotBefore() notBefore} values are optional. The {@link CreateKeyOptions#isEnabled() enabled} field
     * is set to true by Azure Key Vault, if not specified.</p>
     *
     * <p>The {@link CreateKeyOptions#getKeyType() keyType} indicates the type of key to create. Possible values include:
     * {@link KeyType#EC EC}, {@link KeyType#EC_HSM EC-HSM}, {@link KeyType#RSA RSA}, {@link KeyType#RSA_HSM RSA-HSM}
     * and {@link KeyType#OCT OCT}.</p>
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Creates a new RSA key which activates in one day and expires in one year. Prints out the details of the
     * <pre>
     * CreateKeyOptions createKeyOptions = new CreateKeyOptions&#40;&quot;keyName&quot;, KeyType.RSA&#41;
     *     .setNotBefore&#40;OffsetDateTime.now&#40;&#41;.plusDays&#40;1&#41;&#41;
     *     .setExpiresOn&#40;OffsetDateTime.now&#40;&#41;.plusYears&#40;1&#41;&#41;;
     * KeyVaultKey optionsKey = keyClient.createKeyWithResponse&#40;createKeyOptions, new Context&#40;key1, value1&#41;&#41;.getValue&#40;&#41;;
     * System.out.printf&#40;&quot;Key is created with name %s and id %s %n&quot;, optionsKey.getName&#40;&#41;, optionsKey.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param createKeyOptions The key options object containing information about the key being created.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link KeyVaultKey created key}.
     * @throws NullPointerException if {@code keyCreateOptions} is {@code null}.
     * @throws ResourceModifiedException if {@code keyCreateOptions} is malformed.
     */
    public Response<KeyVaultKey> createKeyWithResponse(CreateKeyOptions createKeyOptions, Context context) {
        return client.createKeyWithResponse(createKeyOptions, context).block();
    }

    /**
     * Creates a new Rsa key and stores it in the key vault. The create Rsa key operation can be used to create any Rsa
     * key type in key vault. If the named key already exists, Azure Key Vault creates a new version of the key. It
     * requires the {@code keys/create} permission.
     *
     * <p>The {@link CreateRsaKeyOptions} is required. The {@link CreateRsaKeyOptions#getKeySize() keySize} can be
     * optionally specified. The {@link CreateRsaKeyOptions#getExpiresOn() expires} and {@link
     * CreateRsaKeyOptions#getNotBefore() notBefore} values are optional. The {@link CreateRsaKeyOptions#isEnabled() enabled}
     * field is set to true by Azure Key Vault, if not specified.</p>
     *
     * <p>The {@link CreateRsaKeyOptions#getKeyType() keyType} indicates the type of key to create. Possible values
     * include: {@link KeyType#RSA RSA} and {@link KeyType#RSA_HSM RSA-HSM}.</p>
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Creates a new RSA key with size 2048 which activates in one day and expires in one year. Prints out the
     * details of the created key.</p>
     * <pre>
     * CreateRsaKeyOptions createRsaKeyOptions = new CreateRsaKeyOptions&#40;&quot;keyName&quot;&#41;
     *     .setKeySize&#40;2048&#41;
     *     .setNotBefore&#40;OffsetDateTime.now&#40;&#41;.plusDays&#40;1&#41;&#41;
     *     .setExpiresOn&#40;OffsetDateTime.now&#40;&#41;.plusYears&#40;1&#41;&#41;;
     * KeyVaultKey rsaKey = keyClient.createRsaKey&#40;createRsaKeyOptions&#41;;
     * System.out.printf&#40;&quot;Key is created with name %s and id %s %n&quot;, rsaKey.getName&#40;&#41;, rsaKey.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param createRsaKeyOptions The key options object containing information about the rsa key being created.
     * @return The {@link KeyVaultKey created key}.
     * @throws NullPointerException if {@code rsaKeyCreateOptions} is {@code null}.
     * @throws ResourceModifiedException if {@code rsaKeyCreateOptions} is malformed.
     * @throws HttpResponseException if {@code name} is empty string.
     */
    public KeyVaultKey createRsaKey(CreateRsaKeyOptions createRsaKeyOptions) {
        return createRsaKeyWithResponse(createRsaKeyOptions, Context.NONE).getValue();
    }

    /**
     * Creates a new Rsa key and stores it in the key vault. The create Rsa key operation can be used to create any Rsa
     * key type in key vault. If the named key already exists, Azure Key Vault creates a new version of the key. It
     * requires the {@code keys/create} permission.
     *
     * <p>The {@link CreateRsaKeyOptions} is required. The {@link CreateRsaKeyOptions#getKeySize() keySize} can be
     * optionally specified. The {@link CreateRsaKeyOptions#getExpiresOn() expires} and {@link
     * CreateRsaKeyOptions#getNotBefore() notBefore} values are optional. The {@link CreateRsaKeyOptions#isEnabled() enabled}
     * field is set to true by Azure Key Vault, if not specified.</p>
     *
     * <p>The {@link CreateRsaKeyOptions#getKeyType() keyType} indicates the type of key to create. Possible values
     * include: {@link KeyType#RSA RSA} and {@link KeyType#RSA_HSM RSA-HSM}.</p>
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Creates a new RSA key with size 2048 which activates in one day and expires in one year. Prints out the
     * details of the created key.</p>
     * <pre>
     * CreateRsaKeyOptions createRsaKeyOptions = new CreateRsaKeyOptions&#40;&quot;keyName&quot;&#41;
     *     .setKeySize&#40;2048&#41;
     *     .setNotBefore&#40;OffsetDateTime.now&#40;&#41;.plusDays&#40;1&#41;&#41;
     *     .setExpiresOn&#40;OffsetDateTime.now&#40;&#41;.plusYears&#40;1&#41;&#41;;
     * KeyVaultKey rsaKey = keyClient.createRsaKeyWithResponse&#40;createRsaKeyOptions, new Context&#40;key1, value1&#41;&#41;.getValue&#40;&#41;;
     * System.out.printf&#40;&quot;Key is created with name %s and id %s %n&quot;, rsaKey.getName&#40;&#41;, rsaKey.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param createRsaKeyOptions The key options object containing information about the rsa key being created.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link KeyVaultKey created key}.
     * @throws NullPointerException if {@code rsaKeyCreateOptions} is {@code null}.
     * @throws ResourceModifiedException if {@code rsaKeyCreateOptions} is malformed.
     */
    public Response<KeyVaultKey> createRsaKeyWithResponse(CreateRsaKeyOptions createRsaKeyOptions, Context context) {
        return client.createRsaKeyWithResponse(createRsaKeyOptions, context).block();
    }

    /**
     * Creates a new Ec key and  stores it in the key vault. The create Ec key operation can be used to create any Ec
     * key type in key vault. If the named key already exists, Azure Key Vault creates a new version of the key. It
     * requires the {@code keys/create} permission.
     *
     * <p>The {@link CreateEcKeyOptions} parameter is required. The {@link CreateEcKeyOptions#getCurveName() key curve} can be
     * optionally specified. If not specified, default value of {@link KeyCurveName#P_256 P-256} is used by Azure Key
     * Vault. The {@link CreateEcKeyOptions#getExpiresOn() expires} and {@link CreateEcKeyOptions#getNotBefore() notBefore}
     * values are optional. The {@link CreateEcKeyOptions#isEnabled() enabled} field is set to true by Azure Key Vault, if
     * not specified.</p>
     *
     * <p>The {@link CreateEcKeyOptions#getKeyType() keyType} indicates the type of key to create. Possible values
     * include: {@link KeyType#EC EC} and {@link KeyType#EC_HSM EC-HSM}.</p>
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Creates a new EC key with P-384 web key curve. The key activates in one day and expires in one year. Prints
     * out the details of the created key.</p>
     * <pre>
     * CreateEcKeyOptions createEcKeyOptions = new CreateEcKeyOptions&#40;&quot;keyName&quot;&#41;
     *     .setCurveName&#40;KeyCurveName.P_384&#41;
     *     .setNotBefore&#40;OffsetDateTime.now&#40;&#41;.plusDays&#40;1&#41;&#41;
     *     .setExpiresOn&#40;OffsetDateTime.now&#40;&#41;.plusYears&#40;1&#41;&#41;;
     * KeyVaultKey ecKey = keyClient.createEcKey&#40;createEcKeyOptions&#41;;
     * System.out.printf&#40;&quot;Key is created with name %s and id %s %n&quot;, ecKey.getName&#40;&#41;, ecKey.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param createEcKeyOptions The key options object containing information about the ec key being created.
     * @return The {@link KeyVaultKey created key}.
     * @throws NullPointerException if {@code ecKeyCreateOptions} is {@code null}.
     * @throws ResourceModifiedException if {@code ecKeyCreateOptions} is malformed.
     */
    public KeyVaultKey createEcKey(CreateEcKeyOptions createEcKeyOptions) {
        return createEcKeyWithResponse(createEcKeyOptions, Context.NONE).getValue();
    }

    /**
     * Creates a new Ec key and  stores it in the key vault. The create Ec key operation can be used to create any Ec
     * key type in key vault. If the named key already exists, Azure Key Vault creates a new version of the key. It
     * requires the {@code keys/create} permission.
     *
     * <p>The {@link CreateEcKeyOptions} parameter is required. The {@link CreateEcKeyOptions#getCurveName() key curve} can be
     * optionally specified. If not specified, default value of {@link KeyCurveName#P_256 P-256} is used by Azure Key
     * Vault. The {@link CreateEcKeyOptions#getExpiresOn() expires} and {@link CreateEcKeyOptions#getNotBefore() notBefore}
     * values are optional. The {@link CreateEcKeyOptions#isEnabled()} enabled} field is set to true by Azure Key Vault, if
     * not specified.</p>
     *
     * <p>The {@link CreateEcKeyOptions#getKeyType() keyType} indicates the type of key to create. Possible values
     * include:
     * {@link KeyType#EC EC} and {@link KeyType#EC_HSM EC-HSM}.</p>
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Creates a new EC key with P-384 web key curve. The key activates in one day and expires in one year. Prints
     * out the details of the created key.</p>
     * <pre>
     * CreateEcKeyOptions createEcKeyOptions = new CreateEcKeyOptions&#40;&quot;keyName&quot;&#41;
     *     .setCurveName&#40;KeyCurveName.P_384&#41;
     *     .setNotBefore&#40;OffsetDateTime.now&#40;&#41;.plusDays&#40;1&#41;&#41;
     *     .setExpiresOn&#40;OffsetDateTime.now&#40;&#41;.plusYears&#40;1&#41;&#41;;
     * KeyVaultKey ecKey = keyClient.createEcKeyWithResponse&#40;createEcKeyOptions, new Context&#40;key1, value1&#41;&#41;.getValue&#40;&#41;;
     * System.out.printf&#40;&quot;Key is created with name %s and id %s %n&quot;, ecKey.getName&#40;&#41;, ecKey.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param createEcKeyOptions The key options object containing information about the ec key being created.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link KeyVaultKey created key}.
     * @throws NullPointerException if {@code ecKeyCreateOptions} is {@code null}.
     * @throws ResourceModifiedException if {@code ecKeyCreateOptions} is malformed.
     */
    public Response<KeyVaultKey> createEcKeyWithResponse(CreateEcKeyOptions createEcKeyOptions, Context context) {
        return client.createEcKeyWithResponse(createEcKeyOptions, context).block();
    }

    /**
     * Imports an externally created key and stores it in key vault. The import key operation may be used to import any
     * key type into the Azure Key Vault. If the named key already exists, Azure Key Vault creates a new version of the
     * key. This operation requires the {@code keys/import} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Imports a new key into key vault. Prints out the details of the imported key.</p>
     *
     * <pre>
     * KeyVaultKey importedKey = keyClient.importKey&#40;&quot;keyName&quot;, jsonWebKeyToImport&#41;;
     * System.out.printf&#40;&quot;Key is imported with name %s and id %s &#92;n&quot;, importedKey.getName&#40;&#41;, importedKey.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param name The name for the imported key.
     * @param keyMaterial The Json web key being imported.
     * @return The {@link KeyVaultKey imported key}.
     * @throws HttpResponseException if {@code name} is empty string.
     */
    public KeyVaultKey importKey(String name, JsonWebKey keyMaterial) {
        return importKeyWithResponse(new ImportKeyOptions(name, keyMaterial), Context.NONE).getValue();
    }

    /**
     * Imports an externally created key and stores it in key vault. The import key operation may be used to import any
     * key type into the Azure Key Vault. If the named key already exists, Azure Key Vault creates a new version of the
     * key. This operation requires the {@code keys/import} permission.
     *
     * <p>The {@code keyImportOptions} is required and its fields {@link ImportKeyOptions#getName() name} and {@link
     * ImportKeyOptions#getKey() key material} cannot be null. The {@link ImportKeyOptions#getExpiresOn() expires} and
     * {@link ImportKeyOptions#getNotBefore() notBefore} values in {@code keyImportOptions} are optional. If not specified,
     * no values are set for the fields. The {@link ImportKeyOptions#isEnabled() enabled} field is set to true and the
     * {@link ImportKeyOptions#isHardwareProtected() hsm} field is set to false by Azure Key Vault, if they are not specified.</p>
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Imports a new key into key vault. Prints out the details of the imported key.</p>
     *
     * <pre>
     * ImportKeyOptions options = new ImportKeyOptions&#40;&quot;keyName&quot;, jsonWebKeyToImport&#41;
     *     .setHardwareProtected&#40;false&#41;;
     * 
     * KeyVaultKey importedKeyResponse = keyClient.importKey&#40;options&#41;;
     * System.out.printf&#40;&quot;Key is imported with name %s and id %s &#92;n&quot;, importedKeyResponse.getName&#40;&#41;,
     *     importedKeyResponse.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param importKeyOptions The key import configuration object containing information about the json web key
     *     being imported.
     * @return The {@link KeyVaultKey imported key}.
     * @throws NullPointerException if {@code keyImportOptions} is {@code null}.
     * @throws HttpResponseException if {@code name} is empty string.
     */
    public KeyVaultKey importKey(ImportKeyOptions importKeyOptions) {
        return importKeyWithResponse(importKeyOptions, Context.NONE).getValue();
    }

    /**
     * Imports an externally created key and stores it in key vault. The import key operation may be used to import any
     * key type into the Azure Key Vault. If the named key already exists, Azure Key Vault creates a new version of the
     * key. This operation requires the {@code keys/import} permission.
     *
     * <p>The {@code keyImportOptions} is required and its fields {@link ImportKeyOptions#getName() name} and {@link
     * ImportKeyOptions#getKey() key material} cannot be null. The {@link ImportKeyOptions#getExpiresOn() expires} and
     * {@link ImportKeyOptions#getNotBefore() notBefore} values in {@code keyImportOptions} are optional. If not specified,
     * no values are set for the fields. The {@link ImportKeyOptions#isEnabled() enabled} field is set to true and the
     * {@link ImportKeyOptions#isHardwareProtected() hsm} field is set to false by Azure Key Vault, if they are not specified.</p>
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Imports a new key into key vault. Prints out the details of the imported key.</p>
     *
     * <pre>
     * ImportKeyOptions importKeyOptions = new ImportKeyOptions&#40;&quot;keyName&quot;, jsonWebKeyToImport&#41;
     *     .setHardwareProtected&#40;false&#41;;
     * 
     * KeyVaultKey importedKeyResp = keyClient.importKeyWithResponse&#40;importKeyOptions, new Context&#40;key1, value1&#41;&#41;
     *     .getValue&#40;&#41;;
     * System.out.printf&#40;&quot;Key is imported with name %s and id %s &#92;n&quot;, importedKeyResp.getName&#40;&#41;,
     *     importedKeyResp.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param importKeyOptions The key import configuration object containing information about the json web key
     *     being imported.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link KeyVaultKey imported key}.
     * @throws NullPointerException if {@code keyImportOptions} is {@code null}.
     * @throws HttpResponseException if {@code name} is empty string.
     */
    public Response<KeyVaultKey> importKeyWithResponse(ImportKeyOptions importKeyOptions, Context context) {
        return client.importKeyWithResponse(importKeyOptions, context).block();
    }

    /**
     * Gets the public part of the specified key and key version. The get key operation is applicable to all key types
     * and it requires the {@code keys/get} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets a specific version of the key in the key vault. Prints out the details of the returned key.</p>
     * <pre>
     * String keyVersion = &quot;6A385B124DEF4096AF1361A85B16C204&quot;;
     * KeyVaultKey keyWithVersion = keyClient.getKey&#40;&quot;keyName&quot;, keyVersion&#41;;
     * System.out.printf&#40;&quot;Key is returned with name %s and id %s %n&quot;, keyWithVersion.getName&#40;&#41;,
     *     keyWithVersion.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param name The name of the key, cannot be null
     * @param version The version of the key to retrieve. If this is an empty String or null, this call is
     *     equivalent to calling {@link KeyClient#getKey(String)}, with the latest version being retrieved.
     * @return The requested {@link KeyVaultKey key}. The content of the key is null if
     * both {@code name} and {@code version} are null or empty.
     * @throws ResourceNotFoundException when a key with {@code name} doesn't exist in the key vault or
     * an empty/null {@code name} and a non null/empty {@code version} is provided.
     * @throws HttpResponseException if a valid {@code name} and a non null/empty {@code version} is specified.
     */
    public KeyVaultKey getKey(String name, String version) {
        return getKeyWithResponse(name, version, Context.NONE).getValue();
    }

    /**
     * Gets the public part of the specified key and key version. The get key operation is applicable to all key types
     * and it requires the {@code keys/get} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets a specific version of the key in the key vault. Prints out the details of the returned key.</p>
     * <pre>
     * String keyVersion = &quot;6A385B124DEF4096AF1361A85B16C204&quot;;
     * KeyVaultKey keyWithVersion = keyClient.getKeyWithResponse&#40;&quot;keyName&quot;, keyVersion,
     *     new Context&#40;key1, value1&#41;&#41;.getValue&#40;&#41;;
     * System.out.printf&#40;&quot;Key is returned with name %s and id %s %n&quot;, keyWithVersion.getName&#40;&#41;,
     *     keyWithVersion.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param name The name of the key, cannot be null
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @param version The version of the key to retrieve. If this is an empty String or null, this call is
     *     equivalent to calling {@link KeyClient#getKey(String)}, with the latest version being retrieved.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the requested {@link KeyVaultKey key}.
     * The content of the key is null if both {@code name} and {@code version} are null or empty.
     * @throws ResourceNotFoundException when a key with {@code name} doesn't exist in the key vault or
     * an empty/null {@code name} and a non null/empty {@code version} is provided.
     * @throws HttpResponseException if a valid {@code name} and a non null/empty {@code version} is specified.
     */
    public Response<KeyVaultKey> getKeyWithResponse(String name, String version, Context context) {
        return client.getKeyWithResponse(name, version, context).block();
    }

    /**
     * Get the public part of the latest version of the specified key from the key vault. The get key operation is
     * applicable to all key types and it requires the {@code keys/get} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets the latest version of the key in the key vault. Prints out the details of the returned key.</p>
     * <pre>
     * KeyVaultKey keyWithVersionValue = keyClient.getKey&#40;&quot;keyName&quot;&#41;;
     * System.out.printf&#40;&quot;Key is returned with name %s and id %s %n&quot;, keyWithVersionValue.getName&#40;&#41;,
     *     keyWithVersionValue.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param name The name of the key.
     * @return The requested {@link KeyVaultKey key}. The content of the key is null if {@code name} is null or empty.
     * @throws ResourceNotFoundException when a key with non null/empty {@code name} doesn't exist in the key vault.
     * @throws HttpResponseException if a non null/empty and an invalid {@code name} is specified.
     */
    public KeyVaultKey getKey(String name) {
        return getKeyWithResponse(name, "", Context.NONE).getValue();
    }

    /**
     * Updates the attributes and key operations associated with the specified key, but not the cryptographic key
     * material of the specified key in the key vault. The update operation changes specified attributes of an existing
     * stored key and attributes that are not specified in the request are left unchanged. The cryptographic key
     * material of a key itself cannot be changed. This operation requires the {@code keys/set} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets the latest version of the key, changes its expiry time and key operations and the updates the key in the
     * key vault.</p>
     * <pre>
     * KeyVaultKey key = keyClient.getKey&#40;&quot;keyName&quot;&#41;;
     * key.getProperties&#40;&#41;.setExpiresOn&#40;OffsetDateTime.now&#40;&#41;.plusDays&#40;60&#41;&#41;;
     * KeyVaultKey updatedKey = keyClient.updateKeyProperties&#40;key.getProperties&#40;&#41;, KeyOperation.ENCRYPT,
     *     KeyOperation.DECRYPT&#41;;
     * System.out.printf&#40;&quot;Key is updated with name %s and id %s %n&quot;, updatedKey.getName&#40;&#41;, updatedKey.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param keyProperties The {@link KeyProperties key properties} object with updated properties.
     * @param keyOperations The updated key operations to associate with the key.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link KeyVaultKey updated key}.
     * @throws NullPointerException if {@code key} is {@code null}.
     * @throws ResourceNotFoundException when a key with {@link KeyProperties#getName() name} and {@link KeyProperties#getVersion()
     *     version} doesn't exist in the key vault.
     * @throws HttpResponseException if {@link KeyProperties#getName() name} or {@link KeyProperties#getVersion() version} is empty
     *     string.
     */
    public KeyVaultKey updateKeyProperties(KeyProperties keyProperties, KeyOperation... keyOperations) {
        return updateKeyPropertiesWithResponse(keyProperties, Context.NONE, keyOperations).getValue();
    }

    /**
     * Updates the attributes and key operations associated with the specified key, but not the cryptographic key
     * material of the specified key in the key vault. The update operation changes specified attributes of an existing
     * stored key and attributes that are not specified in the request are left unchanged. The cryptographic key
     * material of a key itself cannot be changed. This operation requires the {@code keys/set} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets the latest version of the key, changes its expiry time and key operations and the updates the key in the
     * key vault.</p>
     * <pre>
     * KeyVaultKey key = keyClient.getKey&#40;&quot;keyName&quot;&#41;;
     * key.getProperties&#40;&#41;.setExpiresOn&#40;OffsetDateTime.now&#40;&#41;.plusDays&#40;60&#41;&#41;;
     * KeyVaultKey updatedKey = keyClient.updateKeyPropertiesWithResponse&#40;key.getProperties&#40;&#41;,
     *     new Context&#40;key1, value1&#41;, KeyOperation.ENCRYPT, KeyOperation.DECRYPT&#41;.getValue&#40;&#41;;
     * System.out.printf&#40;&quot;Key is updated with name %s and id %s %n&quot;, updatedKey.getName&#40;&#41;, updatedKey.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param keyProperties The {@link KeyProperties key properties} object with updated properties.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @param keyOperations The updated key operations to associate with the key.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link KeyVaultKey updated key}.
     * @throws NullPointerException if {@code key} is {@code null}.
     * @throws ResourceNotFoundException when a key with {@link KeyProperties#getName() name} and {@link KeyProperties#getVersion()
     *     version} doesn't exist in the key vault.
     * @throws HttpResponseException if {@link KeyProperties#getName() name} or {@link KeyProperties#getVersion() version} is empty
     *     string.
     */
    public Response<KeyVaultKey> updateKeyPropertiesWithResponse(KeyProperties keyProperties, Context context, KeyOperation... keyOperations) {
        return client.updateKeyPropertiesWithResponse(keyProperties, context, keyOperations).block();
    }

    /**
     * Deletes a key of any type from the key vault. If soft-delete is enabled on the key vault then the key is placed
     * in the deleted state and requires to be purged for permanent deletion else the key is permanently deleted. The
     * delete operation applies to any key stored in Azure Key Vault but it cannot be applied to an individual version
     * of a key. This operation removes the cryptographic material associated with the key, which means the key is not
     * usable for Sign/Verify, Wrap/Unwrap or Encrypt/Decrypt operations. This operation requires the {@code
     * keys/delete} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Deletes the key from the keyvault. Prints out the recovery id of the deleted key returned in the
     * response.</p>
     * <pre>
     * SyncPoller&lt;DeletedKey, Void&gt; deletedKeyPoller = keyClient.beginDeleteKey&#40;&quot;keyName&quot;&#41;;
     * 
     * PollResponse&lt;DeletedKey&gt; deletedKeyPollResponse = deletedKeyPoller.poll&#40;&#41;;
     * 
     * &#47;&#47; Deleted date only works for SoftDelete Enabled Key Vault.
     * DeletedKey deletedKey = deletedKeyPollResponse.getValue&#40;&#41;;
     * System.out.println&#40;&quot;Deleted Date  %s&quot; + deletedKey.getDeletedOn&#40;&#41;.toString&#40;&#41;&#41;;
     * System.out.printf&#40;&quot;Deleted Key's Recovery Id %s&quot;, deletedKey.getRecoveryId&#40;&#41;&#41;;
     * 
     * &#47;&#47; Key is being deleted on server.
     * deletedKeyPoller.waitForCompletion&#40;&#41;;
     * &#47;&#47; Key is deleted
     * </pre>
     *
     * @param name The name of the key to be deleted.
     * @return A {@link SyncPoller} to poll on and retrieve {@link DeletedKey deleted key}
     * @throws ResourceNotFoundException when a key with {@code name} doesn't exist in the key vault.
     * @throws HttpResponseException when a key with {@code name} is empty string.
     */
    @ServiceMethod(returns = ReturnType.LONG_RUNNING_OPERATION)
    public SyncPoller<DeletedKey, Void> beginDeleteKey(String name) {
        return client.beginDeleteKey(name).getSyncPoller();
    }

    /**
     * Gets the public part of a deleted key. The Get Deleted Key operation is applicable for soft-delete enabled
     * vaults. This operation requires the {@code keys/get} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets the deleted key from the key vault enabled for soft-delete. Prints out the details of the deleted key
     * returned in the response.</p>
     * //Assuming key is deleted on a soft-delete enabled key vault.
     * <pre>
     * DeletedKey deletedKey = keyClient.getDeletedKey&#40;&quot;keyName&quot;&#41;;
     * System.out.printf&#40;&quot;Deleted Key's Recovery Id %s&quot;, deletedKey.getRecoveryId&#40;&#41;&#41;;
     * </pre>
     *
     * @param name The name of the deleted key.
     * @return The {@link DeletedKey deleted key}.
     * @throws ResourceNotFoundException when a key with {@code name} doesn't exist in the key vault.
     * @throws HttpResponseException when a key with {@code name} is empty string.
     */
    public DeletedKey getDeletedKey(String name) {
        return getDeletedKeyWithResponse(name, Context.NONE).getValue();
    }

    /**
     * Gets the public part of a deleted key. The Get Deleted Key operation is applicable for soft-delete enabled
     * vaults. This operation requires the {@code keys/get} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Gets the deleted key from the key vault enabled for soft-delete. Prints out the details of the deleted key
     * returned in the response.</p>
     * //Assuming key is deleted on a soft-delete enabled key vault.
     * <pre>
     * DeletedKey deletedKey = keyClient.getDeletedKeyWithResponse&#40;&quot;keyName&quot;, new Context&#40;key1, value1&#41;&#41;
     *     .getValue&#40;&#41;;
     * System.out.printf&#40;&quot;Deleted Key with recovery Id %s %n&quot;, deletedKey.getRecoveryId&#40;&#41;&#41;;
     * </pre>
     *
     * @param name The name of the deleted key.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link DeletedKey deleted key}.
     * @throws ResourceNotFoundException when a key with {@code name} doesn't exist in the key vault.
     * @throws HttpResponseException when a key with {@code name} is empty string.
     */
    public Response<DeletedKey> getDeletedKeyWithResponse(String name, Context context) {
        return client.getDeletedKeyWithResponse(name, context).block();
    }

    /**
     * Permanently deletes the specified key without the possibility of recovery. The Purge Deleted Key operation is
     * applicable for soft-delete enabled vaults. This operation requires the {@code keys/purge} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Purges the deleted key from the key vault enabled for soft-delete. Prints out the status code from the server
     * response.</p>
     * //Assuming key is deleted on a soft-delete enabled key vault.
     * <pre>
     * keyClient.purgeDeletedKey&#40;&quot;deletedKeyName&quot;&#41;;
     * </pre>
     *
     * @param name The name of the deleted key.
     * @throws ResourceNotFoundException when a key with {@code name} doesn't exist in the key vault.
     * @throws HttpResponseException when a key with {@code name} is empty string.
     */
    public void purgeDeletedKey(String name) {
        purgeDeletedKeyWithResponse(name, Context.NONE);
    }

    /**
     * Permanently deletes the specified key without the possibility of recovery. The Purge Deleted Key operation is
     * applicable for soft-delete enabled vaults. This operation requires the {@code keys/purge} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Purges the deleted key from the key vault enabled for soft-delete. Prints out the status code from the server
     * response.</p>
     * //Assuming key is deleted on a soft-delete enabled key vault.
     * <pre>
     * Response&lt;Void&gt; purgedResponse = keyClient.purgeDeletedKeyWithResponse&#40;&quot;deletedKeyName&quot;,
     *     new Context&#40;key2, value2&#41;&#41;;
     * System.out.printf&#40;&quot;Purge Status Code: %d %n&quot;, purgedResponse.getStatusCode&#40;&#41;&#41;;
     * </pre>
     *
     * @param name The name of the deleted key.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return A response containing status code and HTTP headers.
     * @throws ResourceNotFoundException when a key with {@code name} doesn't exist in the key vault.
     * @throws HttpResponseException when a key with {@code name} is empty string.
     */
    public Response<Void> purgeDeletedKeyWithResponse(String name, Context context) {
        return client.purgeDeletedKeyWithResponse(name, context).block();
    }

    /**
     * Recovers the deleted key in the key vault to its latest version and can only be performed on a soft-delete
     * enabled vault. An attempt to recover an non-deleted key will return an error. Consider this the inverse of the
     * delete operation on soft-delete enabled vaults. This operation requires the {@code keys/recover} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Recovers the deleted key from the key vault enabled for soft-delete.</p>
     * //Assuming key is deleted on a soft-delete enabled key vault.
     * <pre>
     * SyncPoller&lt;KeyVaultKey, Void&gt; recoverKeyPoller = keyClient.beginRecoverDeletedKey&#40;&quot;deletedKeyName&quot;&#41;;
     * 
     * PollResponse&lt;KeyVaultKey&gt; recoverKeyPollResponse = recoverKeyPoller.poll&#40;&#41;;
     * 
     * KeyVaultKey recoveredKey = recoverKeyPollResponse.getValue&#40;&#41;;
     * System.out.println&#40;&quot;Recovered Key Name %s&quot; + recoveredKey.getName&#40;&#41;&#41;;
     * System.out.printf&#40;&quot;Recovered Key's Id %s&quot;, recoveredKey.getId&#40;&#41;&#41;;
     * 
     * &#47;&#47; Key is being recovered on server.
     * recoverKeyPoller.waitForCompletion&#40;&#41;;
     * &#47;&#47; Key is recovered
     * </pre>
     *
     * @param name The name of the deleted key to be recovered.
     * @return A {@link SyncPoller} to poll on and retrieve {@link KeyVaultKey recovered key}.
     * @throws ResourceNotFoundException when a key with {@code name} doesn't exist in the key vault.
     * @throws HttpResponseException when a key with {@code name} is empty string.
     */
    @ServiceMethod(returns = ReturnType.LONG_RUNNING_OPERATION)
    public SyncPoller<KeyVaultKey, Void> beginRecoverDeletedKey(String name) {
        return client.beginRecoverDeletedKey(name).getSyncPoller();
    }

    /**
     * Requests a backup of the specified key be downloaded to the client. The Key Backup operation exports a key from
     * Azure Key Vault in a protected form. Note that this operation does not return key material in a form that can be
     * used outside the Azure Key Vault system, the returned key material is either protected to a Azure Key Vault HSM
     * or to Azure Key Vault itself. The intent of this operation is to allow a client to generate a key in one Azure
     * Key Vault instance, backup the key, and then restore it into another Azure Key Vault instance. The backup
     * operation may be used to export, in protected form, any key type from Azure Key Vault. Individual versions of a
     * key cannot be backed up. Backup / Restore can be performed within geographical boundaries only; meaning that a
     * backup from one geographical area cannot be restored to another geographical area. For example, a backup from the
     * US geographical area cannot be restored in an EU geographical area. This operation requires the {@code
     * key/backup} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Backs up the key from the key vault and prints out the length of the key's backup byte array returned in the
     * response</p>
     * <pre>
     * byte[] keyBackup = keyClient.backupKey&#40;&quot;keyName&quot;&#41;;
     * System.out.printf&#40;&quot;Key's Backup Byte array's length %s&quot;, keyBackup.length&#41;;
     * </pre>
     *
     * @param name The name of the key.
     * @return The backed up key blob.
     * @throws ResourceNotFoundException when a key with {@code name} doesn't exist in the key vault.
     * @throws HttpResponseException when a key with {@code name} is empty string.
     */
    public byte[] backupKey(String name) {
        return backupKeyWithResponse(name, Context.NONE).getValue();
    }

    /**
     * Requests a backup of the specified key be downloaded to the client. The Key Backup operation exports a key from
     * Azure Key Vault in a protected form. Note that this operation does not return key material in a form that can be
     * used outside the Azure Key Vault system, the returned key material is either protected to a Azure Key Vault HSM
     * or to Azure Key Vault itself. The intent of this operation is to allow a client to generate a key in one Azure
     * Key Vault instance, backup the key, and then restore it into another Azure Key Vault instance. The backup
     * operation may be used to export, in protected form, any key type from Azure Key Vault. Individual versions of a
     * key cannot be backed up. Backup / Restore can be performed within geographical boundaries only; meaning that a
     * backup from one geographical area cannot be restored to another geographical area. For example, a backup from the
     * US geographical area cannot be restored in an EU geographical area. This operation requires the {@code
     * key/backup} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Backs up the key from the key vault and prints out the length of the key's backup byte array returned in the
     * response</p>
     * <pre>
     * byte[] keyBackup = keyClient.backupKeyWithResponse&#40;&quot;keyName&quot;, new Context&#40;key2, value2&#41;&#41;.getValue&#40;&#41;;
     * System.out.printf&#40;&quot;Key's Backup Byte array's length %s&quot;, keyBackup.length&#41;;
     * </pre>
     *
     * @param name The name of the key.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the backed up key blob.
     * @throws ResourceNotFoundException when a key with {@code name} doesn't exist in the key vault.
     * @throws HttpResponseException when a key with {@code name} is empty string.
     */
    public Response<byte[]> backupKeyWithResponse(String name, Context context) {
        return client.backupKeyWithResponse(name, context).block();
    }

    /**
     * Restores a backed up key to a vault. Imports a previously backed up key into Azure Key Vault, restoring the key,
     * its key identifier, attributes and access control policies. The restore operation may be used to import a
     * previously backed up key. Individual versions of a key cannot be restored. The key is restored in its entirety
     * with the same key name as it had when it was backed up. If the key name is not available in the target Key Vault,
     * the restore operation will be rejected. While the key name is retained during restore, the final key identifier
     * will change if the key is restored to a different vault. Restore will restore all versions and preserve version
     * identifiers. The restore operation is subject to security constraints: The target Key Vault must be owned by the
     * same Microsoft Azure Subscription as the source Key Vault The user must have restore permission in the target Key
     * Vault. This operation requires the {@code keys/restore} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Restores the key in the key vault from its backup. Prints out the details of the restored key returned in the
     * response.</p>
     * //Pass the Key Backup Byte array to the restore operation.
     * <pre>
     * byte[] keyBackupByteArray = &#123;&#125;;
     * KeyVaultKey keyResponse = keyClient.restoreKeyBackup&#40;keyBackupByteArray&#41;;
     * System.out.printf&#40;&quot;Restored Key with name %s and id %s %n&quot;, keyResponse.getName&#40;&#41;, keyResponse.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param backup The backup blob associated with the key.
     * @return The {@link KeyVaultKey restored key}.
     * @throws ResourceModifiedException when {@code backup} blob is malformed.
     */
    public KeyVaultKey restoreKeyBackup(byte[] backup) {
        return restoreKeyBackupWithResponse(backup, Context.NONE).getValue();
    }

    /**
     * Restores a backed up key to a vault. Imports a previously backed up key into Azure Key Vault, restoring the key,
     * its key identifier, attributes and access control policies. The restore operation may be used to import a
     * previously backed up key. Individual versions of a key cannot be restored. The key is restored in its entirety
     * with the same key name as it had when it was backed up. If the key name is not available in the target Key Vault,
     * the restore operation will be rejected. While the key name is retained during restore, the final key identifier
     * will change if the key is restored to a different vault. Restore will restore all versions and preserve version
     * identifiers. The restore operation is subject to security constraints: The target Key Vault must be owned by the
     * same Microsoft Azure Subscription as the source Key Vault The user must have restore permission in the target Key
     * Vault. This operation requires the {@code keys/restore} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Restores the key in the key vault from its backup. Prints out the details of the restored key returned in the
     * response.</p>
     * //Pass the Key Backup Byte array to the restore operation.
     * <pre>
     * byte[] keyBackupByteArray = &#123;&#125;;
     * Response&lt;KeyVaultKey&gt; keyResponse = keyClient.restoreKeyBackupWithResponse&#40;keyBackupByteArray,
     *     new Context&#40;key1, value1&#41;&#41;;
     * System.out.printf&#40;&quot;Restored Key with name %s and id %s %n&quot;,
     *     keyResponse.getValue&#40;&#41;.getName&#40;&#41;, keyResponse.getValue&#40;&#41;.getId&#40;&#41;&#41;;
     * </pre>
     *
     * @param backup The backup blob associated with the key.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return A {@link Response} whose {@link Response#getValue() value} contains the {@link KeyVaultKey restored key}.
     * @throws ResourceModifiedException when {@code backup} blob is malformed.
     */
    public Response<KeyVaultKey> restoreKeyBackupWithResponse(byte[] backup, Context context) {
        return client.restoreKeyBackupWithResponse(backup, context).block();
    }

    /**
     * List keys in the key vault. Retrieves a list of the keys in the Key Vault as JSON Web Key structures that contain
     * the public part of a stored key. The List operation is applicable to all key types and the individual key
     * response in the list is represented by {@link KeyProperties} as only the key identifier, attributes and tags are
     * provided in the response. The key material and individual key versions are not listed in the response. This
     * operation requires the {@code keys/list} permission.
     *
     * <p>It is possible to get full keys with key material from this information. Loop over the {@link KeyProperties key}
     * and call {@link KeyClient#getKey(String, String)}. This will return the {@link KeyVaultKey key} with key material
     * included of its latest version.</p>
     * <pre>
     * for &#40;KeyProperties key : keyClient.listPropertiesOfKeys&#40;&#41;&#41; &#123;
     *     KeyVaultKey keyWithMaterial = keyClient.getKey&#40;key.getName&#40;&#41;, key.getVersion&#40;&#41;&#41;;
     *     System.out.printf&#40;&quot;Received key with name %s and type %s&quot;, keyWithMaterial.getName&#40;&#41;,
     *         keyWithMaterial.getKeyType&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * <p><strong>Code Samples to iterate keys by page</strong></p>
     * <p>It is possible to get full keys with key material from this information. Iterate over all the {@link KeyProperties
     * key} by page and call {@link KeyClient#getKey(String, String)}. This will return the {@link KeyVaultKey key} with key
     * material included of its latest version.</p>
     * <pre>
     * keyClient.listPropertiesOfKeys&#40;&#41;.iterableByPage&#40;&#41;.forEach&#40;resp -&gt; &#123;
     *     System.out.printf&#40;&quot;Got response headers . Url: %s, Status code: %d %n&quot;,
     *         resp.getRequest&#40;&#41;.getUrl&#40;&#41;, resp.getStatusCode&#40;&#41;&#41;;
     *     resp.getItems&#40;&#41;.forEach&#40;value -&gt; &#123;
     *         KeyVaultKey keyWithMaterial = keyClient.getKey&#40;value.getName&#40;&#41;, value.getVersion&#40;&#41;&#41;;
     *         System.out.printf&#40;&quot;Received key with name %s and type %s %n&quot;, keyWithMaterial.getName&#40;&#41;,
     *             keyWithMaterial.getKeyType&#40;&#41;&#41;;
     *     &#125;&#41;;
     * &#125;&#41;;
     * </pre>
     *
     * @return {@link PagedIterable} of {@link KeyProperties key} of all the keys in the vault.
     */
    public PagedIterable<KeyProperties> listPropertiesOfKeys() {
        return listPropertiesOfKeys(Context.NONE);
    }

    /**
     * List keys in the key vault. Retrieves a list of the keys in the Key Vault as JSON Web Key structures that contain
     * the public part of a stored key. The List operation is applicable to all key types and the individual key
     * response in the list is represented by {@link KeyProperties} as only the key identifier, attributes and tags are
     * provided in the response. The key material and individual key versions are not listed in the response. This
     * operation requires the {@code keys/list} permission.
     *
     * <p>It is possible to get full keys with key material from this information. Loop over the {@link KeyProperties key}
     * and call {@link KeyClient#getKey(String, String)}. This will return the {@link KeyVaultKey key} with key material
     * included of its latest version.</p>
     * <pre>
     * for &#40;KeyProperties key : keyClient.listPropertiesOfKeys&#40;new Context&#40;key2, value2&#41;&#41;&#41; &#123;
     *     KeyVaultKey keyWithMaterial = keyClient.getKey&#40;key.getName&#40;&#41;, key.getVersion&#40;&#41;&#41;;
     *     System.out.printf&#40;&quot;Received key with name %s and type %s&quot;, keyWithMaterial.getName&#40;&#41;,
     *         keyWithMaterial.getKeyType&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * <p><strong>Code Samples to iterate keys by page</strong></p>
     * <p>It is possible to get full keys with key material from this information. Iterate over all the {@link KeyProperties
     * key} by page and call {@link KeyClient#getKey(String, String)}. This will return the {@link KeyVaultKey key} with key
     * material included of its latest version.</p>
     * <pre>
     * keyClient.listPropertiesOfKeys&#40;&#41;.iterableByPage&#40;&#41;.forEach&#40;resp -&gt; &#123;
     *     System.out.printf&#40;&quot;Got response headers . Url: %s, Status code: %d %n&quot;,
     *         resp.getRequest&#40;&#41;.getUrl&#40;&#41;, resp.getStatusCode&#40;&#41;&#41;;
     *     resp.getItems&#40;&#41;.forEach&#40;value -&gt; &#123;
     *         KeyVaultKey keyWithMaterial = keyClient.getKey&#40;value.getName&#40;&#41;, value.getVersion&#40;&#41;&#41;;
     *         System.out.printf&#40;&quot;Received key with name %s and type %s %n&quot;, keyWithMaterial.getName&#40;&#41;,
     *             keyWithMaterial.getKeyType&#40;&#41;&#41;;
     *     &#125;&#41;;
     * &#125;&#41;;
     * </pre>
     *
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return {@link PagedIterable} of {@link KeyProperties key} of all the keys in the vault.
     */
    public PagedIterable<KeyProperties> listPropertiesOfKeys(Context context) {
        return new PagedIterable<>(client.listPropertiesOfKeys(context));
    }

    /**
     * Lists {@link DeletedKey deleted keys} of the key vault. The deleted keys are retrieved as JSON Web Key structures
     * that contain the public part of a deleted key. The Get Deleted Keys operation is applicable for vaults enabled
     * for soft-delete. This operation requires the {@code keys/list} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Lists the deleted keys in the key vault and for each deleted key prints out its recovery id.</p>
     * <pre>
     * for &#40;DeletedKey deletedKey : keyClient.listDeletedKeys&#40;&#41;&#41; &#123;
     *     System.out.printf&#40;&quot;Deleted key's recovery Id %s&quot;, deletedKey.getRecoveryId&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * <p><strong>Code Samples to iterate over deleted keys by page</strong></p>
     * <p>Iterate over the lists the deleted keys by each page in the key vault and for each deleted key prints out its
     * recovery id.</p>
     * <pre>
     * keyClient.listDeletedKeys&#40;&#41;.iterableByPage&#40;&#41;.forEach&#40;resp -&gt; &#123;
     *     System.out.printf&#40;&quot;Got response headers . Url: %s, Status code: %d %n&quot;,
     *         resp.getRequest&#40;&#41;.getUrl&#40;&#41;, resp.getStatusCode&#40;&#41;&#41;;
     *     resp.getItems&#40;&#41;.forEach&#40;value -&gt; &#123;
     *         System.out.printf&#40;&quot;Deleted key's recovery Id %s %n&quot;, value.getRecoveryId&#40;&#41;&#41;;
     *     &#125;&#41;;
     * &#125;&#41;;
     * </pre>
     *
     * @return {@link PagedIterable} of all of the {@link DeletedKey deleted keys} in the vault.
     */
    public PagedIterable<DeletedKey> listDeletedKeys() {
        return listDeletedKeys(Context.NONE);
    }

    /**
     * Lists {@link DeletedKey deleted keys} of the key vault. The deleted keys are retrieved as JSON Web Key structures
     * that contain the public part of a deleted key. The Get Deleted Keys operation is applicable for vaults enabled
     * for soft-delete. This operation requires the {@code keys/list} permission.
     *
     * <p><strong>Code Samples</strong></p>
     * <p>Lists the deleted keys in the key vault and for each deleted key prints out its recovery id.</p>
     * <pre>
     * for &#40;DeletedKey deletedKey : keyClient.listDeletedKeys&#40;new Context&#40;key2, value2&#41;&#41;&#41; &#123;
     *     System.out.printf&#40;&quot;Deleted key's recovery Id %s&quot;, deletedKey.getRecoveryId&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * <p><strong>Code Samples to iterate over deleted keys by page</strong></p>
     * <p>Iterate over the lists the deleted keys by each page in the key vault and for each deleted key prints out its
     * recovery id.</p>
     * <pre>
     * keyClient.listDeletedKeys&#40;&#41;.iterableByPage&#40;&#41;.forEach&#40;resp -&gt; &#123;
     *     System.out.printf&#40;&quot;Got response headers . Url: %s, Status code: %d %n&quot;,
     *         resp.getRequest&#40;&#41;.getUrl&#40;&#41;, resp.getStatusCode&#40;&#41;&#41;;
     *     resp.getItems&#40;&#41;.forEach&#40;value -&gt; &#123;
     *         System.out.printf&#40;&quot;Deleted key's recovery Id %s %n&quot;, value.getRecoveryId&#40;&#41;&#41;;
     *     &#125;&#41;;
     * &#125;&#41;;
     * </pre>
     *
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return {@link PagedIterable} of all of the {@link DeletedKey deleted keys} in the vault.
     */
    public PagedIterable<DeletedKey> listDeletedKeys(Context context) {
        return new PagedIterable<>(client.listDeletedKeys(context));
    }

    /**
     * List all versions of the specified key. The individual key response in the flux is represented by {@link KeyProperties}
     * as only the key identifier, attributes and tags are provided in the response. The key material values are
     * not provided in the response. This operation requires the {@code keys/list} permission.
     *
     * <p>It is possible to get full keys with key material for each version from this information. Loop over the
     * {@link KeyProperties key} and call {@link KeyClient#getKey(String, String)}. This will return the {@link KeyVaultKey keys}
     * with key material included of the specified versions.</p>
     * <pre>
     * for &#40;KeyProperties key : keyClient.listPropertiesOfKeyVersions&#40;&quot;keyName&quot;&#41;&#41; &#123;
     *     KeyVaultKey keyWithMaterial  = keyClient.getKey&#40;key.getName&#40;&#41;, key.getVersion&#40;&#41;&#41;;
     *     System.out.printf&#40;&quot;Received key's version with name %s, type %s and version %s&quot;,
     *         keyWithMaterial.getName&#40;&#41;,
     *             keyWithMaterial.getKeyType&#40;&#41;, keyWithMaterial.getProperties&#40;&#41;.getVersion&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * <p><strong>Code Samples to iterate over key versions by page</strong></p>
     * <p>It is possible to get full keys with key material for each version from this information. Iterate over all
     * the {@link KeyProperties key} by page and call {@link KeyClient#getKey(String, String)}. This will return the {@link
     * KeyVaultKey keys} with key material included of the specified versions.</p>
     * <pre>
     * keyClient.listPropertiesOfKeyVersions&#40;&quot;keyName&quot;&#41;.iterableByPage&#40;&#41;.forEach&#40;resp -&gt; &#123;
     *     System.out.printf&#40;&quot;Got response headers . Url: %s, Status code: %d %n&quot;,
     *         resp.getRequest&#40;&#41;.getUrl&#40;&#41;, resp.getStatusCode&#40;&#41;&#41;;
     *     resp.getItems&#40;&#41;.forEach&#40;value -&gt; &#123;
     *         System.out.printf&#40;&quot;Key name: %s, Key version: %s %n&quot;, value.getName&#40;&#41;, value.getVersion&#40;&#41;&#41;;
     *     &#125;&#41;;
     * &#125;&#41;;
     * </pre>
     *
     * @param name The name of the key.
     * @return {@link PagedIterable} of {@link KeyProperties key} of all the versions of the specified key in the vault. List
     *     is empty if key with {@code name} does not exist in key vault.
     * @throws ResourceNotFoundException when a given key {@code name} is null or an empty string.
     */
    public PagedIterable<KeyProperties> listPropertiesOfKeyVersions(String name) {
        return listPropertiesOfKeyVersions(name, Context.NONE);
    }

    /**
     * List all versions of the specified key. The individual key response in the flux is represented by {@link KeyProperties}
     * as only the key identifier, attributes and tags are provided in the response. The key material values are
     * not provided in the response. This operation requires the {@code keys/list} permission.
     *
     * <p>It is possible to get full keys with key material for each version from this information. Loop over the
     * {@link KeyProperties key} and call {@link KeyClient#getKey(String, String)}. This will return the {@link KeyVaultKey keys}
     * with key material included of the specified versions.</p>
     * <pre>
     * for &#40;KeyProperties key : keyClient.listPropertiesOfKeyVersions&#40;&quot;keyName&quot;&#41;&#41; &#123;
     *     KeyVaultKey keyWithMaterial  = keyClient.getKey&#40;key.getName&#40;&#41;, key.getVersion&#40;&#41;&#41;;
     *     System.out.printf&#40;&quot;Received key's version with name %s, type %s and version %s&quot;,
     *         keyWithMaterial.getName&#40;&#41;,
     *             keyWithMaterial.getKeyType&#40;&#41;, keyWithMaterial.getProperties&#40;&#41;.getVersion&#40;&#41;&#41;;
     * &#125;
     * </pre>
     *
     * <p><strong>Code Samples to iterate over key versions by page</strong></p>
     * <p>It is possible to get full keys with key material for each version from this information. Iterate over all
     * the {@link KeyProperties key} by page and call {@link KeyClient#getKey(String, String)}. This will return the
     * {@link KeyVaultKey keys} with key material included of the specified versions.</p>
     *
     * <pre>
     * keyClient.listPropertiesOfKeyVersions&#40;&quot;keyName&quot;&#41;.iterableByPage&#40;&#41;.forEach&#40;resp -&gt; &#123;
     *     System.out.printf&#40;&quot;Got response headers . Url: %s, Status code: %d %n&quot;,
     *         resp.getRequest&#40;&#41;.getUrl&#40;&#41;, resp.getStatusCode&#40;&#41;&#41;;
     *     resp.getItems&#40;&#41;.forEach&#40;value -&gt; &#123;
     *         System.out.printf&#40;&quot;Key name: %s, Key version: %s %n&quot;, value.getName&#40;&#41;, value.getVersion&#40;&#41;&#41;;
     *     &#125;&#41;;
     * &#125;&#41;;
     * </pre>
     *
     * @param name The name of the key.
     * @param context Additional context that is passed through the Http pipeline during the service call.
     * @return {@link PagedIterable} of {@link KeyProperties key} of all the versions of the specified key in the vault. List
     *     is empty if key with {@code name} does not exist in key vault.
     * @throws ResourceNotFoundException when a given key {@code name} is null or an empty string.
     */
    public PagedIterable<KeyProperties> listPropertiesOfKeyVersions(String name, Context context) {
        return new PagedIterable<>(client.listPropertiesOfKeyVersions(name, context));
    }
}
