/*
 * Copyright 2017 Google LLC
 *
 * Licensed for use under "ARCore Additional Terms of Service". You may obtain
 * a copy of the license at https://developers.google.com/ar/develop/terms.
 */

package com.google.ar.core;

/**
 * Describes a fixed location and orientation in the real world. To stay at a fixed location in
 * physical space, the numerical description of this position will update as ARCore's understanding
 * of the space improves.
 *
 * <p>Use {@link #getPose()} to get the current numerical location of this anchor. This location may
 * change any time {@link com.google.ar.core.Session#update() Session#update()} is called, but will
 * never spontaneously change.
 *
 * <p>mathematically:<br>
 * {@code point_world = anchor.getPose().transformPoint(point_local);}<br>
 * {@code point_world = anchor.getPose().toMatrix() * point_local;}
 *
 * <p>{@code anchor.getPose().toMatrix(...)} provides a matrix appropriate for composing with the
 * view and projection matrices when rendering an object at this anchor's location.
 *
 * <p>{@code Anchor}s are hashable and may for example be used as keys in {@code HashMap}s.
 *
 * <p>Anchors incur ongoing processing overhead within ARCore. To release unneeded anchors use
 * {@link #detach()}.
 */
@SuppressWarnings({"unchecked", "deprecation", "all"})
public class Anchor {

  Anchor() {
    // Method contents removed.
  }

  /**
   * Gets the current Terrain anchor state of this anchor. This state is guaranteed not to change
   * until {@link com.google.ar.core.Session#update Session#update} is called. For Anchors that are
   * not Terrain anchors, this function returns {@link
   * com.google.ar.core.Anchor.TerrainAnchorState#NONE TerrainAnchorState#NONE}. See {@link
   * com.google.ar.core.Anchor.TerrainAnchorState TerrainAnchorState} for the possible Terrain
   * anchor states.
   *
   * @deprecated For resolve operations created by the async method {@link
   *     com.google.ar.core.Earth#resolveAnchorOnTerrainAsync(double,double,double,float,float,float,float,BiConsumer)
   *     }, the result state can be obtained for the callback or future object.
   */
  @Deprecated
  @androidx.annotation.NonNull
  public com.google.ar.core.Anchor.TerrainAnchorState getTerrainAnchorState() {
    // Method contents removed.
  }

  /**
   * Indicates whether some other object is an {@code Anchor} referencing the same logical anchor as
   * this one.
   *
   * @param obj the reference object with which to compare.
   * @return {@code true} if this object is the same as the obj argument; {@code false} otherwise.
   * @see #hashCode()
   * @see java.util.HashMap
   */
  public boolean equals(java.lang.Object obj) {
    // Method contents removed.
  }

  /**
   * Returns a hash code value for the object. This method is supported for the benefit of hash
   * tables such as those provided by {@link java.util.HashMap}.
   *
   * @return a hash code value for this object.
   * @see #equals(Object)
   */
  public int hashCode() {
    // Method contents removed.
  }

  /** @hide */
  protected void finalize() throws java.lang.Throwable {
    // Method contents removed.
  }

  /**
   * Returns the pose of the anchor in the world coordinate space. This pose may change each time
   * {@link com.google.ar.core.Session#update() Session#update()} is called. This pose should only
   * be used for rendering if {@link #getTrackingState()} returns {@link
   * com.google.ar.core.TrackingState#TRACKING}.
   */
  public com.google.ar.core.Pose getPose() {
    // Method contents removed.
  }

  /**
   * Returns the current state of the pose of this anchor. If this state is anything other than
   * {@link com.google.ar.core.TrackingState#TRACKING} the pose should not be considered useful.
   *
   * <p>Note: Starting in ARCore 1.12, changing the active camera config using {@link
   * com.google.ar.core.Session#setCameraConfig(com.google.ar.core.CameraConfig) } may cause the
   * tracking state on certain devices to become permanently PAUSED. For consistent behavior across
   * all supported devices, release any previously created anchors when setting a new camera config.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.TrackingState getTrackingState() {
    // Method contents removed.
  }

  /**
   * Gets the current cloud anchor ID for this anchor. Returns an empty string if {@link
   * com.google.ar.core.Anchor#getCloudAnchorState() } returns {@link
   * com.google.ar.core.Anchor.CloudAnchorState#TASK_IN_PROGRESS
   * Anchor.CloudAnchorState#TASK_IN_PROGRESS} or {@link
   * com.google.ar.core.Anchor.CloudAnchorState#NONE Anchor.CloudAnchorState#NONE}.
   *
   * @deprecated For cloud anchors hosted by the async method {@link
   *     com.google.ar.core.Session#hostCloudAnchorAsync(com.google.ar.core.Anchor,int,BiConsumer)
   *     }, the result Cloud Anchor ID can be obtained from the corresponding callback or Future
   *     object. This method will always return the empty string for anchors created through async
   *     methods.
   */
  @Deprecated
  @androidx.annotation.NonNull
  public java.lang.String getCloudAnchorId() {
    // Method contents removed.
  }

  /**
   * Gets the current Cloud Anchor state of this anchor.
   *
   * @deprecated For Cloud Anchor operations created by the async Cloud Anchor methods, the result
   *     state can be obtained from the corresponding callback or {@link com.google.ar.core.Future
   *     Future} object. This method will always return {@link
   *     com.google.ar.core.Anchor.CloudAnchorState#NONE CloudAnchorState#NONE} for anchors created
   *     through async methods.
   */
  @Deprecated
  @androidx.annotation.NonNull
  public com.google.ar.core.Anchor.CloudAnchorState getCloudAnchorState() {
    // Method contents removed.
  }

  /**
   * Detaches this Anchor from its Trackable removes it from the Session. After calling,
   * getTrackingState() will return {@link com.google.ar.core.TrackingState#STOPPED} and ARCore will
   * no longer update this Anchor's pose or return it from {@link
   * com.google.ar.core.Session#getAllAnchors() Session#getAllAnchors()}.
   */
  public void detach() {
    // Method contents removed.
  }

  /** Result of a Cloud Anchor hosting or resolving operation. */
  @SuppressWarnings({"unchecked", "deprecation", "all"})
  public enum CloudAnchorState {
    /** Not a valid value for a Cloud Anchor hosting or resolving operation. */
    NONE,
    /**
     * A hosting/resolving task for the anchor is in progress. Once the task completes in the
     * background, the anchor will get a new cloud state after the next {@link
     * com.google.ar.core.Session#update() Session#update()} call.
     *
     * @deprecated For async Cloud Anchor tasks, the task is in progress when {@link
     *     com.google.ar.core.Future#getState() } returns {@link
     *     com.google.ar.core.FutureState#PENDING FutureState#PENDING}.
     */
    @Deprecated
    TASK_IN_PROGRESS,
    /** A hosting/resolving task for this anchor completed successfully. */
    SUCCESS,
    /**
     * A hosting/resolving task for this anchor finished with an internal error. The app should not
     * attempt to recover from this error.
     */
    ERROR_INTERNAL,
    /**
     * The app cannot communicate with the ARCore Cloud Anchor service. When using API key
     * authentication, this will happen if the API key in the manifest is invalid, unauthorized or
     * missing.
     */
    ERROR_NOT_AUTHORIZED,
    /**
     * ERROR_SERVICE_UNAVAILABLE
     *
     * @deprecated ERROR_SERVICE_UNAVAILABLE is deprecated in ARCore SDK 1.12. See release notes to
     *     learn more.
     */
    @Deprecated
    ERROR_SERVICE_UNAVAILABLE,
    /**
     * The application has exhausted the request quota allotted to the given Google Cloud project.
     * You should request additional quota for the ARCore API for your project from the Google
     * Developers Console.
     */
    ERROR_RESOURCE_EXHAUSTED,
    /**
     * Hosting failed because the server could not successfully process the dataset for the given
     * anchor. Try again after the device has gathered more data from the environment.
     */
    ERROR_HOSTING_DATASET_PROCESSING_FAILED,
    /**
     * Resolving failed because the ARCore Cloud Anchor service could not find the provided Cloud
     * Anchor ID.
     */
    ERROR_CLOUD_ID_NOT_FOUND,
    /**
     * ERROR_RESOLVING_LOCALIZATION_NO_MATCH
     *
     * @deprecated ERROR_RESOLVING_LOCALIZATION_NO_MATCH is deprecated in ARCore SDK 1.12. See
     *     release notes to learn more.
     */
    @Deprecated
    ERROR_RESOLVING_LOCALIZATION_NO_MATCH,
    /**
     * The Cloud Anchor could not be resolved because the SDK version used to resolve the anchor is
     * older than and incompatible with the version used to host it.
     */
    ERROR_RESOLVING_SDK_VERSION_TOO_OLD,
    /**
     * The Cloud Anchor could not be resolved because the SDK version used to resolve the anchor is
     * newer than and incompatible with the version used to host it.
     */
    ERROR_RESOLVING_SDK_VERSION_TOO_NEW,
    /**
     * The ARCore Cloud Anchor service was unreachable. This can happen for a number of reasons. The
     * device might be in airplane mode or may not have a working internet connection. The request
     * sent to the server might have timed out with no response, or there might be a bad network
     * connection, DNS unavailability, firewall issues, or anything else that might affect the
     * device's ability to connect to the ARCore Cloud Anchor service.
     */
    ERROR_HOSTING_SERVICE_UNAVAILABLE;

    public boolean isError() {
      // Method contents removed.
    }

    @androidx.annotation.NonNull
    public static com.google.ar.core.Anchor.CloudAnchorState forNumber(int nativeCode) {
      // Method contents removed.
    }
  }

  /**
   * Describes the Rooftop anchor state of asynchronous operation launched by {@link
   * com.google.ar.core.Earth#resolveAnchorOnRooftopAsync(double,double,double,float,float,float,float,BiConsumer)
   * }.
   *
   * <p>Obtained by {@link
   * com.google.ar.core.ResolveAnchorOnRooftopFuture#getResultRooftopAnchorState()
   * ResolveAnchorOnRooftopFuture#getResultRooftopAnchorState()}.
   */
  @SuppressWarnings({"unchecked", "deprecation", "all"})
  public enum RooftopAnchorState {
    /** Not a valid value for a Rooftop anchor operation. */
    NONE,
    /** A resolving task for this Rooftop anchor has finished successfully. */
    SUCCESS,
    /**
     * Resolving task for this Rooftop anchor finished with an internal error. The app should not
     * attempt to recover from this error.
     */
    ERROR_INTERNAL,
    /**
     * The authorization provided by the application is not valid. The Google Cloud project may not
     * have enabled the ARCore API. When using API key authentication, this will happen if the API
     * key in the manifest is invalid or unauthorized. It may also fail if the API key is restricted
     * to a set of apps not including the current one. When using keyless authentication, this may
     * happen when no OAuth client has been created, or when the signing key and package name
     * combination does not match the values used in the Google Cloud project. It may also fail if
     * Google Play Services isn't installed, is too old, or is malfunctioning for some reason (e.g.
     * killed due to memory pressure).
     */
    ERROR_NOT_AUTHORIZED,
    /**
     * There is no rooftop or terrain information at this location, such as the center of the ocean.
     */
    ERROR_UNSUPPORTED_LOCATION;

    public boolean isError() {
      // Method contents removed.
    }

    @androidx.annotation.NonNull
    public static com.google.ar.core.Anchor.RooftopAnchorState forNumber(int nativeCode) {
      // Method contents removed.
    }
  }

  /** Describes the result of a Terrain anchor operation. */
  @SuppressWarnings({"unchecked", "deprecation", "all"})
  public enum TerrainAnchorState {
    /** Not a valid state for a terrain anchor operation. */
    NONE,
    /**
     * Resolving the Terrain anchor is in progress. Once the task completes in the background, the
     * anchor will get a new Terrain anchor state after the next {@link
     * com.google.ar.core.Session#update() Session#update()} call.
     *
     * @deprecated For async resolving tasks, the task is in progress when {@link
     *     com.google.ar.core.Future#getState() Future#getState()} returns {@link
     *     com.google.ar.core.FutureState#PENDING FutureState#PENDING}.
     */
    @Deprecated
    TASK_IN_PROGRESS,
    /** A resolving task for this Terrain anchor has finished successfully. */
    SUCCESS,
    /**
     * Resolving task for this Terrain anchor finished with an internal error. The app should not
     * attempt to recover from this error.
     */
    ERROR_INTERNAL,
    /**
     * The authorization provided by the application is not valid. The Google Cloud project may not
     * have enabled the ARCore API. When using API key authentication, this will happen if the API
     * key in the manifest is invalid or unauthorized. It may also fail if the API key is restricted
     * to a set of apps not including the current one. When using keyless authentication, this may
     * happen when no OAuth client has been created, or when the signing key and package name
     * combination does not match the values used in the Google Cloud project. It may also fail if
     * Google Play Services isn't installed, is too old, or is malfunctioning for some reason (e.g.
     * killed due to memory pressure).
     */
    ERROR_NOT_AUTHORIZED,
    /** There is no terrain information at this location, such as the center of the ocean. */
    ERROR_UNSUPPORTED_LOCATION;

    public boolean isError() {
      // Method contents removed.
    }

    @androidx.annotation.NonNull
    public static com.google.ar.core.Anchor.TerrainAnchorState forNumber(int nativeCode) {
      // Method contents removed.
    }
  }
}
