/*
 * Copyright 2018 Google LLC
 *
 * Licensed for use under "ARCore Additional Terms of Service". You may obtain
 * a copy of the license at https://developers.google.com/ar/develop/terms.
 */

package com.google.ar.core;


/**
 * Describes a face detected by ARCore and provides methods to access additional center and face
 * region poses as well as face mesh related data.
 *
 * <p>Augmented Faces supports front-facing (selfie) camera only, and does not support attaching
 * anchors nor raycast hit testing. Calling {@link
 * com.google.ar.core.Trackable#createAnchor(com.google.ar.core.Pose) Trackable#createAnchor(Pose)}
 * will result in an {@code IllegalStateException}.
 *
 * <p>To use Augmented Faces, enable the feature in the session. This can be done at session
 * creation time, or at any time during session runtime:
 *
 * <pre>{@code
 * Session session = new Session(context, EnumSet.of(Session.Feature.FRONT_CAMERA));
 * Config config = ...
 * config.setAugmentedFaceMode(AugmentedFaceMode.MESH3D);
 * session.configure(config);
 * }</pre>
 *
 * When Augmented Face mode is enabled, ARCore updates the list of detected faces for each frame.
 * Use {@link com.google.ar.core.Session#getAllTrackables(java.lang.Class)
 * Session#getAllTrackables(Class)} and {@link com.google.ar.core.Trackable#getTrackingState()
 * Trackable#getTrackingState()} to get a list of faces that have valid meshes that can be rendered.
 *
 * <pre>{@code
 * for (AugmentedFace face : session.getAllTrackables(AugmentedFace.class)) {
 *   if (face.getTrackingState() == TrackingState.TRACKING) {
 *     // Render face mesh ...
 *   }
 * }
 * }</pre>
 *
 * Faces provide static mesh data that does not change during the session, as well as pose and mesh
 * data that is updated each frame:
 *
 * <pre>{@code
 * // UVs and indices can be cached as they do not change during the session.
 * FloatBuffer uvs = face.getMeshTextureCoordinates();
 * ShortBuffer indices = face.getMeshTriangleIndices();
 *
 * // Center and region poses, mesh vertices, and normals are updated each frame.
 * Pose facePose = face.getCenterPose();
 * FloatBuffer faceVertices = face.getMeshVertices();
 * FloatBuffer faceNormals = face.getMeshNormals();
 * }</pre>
 */
@SuppressWarnings({"unchecked", "deprecation", "all"})
public class AugmentedFace implements com.google.ar.core.Trackable {

  AugmentedFace() {
    // Method contents removed.
  }

  /**
   * Returns a float buffer of 3D vertices in (x, y, z) packing.
   *
   * <p>These vertices are relative to the center pose of the face with units in meters. When the
   * face trackable state is TRACKING, this data is synced with the latest frame. While face
   * trackable state is PAUSED the returned buffer size will be 0.
   *
   * @return a FloatBuffer of vertices when TRACKING, or an empty FloatBuffer when PAUSED.
   */
  public java.nio.FloatBuffer getMeshVertices() {
    // Method contents removed.
  }

  /**
   * Returns a float buffer of 3D normals in (x, y, z) packing.
   *
   * <p>Each (x, y, z) is a unit vector of the normal to the surface at each vertex. There is
   * exactly one normal vector for each vertex. These normals are relative to the center pose of the
   * face. When the face trackable state is TRACKING, this pose is synced with the latest frame.
   * While face trackable state is PAUSED the returned buffer size will be 0.
   *
   * @return a FloatBuffer of normals when TRACKING, or an empty FloatBuffer when PAUSED.
   */
  public java.nio.FloatBuffer getMeshNormals() {
    // Method contents removed.
  }

  /**
   * Returns a float buffer of UV texture coordinates in (u, v) packing.
   *
   * <p>There is a pair of texture coordinates for each vertex. These values do not change over a
   * session, however while face trackable state is PAUSED the returned buffer size will be 0.
   *
   * @return a FloatBuffer of texture coordinates when TRACKING, or an empty FloatBuffer when
   *     PAUSED.
   */
  public java.nio.FloatBuffer getMeshTextureCoordinates() {
    // Method contents removed.
  }

  /**
   * Returns a char buffer of triangles' indices in consecutive triplets.
   *
   * <p>Every three consecutive values are indices that represent a triangle. The vertex position
   * and texture coordinates are mapped by the indices. The front face of each triangle is defined
   * by the face where the vertices are in counter clockwise winding order. These values do not
   * change, however while face trackable state is PAUSED the returned buffer size will be 0.
   *
   * @return a ShortBuffer of triangle indices when TRACKING, or an empty ShortBuffer when PAUSED.
   */
  public java.nio.ShortBuffer getMeshTriangleIndices() {
    // Method contents removed.
  }

  /**
   * Returns the pose of a region in world coordinates when the face trackable state is TRACKING.
   * When face trackable state is PAUSED, an identity pose will be returned.
   *
   * @param regionType The face region for which to get the pose.
   * @return the Pose of the selected region when TRACKING, or an identity pose when PAUSED.
   */
  public com.google.ar.core.Pose getRegionPose(
      com.google.ar.core.AugmentedFace.RegionType regionType) {
    // Method contents removed.
  }

  /**
   * Returns the pose of the center of the face, defined to have the origin located behind the nose
   * and between the two cheek bones. Z+ is forward out of the nose, Y+ is upwards, and X+ is
   * towards the left. The units are in meters. When the face trackable state is TRACKING, this pose
   * is synced with the latest frame. When face trackable state is PAUSED, an identity pose will be
   * returned.
   *
   * <p>Use {@link
   * com.google.ar.core.AugmentedFace#getRegionPose(com.google.ar.core.AugmentedFace.RegionType)
   * AugmentedFace#getRegionPose(AugmentedFace.RegionType)} to retrieve poses of specific regions of
   * the face.
   */
  public com.google.ar.core.Pose getCenterPose() {
    // Method contents removed.
  }

  /** Gets this trackable's {@link com.google.ar.core.TrackingState TrackingState}. */
  @androidx.annotation.NonNull
  public com.google.ar.core.TrackingState getTrackingState() {
    // Method contents removed.
  }

  /**
   * Returns a hash code value for the object. This method is supported for the benefit of hash
   * tables such as those provided by {@link java.util.HashMap}.
   *
   * @return a hash code value for this object.
   * @see #equals(Object)
   */
  public int hashCode() {
    // Method contents removed.
  }

  @androidx.annotation.NonNull
  public com.google.ar.core.Anchor createAnchor(com.google.ar.core.Pose pose) {
    // Method contents removed.
  }

  /**
   * Indicates whether some other object is a {@code Trackable} referencing the same logical
   * trackable as this one.
   *
   * @param obj the reference object with which to compare.
   * @return {@code true} if this object is the same as the obj argument; {@code false} otherwise.
   * @see #hashCode()
   * @see java.util.HashMap
   */
  public boolean equals(java.lang.Object obj) {
    // Method contents removed.
  }

  @androidx.annotation.NonNull
  public java.util.Collection<com.google.ar.core.Anchor> getAnchors() {
    // Method contents removed.
  }

  /**
   * Defines face regions to query the pose for. Left and right are defined relative to the person
   * that the mesh belongs to. To retrieve the center pose use {@link
   * com.google.ar.core.AugmentedFace#getCenterPose() }.
   */
  @SuppressWarnings({"unchecked", "deprecation", "all"})
  public enum RegionType {
    /** The region at the tip of the nose. */
    NOSE_TIP,
    /** The region at the detected face's left side of the forehead. */
    FOREHEAD_LEFT,
    /** The region at the detected face's right side of the forehead. */
    FOREHEAD_RIGHT;
  }
}
