/*
 * Copyright 2018 Google LLC
 *
 * Licensed for use under "ARCore Additional Terms of Service". You may obtain
 * a copy of the license at https://developers.google.com/ar/develop/terms.
 */

package com.google.ar.core;


/**
 * A {@link com.google.ar.core.Trackable Trackable} implementation representing the Earth. Provides
 * localization ability in Earth-relative coordinates.
 *
 * <p>To access {@link com.google.ar.core.Earth Earth}, configure the session with an appropriate
 * {@link com.google.ar.core.Config.GeospatialMode Config.GeospatialMode} and use {@link
 * com.google.ar.core.Session#getEarth Session#getEarth}.
 *
 * <p>Not all devices support {@link com.google.ar.core.Config.GeospatialMode#ENABLED
 * Config.GeospatialMode#ENABLED}, use {@link com.google.ar.core.Session#isGeospatialModeSupported }
 * to check if the current device and selected camera support enabling this mode.
 *
 * <p>The {@link com.google.ar.core.Earth Earth} object should only be used when its {@link
 * com.google.ar.core.TrackingState TrackingState} is {@link
 * com.google.ar.core.TrackingState#TRACKING }, and otherwise should not be used. Use {@link
 * com.google.ar.core.Earth#getTrackingState Earth#getTrackingState} to obtain the current {@link
 * com.google.ar.core.TrackingState TrackingState}. If the {@link com.google.ar.core.TrackingState
 * TrackingState} does not become {@link com.google.ar.core.TrackingState#TRACKING }, then {@link
 * com.google.ar.core.Earth#getEarthState Earth#getEarthState} may contain more information in a
 * {@link com.google.ar.core.Earth.EarthState EarthState}.
 *
 * <p>Use {@link com.google.ar.core.Earth#getCameraGeospatialPose Earth#getCameraGeospatialPose} to
 * obtain the Earth-relative virtual camera pose for the latest frame.
 *
 * <p>Use {@link com.google.ar.core.Earth#createAnchor(double,double,double,float,float,float,float)
 * Earth#createAnchor(double, double, double, float, float, float, float)} to attach anchors to
 * Earth. Calling {@link com.google.ar.core.Earth#createAnchor(com.google.ar.core.Pose)
 * Earth#createAnchor(Pose)} on {@link com.google.ar.core.Earth Earth} will fail to create a new
 * anchor and will throw {@link java.lang.IllegalArgumentException IllegalArgumentException}.
 *
 * <p>{@link com.google.ar.core.Earth Earth} does not support hit testing and will never be a result
 * from {@link com.google.ar.core.Frame#hitTest }. Because {@link com.google.ar.core.Earth Earth} is
 * a type of {@link com.google.ar.core.Trackable Trackable}, the singleton {@link
 * com.google.ar.core.Earth Earth} instance may also be returned by {@link
 * com.google.ar.core.Session#getAllTrackables Session#getAllTrackables} when enabled.
 */
@SuppressWarnings({"unchecked", "deprecation", "all"})
public class Earth implements com.google.ar.core.Trackable {

  Earth() {
    // Method contents removed.
  }

  /**
   * Gets the current {@link com.google.ar.core.Earth.EarthState EarthState} of the {@link
   * com.google.ar.core.Earth Earth}.
   *
   * <p>This state is guaranteed not to change until {@link com.google.ar.core.Session#update
   * Session#update} is called.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.Earth.EarthState getEarthState() {
    // Method contents removed.
  }

  /**
   * Returns the {@link com.google.ar.core.GeospatialPose GeospatialPose} for the camera in the
   * latest frame, describing the geospatial positioning of the device.
   *
   * <p>The orientation of the obtained {@link com.google.ar.core.GeospatialPose GeospatialPose}
   * approximates the direction the user is facing in the EUS coordinate system. The EUS coordinate
   * system has X+ pointing east, Y+ pointing up, and Z+ pointing south.
   *
   * <p>Note: This pose is only valid when {@link com.google.ar.core.Earth#getTrackingState()
   * Earth#getTrackingState()} returns {@link com.google.ar.core.TrackingState#TRACKING },
   * otherwise, the resulting {@link com.google.ar.core.GeospatialPose GeospatialPose} will contain
   * invalid values.
   *
   * @throws java.lang.IllegalStateException when {@link com.google.ar.core.Session#getConfig
   *     Session#getConfig} does not have {@link com.google.ar.core.Config.GeospatialMode#ENABLED }
   *     set.
   * @throws com.google.ar.core.exceptions.NotTrackingException when localisation is not yet
   *     available.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.GeospatialPose getCameraGeospatialPose() {
    // Method contents removed.
  }

  /**
   * Converts the input pose to a {@link com.google.ar.core.GeospatialPose GeospatialPose} in the
   * same position as the original {@link com.google.ar.core.Pose Pose}.
   *
   * <p>{@link com.google.ar.core.Earth Earth's} current {@link
   * com.google.ar.core.Earth#getTrackingState() tracking state} must be {@link
   * com.google.ar.core.TrackingState#TRACKING } in order to obtain a value from this function
   * Otherwise, a {@link com.google.ar.core.exceptions.NotTrackingException} may be thrown.
   *
   * <p>The returned {@link com.google.ar.core.GeospatialPose GeospatialPose} does not contain a
   * valid heading value. See {@link com.google.ar.core.GeospatialPose#getEastUpSouthQuaternion() }
   * to obtain an orientation in 3D space.
   *
   * @param pose the {@link com.google.ar.core.Pose Pose} to be converted into a {@link
   *     com.google.ar.core.GeospatialPose GeospatialPose}.
   * @return the {@link com.google.ar.core.GeospatialPose GeospatialPose} in the same position.
   * @throws com.google.ar.core.exceptions.NotTrackingException when {@link com.google.ar.core.Earth
   *     Earth's} current {@link com.google.ar.core.Earth#getTrackingState() tracking state} is not
   *     {@link com.google.ar.core.TrackingState#TRACKING TrackingState#TRACKING}.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.GeospatialPose getGeospatialPose(com.google.ar.core.Pose pose) {
    // Method contents removed.
  }

  /**
   * Converts the input geospatial location and orientation relative to the Earth to a {@link
   * com.google.ar.core.Pose Pose} in the same position.
   *
   * <p>{@link com.google.ar.core.Earth Earth's} current {@link
   * com.google.ar.core.Earth#getTrackingState() tracking state} must be {@link
   * com.google.ar.core.TrackingState#TRACKING } in order to obtain a value from this function.
   * Otherwise, a {@link com.google.ar.core.exceptions.NotTrackingException} may be thrown.
   *
   * <p>Position near the north pole or south pole is not supported. If the latitude is within 0.1
   * degrees of the north pole or south pole (90 degrees or -90 degrees), this function will throw
   * {@link java.lang.IllegalArgumentException IllegalArgumentException}.
   *
   * @param latitude The latitude of the anchor relative to the WGS84 ellipsoid.
   * @param longitude The longitude of the anchor relative to the WGS84 ellipsoid.
   * @param altitude The altitude of the anchor relative to the WGS84 ellipsoid, in meters.
   * @param qx The x component of the EUS (east-up-south) rotation quaternion.
   * @param qy The y component of the EUS (east-up-south) rotation quaternion.
   * @param qz The z component of the EUS (east-up-south) rotation quaternion.
   * @param qw The w component of the EUS (east-up-south) rotation quaternion.
   * @return the {@link com.google.ar.core.Pose Pose} in the same position in gl world coordinate.
   * @throws com.google.ar.core.exceptions.NotTrackingException when {@link com.google.ar.core.Earth
   *     Earth's} current {@link com.google.ar.core.Earth#getTrackingState() tracking state} is not
   *     {@link com.google.ar.core.TrackingState#TRACKING TrackingState#TRACKING}.
   * @throws java.lang.IllegalArgumentException if {@code latitude} is outside the allowable range.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.Pose getPose(
      double latitude, double longitude, double altitude, float qx, float qy, float qz, float qw) {
    // Method contents removed.
  }

  /**
   * {@link com.google.ar.core.Earth Earth} does not support creating an {@link
   * com.google.ar.core.Anchor Anchor} using a {@link com.google.ar.core.Pose Pose}. Use {@link
   * com.google.ar.core.Earth#createAnchor(double,double,double,float,float,float,float) } instead.
   *
   * @deprecated use {@link
   *     com.google.ar.core.Earth#createAnchor(double,double,double,float,float,float,float)
   *     Earth#createAnchor(double, double, double, float, float, float, float)} instead.
   */
  @Deprecated
  public com.google.ar.core.Anchor createAnchor(com.google.ar.core.Pose pose) {
    // Method contents removed.
  }

  /**
   * Creates a new {@link com.google.ar.core.Anchor Anchor} at the specified geospatial location and
   * orientation relative to the Earth.
   *
   * <p>Latitude and longitude are defined by the <a
   * href="https://en.wikipedia.org/wiki/World_Geodetic_System">WGS84 specification</a>, and the
   * altitude value is defined by the elevation above the WGS84 ellipsoid in meters. To create an
   * anchor using an altitude relative to the Earth's terrain instead of altitude above the WGS84
   * ellipsoid, use {@link #resolveAnchorOnTerrainAsync}.
   *
   * <p>The rotation quaternion provided is with respect to an east-up-south coordinate frame. An
   * identity rotation will have the anchor oriented such that X+ points to the east, Y+ points up
   * away from the center of the earth, and Z+ points to the south.
   *
   * <p>An anchor's {@link com.google.ar.core.TrackingState TrackingState} will be {@link
   * com.google.ar.core.TrackingState#PAUSED TrackingState#PAUSED} while {@link
   * com.google.ar.core.Earth#getTrackingState() } is {@link com.google.ar.core.TrackingState#PAUSED
   * TrackingState#PAUSED}. Its tracking state will permanently become {@link
   * com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED} if {@link
   * com.google.ar.core.Session#configure(com.google.ar.core.Config) Session#configure(Config)} sets
   * the Geospatial mode to {@link com.google.ar.core.Config.GeospatialMode#DISABLED
   * Config.GeospatialMode#DISABLED}.
   *
   * <p>Creating anchors near the north pole or south pole is not supported. If the latitude is
   * within 0.1 degrees of the north pole or south pole (90 degrees or -90 degrees), this function
   * will throw {@link java.lang.IllegalArgumentException IllegalArgumentException}.
   *
   * @param latitude The latitude of the anchor relative to the WGS84 ellipsoid.
   * @param longitude The longitude of the anchor relative to the WGS84 ellipsoid.
   * @param altitude The altitude of the anchor relative to the WGS84 ellipsoid, in meters.
   * @param qx The x component of the EUS (east-up-south) rotation quaternion.
   * @param qy The y component of the EUS (east-up-south) rotation quaternion.
   * @param qz The z component of the EUS (east-up-south) rotation quaternion.
   * @param qw The w component of the EUS (east-up-south) rotation quaternion.
   * @return the anchor if successful; otherwise throws an exception.
   * @throws java.lang.IllegalStateException if {@link com.google.ar.core.Earth Earth} is {@link
   *     com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED} due to {@link
   *     com.google.ar.core.Config.GeospatialMode#DISABLED } configured on the {@link
   *     com.google.ar.core.Session Session}. Reacquire {@link com.google.ar.core.Earth Earth} if
   *     geospatial mode was reenabled.
   * @throws java.lang.IllegalArgumentException if {@code latitude} is outside the allowable range.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.Anchor createAnchor(
      double latitude, double longitude, double altitude, float qx, float qy, float qz, float qw) {
    // Method contents removed.
  }

  /**
   * Creates a new {@link com.google.ar.core.Anchor Anchor} at the specified geospatial location and
   * orientation relative to the Earth.
   *
   * <p>Latitude and longitude are defined by the <a
   * href="https://en.wikipedia.org/wiki/World_Geodetic_System">WGS84 specification</a>, and the
   * altitude value is defined by the elevation above the WGS84 ellipsoid in meters. To create an
   * anchor using an altitude relative to the Earth's terrain instead of altitude above the WGS84
   * ellipsoid, use {@link #resolveAnchorOnTerrainAsync}.
   *
   * <p>The rotation quaternion provided is with respect to an east-up-south coordinate frame. An
   * identity rotation will have the anchor oriented such that X+ points to the east, Y+ points up
   * away from the center of the earth, and Z+ points to the south.
   *
   * <p>An anchor's tracking state will be {@link com.google.ar.core.TrackingState#PAUSED
   * TrackingState#PAUSED} while {@link com.google.ar.core.Earth Earth}'s {@link
   * com.google.ar.core.Earth#getTrackingState() } is {@link com.google.ar.core.TrackingState#PAUSED
   * TrackingState#PAUSED}. Its tracking state will permanently become {@link
   * com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED} if {@link
   * com.google.ar.core.Session#configure(com.google.ar.core.Config) Session#configure(Config)} sets
   * the Geospatial mode to {@link com.google.ar.core.Config.GeospatialMode#DISABLED
   * Config.GeospatialMode#DISABLED}.
   *
   * <p>Creating anchors near the north pole or south pole is not supported. If the latitude is
   * within 0.1 degrees of the north pole or south pole (90 degrees or -90 degrees), this function
   * will throw {@link java.lang.IllegalArgumentException IllegalArgumentException}.
   *
   * @param latitude The latitude of the anchor relative to the WGS84 ellipsoid.
   * @param longitude The longitude of the anchor relative to the WGS84 ellipsoid.
   * @param altitude The altitude of the anchor relative to the WGS84 ellipsoid.
   * @param rotation EUS (east-up-south) rotation quaternion in the order {x, y, z, w}.
   * @return the anchor if successful; otherwise throws an exception.
   * @throws java.lang.IllegalStateException if {@link com.google.ar.core.Earth Earth} is {@link
   *     com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED} due to {@link
   *     com.google.ar.core.Config.GeospatialMode#DISABLED } configured on the {@link
   *     com.google.ar.core.Session Session}. Reacquire {@link com.google.ar.core.Earth Earth} if
   *     geospatial mode was reenabled.
   * @throws java.lang.IllegalArgumentException if {@code latitude} is outside the allowable range.
   * @throws com.google.ar.core.exceptions.NotTrackingException when localisation is not yet
   *     available.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.Anchor createAnchor(
      double latitude, double longitude, double altitude, float[] rotation) {
    // Method contents removed.
  }

  /**
   * Creates a new {@link com.google.ar.core.Anchor Anchor} at a specified horizontal position and
   * altitude relative to the horizontal position's terrain. Terrain means the ground or ground
   * floor inside a building with VPS coverage. If the altitude relative to the WGS84 ellipsoid is
   * known, use {@link #createAnchor} instead.
   *
   * <p>The specified {@code altitudeAboveTerrain} is interpreted to be relative to the Earth's
   * terrain (or floor) at the specified latitude/longitude geospatial coordinates, rather than
   * relative to the WGS84 ellipsoid. Specifying an altitude of 0 will position the anchor directly
   * on the terrain (or floor) whereas specifying a positive altitude will position the anchor above
   * the terrain (or floor), against the direction of gravity.
   *
   * <p>This creates a new {@link com.google.ar.core.Anchor Anchor} and schedules a task to resolve
   * the anchor's pose using the given parameters. You may resolve multiple anchors at a time, but a
   * session cannot be tracking more than 100 Terrain anchors at time. Attempting to resolve more
   * than 100 Terrain Anchors will throw {@link
   * com.google.ar.core.exceptions.ResourceExhaustedException}.
   *
   * <p>The returned {@link com.google.ar.core.Anchor Anchor} will have its {@link
   * com.google.ar.core.Anchor.TerrainAnchorState Anchor.TerrainAnchorState} set to {@link
   * com.google.ar.core.Anchor.TerrainAnchorState#TASK_IN_PROGRESS }, and its {@link
   * com.google.ar.core.TrackingState TrackingState} will be {@link
   * com.google.ar.core.TrackingState#PAUSED }. The anchor will remain in this state until its pose
   * has been successfully resolved. If the resolving task results in an error, the anchor's {@link
   * com.google.ar.core.TrackingState TrackingState} will be permanently set to {@link
   * com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED}, and {@link
   * com.google.ar.core.Anchor#getTerrainAnchorState } will contain a {@link
   * com.google.ar.core.Anchor.TerrainAnchorState Anchor.TerrainAnchorState} detailing error
   * information.
   *
   * <p>Creating a Terrain anchor requires an active {@link com.google.ar.core.Earth Earth} which is
   * {@link com.google.ar.core.Earth.EarthState#ENABLED }. If it is not, then this function throws
   * {@link java.lang.IllegalStateException IllegalStateException}. This call also requires a
   * working internet connection to communicate with the ARCore API on Google Cloud. ARCore will
   * continue to retry if it is unable to establish a connection to the ARCore service.
   *
   * <p>A Terrain anchor's tracking state will be {@link com.google.ar.core.TrackingState#PAUSED
   * TrackingState#PAUSED} while {@link com.google.ar.core.Earth Earth}'s {@link
   * com.google.ar.core.Earth#getTrackingState() Earth#getTrackingState()} is {@link
   * com.google.ar.core.TrackingState#PAUSED TrackingState#PAUSED}. Its tracking state will
   * permanently become {@link com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED} if
   * {@link com.google.ar.core.Session#configure(com.google.ar.core.Config)
   * Session#configure(Config)} is used to set {@link
   * com.google.ar.core.Config.GeospatialMode#DISABLED Config.GeospatialMode#DISABLED}.
   *
   * <p>Latitude and longitude are defined by the <a
   * href="https://en.wikipedia.org/wiki/World_Geodetic_System">WGS84 specification</a>.
   *
   * <p>The rotation quaternion provided is with respect to an east-up-south coordinate frame. An
   * identity rotation will have the anchor oriented such that X+ points to the east, Y+ points up
   * away from the center of the earth, and Z+ points to the south.
   *
   * @param latitude The latitude of the anchor relative to the WGS84 ellipsoid.
   * @param longitude The longitude of the anchor relative to the WGS84 ellipsoid.
   * @param altitudeAboveTerrain The altitude of the anchor above the Earth's terrain (or floor).
   * @param qx The x component of the EUS (east-up-south) rotation quaternion.
   * @param qy The y component of the EUS (east-up-south) rotation quaternion.
   * @param qz The z component of the EUS (east-up-south) rotation quaternion.
   * @param qw The w component of the EUS (east-up-south) rotation quaternion.
   * @return the anchor if successful; otherwise throws an exception.
   * @throws java.lang.IllegalStateException if {@link com.google.ar.core.Earth Earth} is {@link
   *     com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED} due to {@link
   *     com.google.ar.core.Config.GeospatialMode#DISABLED } configured on the {@link
   *     com.google.ar.core.Session Session}. Reacquire {@link com.google.ar.core.Earth Earth} if
   *     geospatial mode was reenabled.
   * @throws java.lang.IllegalArgumentException if {@code latitude} is outside the allowable range.
   * @throws com.google.ar.core.exceptions.NotTrackingException when localisation is not yet
   *     available.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if too many terrain anchors
   *     have already been held.
   * @deprecated Use {@link
   *     #resolveAnchorOnTerrainAsync(double,double,double,float,float,float,float,java.util.function.BiConsumer)}
   *     instead.
   */
  @Deprecated
  @androidx.annotation.NonNull
  public com.google.ar.core.Anchor resolveAnchorOnTerrain(
      double latitude,
      double longitude,
      double altitudeAboveTerrain,
      float qx,
      float qy,
      float qz,
      float qw) {
    // Method contents removed.
  }

  /**
   * Asynchronously creates a new {@link com.google.ar.core.Anchor Anchor} at a specified horizontal
   * position and altitude relative to the horizontal position's terrain. See the <a
   * href="https://developers.google.com/ar/develop/geospatial/java/anchors#terrain-anchors">Terrain
   * anchors developer guide</a> for more information.
   *
   * <p>The specified {@code altitudeAboveTerrain} is interpreted to be relative to the Earth's
   * terrain (or floor) at the specified latitude/longitude geospatial coordinates, rather than
   * relative to the WGS84 ellipsoid. Specifying an altitude of 0 will position the anchor directly
   * on the terrain (or floor) whereas specifying a positive altitude will position the anchor above
   * the terrain (or floor), against the direction of gravity.
   *
   * <p>This launches an asynchronous operation used to query the Google Cloud ARCore API. See
   * {@link com.google.ar.core.Future Future} for information on obtaining results and cancelling
   * the operation.
   *
   * <p>You may resolve multiple anchors at a time, but a session cannot be tracking more than 100
   * Terrain and Rooftop anchors at time. Attempting to resolve more than 100 Terrain or Rooftop
   * anchors will throw {@link com.google.ar.core.exceptions.ResourceExhaustedException}.
   *
   * <p>Creating a Terrain anchor requires an active {@link com.google.ar.core.Earth Earth} which is
   * {@link com.google.ar.core.Earth.EarthState#ENABLED }. If it is not, then this function throws
   * {@link java.lang.IllegalStateException IllegalStateException}. This call also requires a
   * working internet connection to communicate with the ARCore API on Google Cloud. ARCore will
   * continue to retry if it is unable to establish a connection to the ARCore service.
   *
   * <p>A Terrain anchor's tracking state will be {@link com.google.ar.core.TrackingState#PAUSED
   * TrackingState#PAUSED} while {@link com.google.ar.core.Earth Earth}'s {@link
   * com.google.ar.core.Earth#getTrackingState() Earth#getTrackingState()} is {@link
   * com.google.ar.core.TrackingState#PAUSED TrackingState#PAUSED}. Its tracking state will
   * permanently become {@link com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED} if
   * {@link com.google.ar.core.Session#configure(com.google.ar.core.Config)
   * Session#configure(Config)} is used to set {@link
   * com.google.ar.core.Config.GeospatialMode#DISABLED Config.GeospatialMode#DISABLED}.
   *
   * <p>Latitude and longitude are defined by the <a
   * href="https://en.wikipedia.org/wiki/World_Geodetic_System">WGS84 specification</a>.
   *
   * <p>The rotation quaternion provided is with respect to an east-up-south coordinate frame. An
   * identity rotation will have the anchor oriented such that X+ points to the east, Y+ points up
   * away from the center of the earth, and Z+ points to the south.
   *
   * @param latitude The latitude of the anchor relative to the WGS84 ellipsoid.
   * @param longitude The longitude of the anchor relative to the WGS84 ellipsoid.
   * @param altitudeAboveTerrain The altitude of the anchor above the Earth's terrain (or floor).
   * @param qx The x component of the EUS (east-up-south) rotation quaternion.
   * @param qy The y component of the EUS (east-up-south) rotation quaternion.
   * @param qz The z component of the EUS (east-up-south) rotation quaternion.
   * @param qw The w component of the EUS (east-up-south) rotation quaternion.
   * @param callback The callback to invoke on the Main thread, if any.
   * @return A handle to the async operation.
   * @throws java.lang.IllegalStateException if {@link com.google.ar.core.Earth Earth} is {@link
   *     com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED} due to {@link
   *     com.google.ar.core.Config.GeospatialMode#DISABLED } configured on the {@link
   *     com.google.ar.core.Session Session}. Reacquire {@link com.google.ar.core.Earth Earth} if
   *     geospatial mode was reenabled.
   * @throws java.lang.IllegalArgumentException if {@code latitude} is outside the allowable range.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if too many anchors (counting
   *     both Terrain and Rooftop anchors) have already been held or are being resolved.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.ResolveAnchorOnTerrainFuture resolveAnchorOnTerrainAsync(
      double latitude,
      double longitude,
      double altitudeAboveTerrain,
      float qx,
      float qy,
      float qz,
      float qw,
      @androidx.annotation.Nullable
          java.util.function.BiConsumer<
                  com.google.ar.core.Anchor, com.google.ar.core.Anchor.TerrainAnchorState>
              callback) {
    // Method contents removed.
  }

  /**
   * Creates an anchor at a specified horizontal position and altitude relative to the horizontal
   * position's rooftop. See the <a
   * href="https://developers.google.com/ar/develop/geospatial/java/anchors#rooftop-anchors">Rooftop
   * anchors developer guide</a> for more information.
   *
   * <p>The specified {@code altitudeAboveRooftop} is interpreted to be relative to the top of a
   * building at the given horizontal location, rather than relative to the WGS84 ellipsoid. If
   * there is no building at the given location, then the altitude is interpreted to be relative to
   * the terrain instead. Specifying an altitude of 0 will position the anchor directly on the
   * rooftop whereas specifying a positive altitude will position the anchor above the rooftop,
   * against the direction of gravity.
   *
   * <p>This launches an asynchronous operation used to query the Google Cloud ARCore API. See
   * {@link com.google.ar.core.Future Future} for information on obtaining results and cancelling
   * the operation.
   *
   * <p>You may resolve multiple anchors at a time, but a session cannot be tracking more than 100
   * Terrain and Rooftop anchors at time. Attempting to resolve more than 100 Terrain or Rooftop
   * anchors will throw {@link com.google.ar.core.exceptions.ResourceExhaustedException}.
   *
   * <p>Creating a Rooftop anchor requires an active {@link com.google.ar.core.Earth Earth} which is
   * {@link com.google.ar.core.Earth.EarthState#ENABLED }. If it is not, then this function throws
   * {@link java.lang.IllegalStateException IllegalStateException}. This call also requires a
   * working internet connection to communicate with the ARCore API on Google Cloud. ARCore will
   * continue to retry if it is unable to establish a connection to the ARCore service.
   *
   * <p>A Rooftop anchor's tracking state will be {@link com.google.ar.core.TrackingState#PAUSED
   * TrackingState#PAUSED} while {@link com.google.ar.core.Earth Earth}'s {@link
   * com.google.ar.core.Earth#getTrackingState() Earth#getTrackingState()} is {@link
   * com.google.ar.core.TrackingState#PAUSED TrackingState#PAUSED}. Its tracking state will
   * permanently become {@link com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED} if
   * {@link com.google.ar.core.Session#configure(com.google.ar.core.Config)
   * Session#configure(Config)} is used to set {@link
   * com.google.ar.core.Config.GeospatialMode#DISABLED Config.GeospatialMode#DISABLED}.
   *
   * <p>Latitude and longitude are defined by the <a
   * href="https://en.wikipedia.org/wiki/World_Geodetic_System">WGS84 specification</a>.
   *
   * <p>The rotation quaternion provided is with respect to an east-up-south coordinate frame. An
   * identity rotation will have the anchor oriented such that X+ points to the east, Y+ points up
   * away from the center of the earth, and Z+ points to the south.
   *
   * @param latitude The latitude of the anchor relative to the WGS84 ellipsoid.
   * @param longitude The longitude of the anchor relative to the WGS84 ellipsoid.
   * @param altitudeAboveRooftop The altitude of the anchor above the Earth's rooftop.
   * @param qx The x component of the EUS (east-up-south) rotation quaternion.
   * @param qy The y component of the EUS (east-up-south) rotation quaternion.
   * @param qz The z component of the EUS (east-up-south) rotation quaternion.
   * @param qw The w component of the EUS (east-up-south) rotation quaternion.
   * @param callback An optional <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_callback">callback</a>,
   *     called when the operation is finished, unless the future is <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_cancellation">cancelled</a>.
   *     May be null.
   * @return A handler that can be <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_polling">polled</a>
   *     or <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_cancellation">cancelled</a>.
   * @throws java.lang.IllegalStateException if {@link com.google.ar.core.Earth Earth} is {@link
   *     com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED} due to {@link
   *     com.google.ar.core.Config.GeospatialMode#DISABLED } configured on the {@link
   *     com.google.ar.core.Session Session}. Reacquire {@link com.google.ar.core.Earth Earth} if
   *     the Geospatial mode was re-enabled.
   * @throws java.lang.IllegalArgumentException if {@code latitude} is outside the allowable range.
   * @throws com.google.ar.core.exceptions.NotTrackingException when localisation is not yet
   *     available.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if too many Terrain and
   *     Rooftop anchors have already been held.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.ResolveAnchorOnRooftopFuture resolveAnchorOnRooftopAsync(
      double latitude,
      double longitude,
      double altitudeAboveRooftop,
      float qx,
      float qy,
      float qz,
      float qw,
      @androidx.annotation.Nullable
          java.util.function.BiConsumer<
                  com.google.ar.core.Anchor, com.google.ar.core.Anchor.RooftopAnchorState>
              callback) {
    // Method contents removed.
  }

  /**
   * Gets the {@link com.google.ar.core.TrackingState TrackingState} of the Earth object as of the
   * last frame.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.TrackingState getTrackingState() {
    // Method contents removed.
  }

  /** @hide */
  protected void finalize() throws java.lang.Throwable {
    // Method contents removed.
  }

  /**
   * Returns a hash code value for the object. This method is supported for the benefit of hash
   * tables such as those provided by {@link java.util.HashMap}.
   *
   * @return a hash code value for this object.
   * @see #equals(Object)
   */
  public int hashCode() {
    // Method contents removed.
  }

  /**
   * Indicates whether some other object is a {@code Trackable} referencing the same logical
   * trackable as this one.
   *
   * @param obj the reference object with which to compare.
   * @return {@code true} if this object is the same as the obj argument; {@code false} otherwise.
   * @see #hashCode()
   * @see java.util.HashMap
   */
  public boolean equals(java.lang.Object obj) {
    // Method contents removed.
  }

  @androidx.annotation.NonNull
  public java.util.Collection<com.google.ar.core.Anchor> getAnchors() {
    // Method contents removed.
  }

  /**
   * Describes the current state of {@link com.google.ar.core.Earth Earth}. When {@link
   * com.google.ar.core.Earth#getTrackingState Earth#getTrackingState} does not become {@link
   * com.google.ar.core.TrackingState#TRACKING TrackingState#TRACKING}, {@link
   * com.google.ar.core.Earth.EarthState EarthState} may contain the cause of this failure.
   *
   * <p>Obtain using {@link com.google.ar.core.Earth#getEarthState Earth#getEarthState}.
   */
  @SuppressWarnings({"unchecked", "deprecation", "all"})
  public enum EarthState {
    /**
     * {@link com.google.ar.core.Earth Earth} is enabled, and has not encountered any problems.
     * Check {@link com.google.ar.core.Earth#getTrackingState } to determine if it can be used.
     */
    ENABLED,
    /**
     * Earth localization has encountered an internal error. The app should not attempt to recover
     * from this error. Please see the Android logs for additional information.
     */
    ERROR_INTERNAL,
    /**
     * The given {@link com.google.ar.core.Earth Earth} is no longer valid and has {@link
     * com.google.ar.core.TrackingState TrackingState} {@link
     * com.google.ar.core.TrackingState#STOPPED } due to {@link
     * com.google.ar.core.Config.GeospatialMode#DISABLED Config.GeospatialMode#DISABLED} being set
     * on the {@link com.google.ar.core.Session Session}.
     */
    ERROR_GEOSPATIAL_MODE_DISABLED,
    /**
     * The authorization provided by the application is not valid.
     *
     * <ul>
     *   <li>The associated Google Cloud project may not have enabled the ARCore API.
     *   <li>When using API key authentication, this will happen if the API key in the manifest is
     *       invalid or unauthorized. It may also fail if the API key is restricted to a set of apps
     *       not including the current one.
     *   <li>When using keyless authentication, this may happen when no OAuth client has been
     *       created, or when the signing key and package name combination does not match the values
     *       used in the Google Cloud project. It may also fail if Google Play Services isn't
     *       installed, is too old, or is malfunctioning for some reason (e.g. killed due to memory
     *       pressure).
     * </ul>
     */
    ERROR_NOT_AUTHORIZED,
    /**
     * The application has exhausted the quota allotted to the given Google Cloud project. The
     * developer should <a
     * href="https://cloud.google.com/docs/quota#requesting_higher_quota">request additional
     * quota</a> for the ARCore API for their project from the Google Cloud Console.
     */
    ERROR_RESOURCE_EXHAUSTED,
    /** The APK is older than the current supported version. */
    ERROR_APK_VERSION_TOO_OLD;
  }
}
