/*
 * Copyright 2017 Google LLC
 *
 * Licensed for use under "ARCore Additional Terms of Service". You may obtain
 * a copy of the license at https://developers.google.com/ar/develop/terms.
 */

package com.google.ar.core;


/**
 * Captures the state and changes to the AR system from a call to {@link
 * com.google.ar.core.Session#update() Session#update()}.
 */
@SuppressWarnings({"unchecked", "deprecation", "all"})
public class Frame {

  Frame() {
    // Method contents removed.
  }

  /** @hide */
  protected void finalize() throws java.lang.Throwable {
    // Method contents removed.
  }

  /**
   * Returns the {@link com.google.ar.core.Camera Camera} object for the session. Note that {@link
   * com.google.ar.core.Camera Camera} instances are long-lived and may be kept across frames for
   * the duration of the session. Repeated calls to this method will return distinct {@link
   * com.google.ar.core.Camera Camera} objects that refer to the same underlying camera state and
   * are {@link #equals(java.lang.Object) equal} to each other.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.Camera getCamera() {
    // Method contents removed.
  }

  /**
   * Checks if the display rotation or viewport geometry changed since the previous {@code Frame}.
   * The application should re-query {@link
   * com.google.ar.core.Camera#getProjectionMatrix(float[],int,float,float)
   * Camera#getProjectionMatrix(float[], int, float, float)} and {@link
   * #transformCoordinates2d(com.google.ar.core.Coordinates2d,float[],com.google.ar.core.Coordinates2d,float[])}
   * whenever this is true.
   */
  public boolean hasDisplayGeometryChanged() {
    // Method contents removed.
  }

  /**
   * Transform the given texture coordinates to correctly show the background image. This will
   * account for the display rotation, and any additional required adjustment. For performance, this
   * function should be called only if {@link #hasDisplayGeometryChanged()} returns true.
   *
   * <p><b>Usage Notes / Bugs:</b>
   *
   * <ul>
   *   <li>Both input and output buffers must be direct and native byte order.
   *   <li>Position and limit of buffers is ignored.
   *   <li>Capacity of both buffers must be identical.
   *   <li>Capacity of both buffers must be a multiple of 2.
   * </ul>
   *
   * <p>Note: both buffer positions will remain unmodified after this call.
   *
   * @param uvCoords The uv coordinates to transform.
   * @param outUvCoords The buffer to hold the transformed uv coordinates. Must have enough
   *     remaining elements to fit the input uvCoords.
   * @deprecated Replaced by {@code frame.transformCoordinates2d(Coordinates2d.VIEW_NORMALIZED, ..,
   *     Coordinates2d.TEXTURE_NORMALIZED, ..)}.
   */
  @Deprecated
  public void transformDisplayUvCoords(
      java.nio.FloatBuffer uvCoords, java.nio.FloatBuffer outUvCoords) {
    // Method contents removed.
  }

  /**
   * Returns the timestamp in nanoseconds when this image was captured. This can be used to detect
   * dropped frames or measure the camera frame rate. The time base of this value is specifically
   * <b>not</b> defined, but it is likely similar to {@code System.nanoTime()}.
   */
  public long getTimestamp() {
    // Method contents removed.
  }

  /**
   * Returns the (<a
   * href="https://developer.android.com/reference/android/hardware/camera2/CaptureResult.html#SENSOR_TIMESTAMP">Android
   * Camera timestamp</a>) of the image.
   */
  public long getAndroidCameraTimestamp() {
    // Method contents removed.
  }

  /**
   * Returns the pose of the <a
   * href="https://developer.android.com/guide/topics/sensors/sensors_overview#sensors-coords">Android
   * Sensor Coordinate System</a> in the world coordinate space for this frame. The orientation
   * follows the device's "native" orientation (it is not affected by display rotation) with all
   * axes corresponding to those of the Android sensor coordinates.
   *
   * <p>See Also:
   *
   * <ul>
   *   <li>{@link com.google.ar.core.Camera#getPose() Camera#getPose()} for the pose of the physical
   *       camera.
   *   <li>{@link com.google.ar.core.Camera#getDisplayOrientedPose()
   *       Camera#getDisplayOrientedPose()} for the pose of the virtual camera.
   * </ul>
   *
   * Note: This pose is only useful when {@link com.google.ar.core.Camera#getTrackingState()
   * Camera#getTrackingState()} returns {@link com.google.ar.core.TrackingState#TRACKING } and
   * otherwise should not be used.
   */
  public com.google.ar.core.Pose getAndroidSensorPose() {
    // Method contents removed.
  }

  /**
   * Returns the current ambient light estimate, if light estimation was enabled.
   *
   * <p>If lighting estimation is not enabled in the session configuration, the returned
   * LightingEstimate will always return {@link com.google.ar.core.LightEstimate.State#NOT_VALID
   * LightEstimate.State#NOT_VALID} from {@link com.google.ar.core.LightEstimate#getState() }.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.LightEstimate getLightEstimate() {
    // Method contents removed.
  }

  /**
   * Returns the camera metadata for the current camera image, if available. Throws {@link
   * com.google.ar.core.exceptions.NotYetAvailableException NotYetAvailableException} when metadata
   * is not yet available due to sensors data not yet being available.
   *
   * <p>If the AR session was created for shared camera access, this method will throw {@link
   * java.lang.IllegalStateException IllegalStateException}. To retrieve image metadata in shared
   * camera mode, use {@link com.google.ar.core.SharedCamera#setCaptureCallback }, then use {@link
   * com.google.ar.core.Frame#getAndroidCameraTimestamp() Frame#getAndroidCameraTimestamp()} to
   * correlate the frame to metadata retrieved from {@link
   * android.hardware.camera2.CameraCaptureSession.CaptureCallback}.
   *
   * @throws com.google.ar.core.exceptions.NotYetAvailableException when metadata is not available
   *     because the sensors are not ready.
   * @throws com.google.ar.core.exceptions.DeadlineExceededException if this method was called on a
   *     frame that isn't the latest frame.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if too many metadata objects
   *     are currently held.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.ImageMetadata getImageMetadata()
      throws com.google.ar.core.exceptions.NotYetAvailableException {
    // Method contents removed.
  }

  /**
   * Acquires the current set of estimated 3d points attached to real-world geometry. {@link
   * com.google.ar.core.PointCloud#release } must be called after application is done using the
   * PointCloud object.
   *
   * <p>Note: This information is for visualization and debugging purposes only. Its characteristics
   * and format are subject to change in subsequent versions of the API.
   *
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if too many point clouds are
   *     acquired without being released.
   * @throws com.google.ar.core.exceptions.DeadlineExceededException if this method was called on a
   *     frame that isn't the latest frame.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.PointCloud acquirePointCloud() {
    // Method contents removed.
  }

  /**
   * Performs a ray cast from the user's device in the direction of the given location in the camera
   * view. Intersections with detected scene geometry are returned, sorted by distance from the
   * device; the nearest intersection is returned first.
   *
   * <p>Note: Significant geometric leeway is given when returning hit results. For example, a plane
   * hit may be generated if the ray came close, but did not actually hit within the plane extents
   * or plane bounds ({@link com.google.ar.core.Plane#isPoseInExtents(com.google.ar.core.Pose)
   * Plane#isPoseInExtents(Pose)} and {@link
   * com.google.ar.core.Plane#isPoseInPolygon(com.google.ar.core.Pose) Plane#isPoseInPolygon(Pose)}
   * can be used to determine these cases). A point (point cloud) hit is generated when a point is
   * roughly within one finger-width of the provided screen coordinates.
   *
   * <p>Note: When using {@link com.google.ar.core.Session.Feature#FRONT_CAMERA
   * Session.Feature#FRONT_CAMERA}, the returned hit result list will always be empty, as the camera
   * is not {@link com.google.ar.core.TrackingState#TRACKING TrackingState#TRACKING}. Hit testing
   * against tracked faces is not currently supported.
   *
   * <p>Note: In ARCore 1.24.0 or later on supported devices, if depth is enabled by calling {@link
   * com.google.ar.core.Config#setDepthMode } with the value {@link
   * com.google.ar.core.Config.DepthMode#AUTOMATIC Config.DepthMode#AUTOMATIC}, the returned list
   * includes {@link com.google.ar.core.DepthPoint DepthPoint} values sampled from the latest
   * computed depth image.
   *
   * @param xPx x coordinate in pixels
   * @param yPx y coordinate in pixels
   * @return an ordered list of intersections with scene geometry, nearest hit first
   */
  @androidx.annotation.NonNull
  public java.util.List<com.google.ar.core.HitResult> hitTest(float xPx, float yPx) {
    // Method contents removed.
  }

  /**
   * Similar to {@link #hitTest(float,float)}, but will take values from Android {@link
   * android.view.MotionEvent MotionEvent}. It is assumed that the {@code MotionEvent} is received
   * from the same view that was used as the size for {@link
   * com.google.ar.core.Session#setDisplayGeometry(int,int,int) Session#setDisplayGeometry(int, int,
   * int)}.
   *
   * <p>Note: this method does not consider the {@link android.view.MotionEvent#getAction() action}
   * of the {@code MotionEvent}. The caller must check for appropriate action, if needed, before
   * calling this method.
   *
   * <p>Note: When using {@link com.google.ar.core.Session.Feature#FRONT_CAMERA
   * Session.Feature#FRONT_CAMERA}, the returned hit result list will always be empty, as the camera
   * is not {@link com.google.ar.core.TrackingState#TRACKING TrackingState#TRACKING}. Hit testing
   * against tracked faces is not currently supported.
   *
   * @param motionEvent an event containing the x,y coordinates to hit test
   */
  @androidx.annotation.NonNull
  public java.util.List<com.google.ar.core.HitResult> hitTest(
      android.view.MotionEvent motionEvent) {
    // Method contents removed.
  }

  /**
   * Similar to {@link #hitTest(float,float)}, but takes an arbitrary ray in world space coordinates
   * instead of a screen-space point.
   *
   * <p>Note: When using {@link com.google.ar.core.Session.Feature#FRONT_CAMERA
   * Session.Feature#FRONT_CAMERA}, the returned hit result list will always be empty, as the camera
   * is not {@link com.google.ar.core.TrackingState#TRACKING TrackingState#TRACKING}. Hit testing
   * against tracked faces is not currently supported.
   *
   * @param origin3 an array of 3 floats containing ray origin in world space coordinates.
   * @param originOffset the offset into {@code origin3} array.
   * @param direction3 an array of 3 floats containing ray direction in world space coordinates.
   *     Does not have to be normalized.
   * @param directionOffset the offset into {@code direction3} array.
   * @return an ordered list of intersections with scene geometry, nearest hit first.
   */
  @androidx.annotation.NonNull
  public java.util.List<com.google.ar.core.HitResult> hitTest(
      float[] origin3, int originOffset, float[] direction3, int directionOffset) {
    // Method contents removed.
  }

  /**
   * Performs a ray cast that can return a result before ARCore establishes full tracking.
   *
   * <p>The pose and apparent scale of attached objects depends on the {@link
   * com.google.ar.core.InstantPlacementPoint InstantPlacementPoint} tracking method and the
   * provided approximateDistanceMeters. A discussion of the different tracking methods and the
   * effects of apparent object scale are described in {@link
   * com.google.ar.core.InstantPlacementPoint InstantPlacementPoint}.
   *
   * <p>This function will succeed only if {@link com.google.ar.core.Config.InstantPlacementMode
   * Config.InstantPlacementMode} is {@link
   * com.google.ar.core.Config.InstantPlacementMode#LOCAL_Y_UP } in the ARCore session
   * configuration, the ARCore session tracking state is {@link
   * com.google.ar.core.TrackingState#TRACKING TrackingState#TRACKING}, and there are sufficient
   * feature points to track the point in screen space.
   *
   * @param xPx x screen coordinate in pixels
   * @param yPx y screen coordinate in pixels
   * @param approximateDistanceMeters the distance at which to create an {@link
   *     com.google.ar.core.InstantPlacementPoint InstantPlacementPoint}. This is only used while
   *     the tracking method for the returned point is {@link
   *     com.google.ar.core.InstantPlacementPoint.TrackingMethod#SCREENSPACE_WITH_APPROXIMATE_DISTANCE
   *     InstantPlacementPoint.TrackingMethod#SCREENSPACE_WITH_APPROXIMATE_DISTANCE}.
   * @return if successful a list containing a single {@link com.google.ar.core.HitResult
   *     HitResult}, otherwise an empty list. The {@link com.google.ar.core.HitResult HitResult}
   *     will have a trackable of type {@link com.google.ar.core.InstantPlacementPoint
   *     InstantPlacementPoint}.
   */
  @androidx.annotation.NonNull
  public java.util.List<com.google.ar.core.HitResult> hitTestInstantPlacement(
      float xPx, float yPx, float approximateDistanceMeters) {
    // Method contents removed.
  }

  /**
   * Returns the anchors that were changed by the {@link com.google.ar.core.Session#update()
   * Session#update()} that returned this Frame.
   */
  @androidx.annotation.NonNull
  public java.util.Collection<com.google.ar.core.Anchor> getUpdatedAnchors() {
    // Method contents removed.
  }

  /**
   * Returns the trackables of a particular type that were changed by the {@link
   * com.google.ar.core.Session#update() Session#update()} that returned this Frame. {@code
   * filterType} may be {@link com.google.ar.core.Plane Plane.class} or {@link
   * com.google.ar.core.Point Point.class}, or {@code Trackable.class} to retrieve all changed
   * trackables.
   */
  @androidx.annotation.NonNull
  public <T extends com.google.ar.core.Trackable> java.util.Collection<T> getUpdatedTrackables(
      java.lang.Class<T> filterType) {
    // Method contents removed.
  }

  /**
   * Writes a data sample in the specified track. The samples recorded using this API are muxed into
   * the recorded MP4 dataset as an additional MP4 stream.
   *
   * <p>Multiple samples can be recorded to the same frame and will be played back together.
   *
   * <p>For smooth playback of the MP4 on video players and for future compatibility of the MP4
   * datasets with ARCore's playback of data tracks it is recommended that the samples are recorded
   * at a frequency no higher than 90kHz.
   *
   * <p>Additionally, if the samples are recorded at a frequency lower than 1Hz, empty (zero byte)
   * padding samples will be automatically recorded at approximately one second intervals to fill in
   * the gaps.
   *
   * <p>Recording samples introduces additional CPU and/or I/O overhead and may affect app
   * performance.
   *
   * @param trackId The {@link java.util.UUID UUID} of the track.
   * @param sample The {@link java.nio.ByteBuffer ByteBuffer} representation of the sample to be
   *     written.
   * @throws java.lang.IllegalStateException when either {@link
   *     com.google.ar.core.Session#getRecordingStatus() Session#getRecordingStatus()} is not
   *     currently {@link com.google.ar.core.RecordingStatus#OK RecordingStatus#OK} or the system is
   *     currently under excess load for images to be produced. The system should not be under such
   *     excess load for more than a few frames and an app should try to record the data again
   *     during the next frame.
   * @throws java.lang.IllegalArgumentException when any argument is invalid, e.g. null.
   * @throws com.google.ar.core.exceptions.DeadlineExceededException when the frame is not the
   *     current frame.
   */
  public void recordTrackData(java.util.UUID trackId, java.nio.ByteBuffer sample) {
    // Method contents removed.
  }

  /**
   * Retrieve all track data that was written to the specified track during the current frame. If
   * frames are skipped during playback, which can happen when the device is under load, played back
   * track data will be attached to a later frame in order.
   *
   * <p>Each call to {@link
   * com.google.ar.core.Frame#recordTrackData(java.util.UUID,java.nio.ByteBuffer)
   * Frame#recordTrackData(UUID, ByteBuffer)} at recording time will be returned as a separate
   * {@link com.google.ar.core.TrackData TrackData} entry in the collection.
   *
   * @throws com.google.ar.core.exceptions.DeadlineExceededException if this method is called on a
   *     frame that is not the current frame (most recent produced by {@link
   *     com.google.ar.core.Session#update() Session#update()}).
   */
  @androidx.annotation.NonNull
  public java.util.Collection<com.google.ar.core.TrackData> getUpdatedTrackData(
      java.util.UUID trackUuid) {
    // Method contents removed.
  }

  /**
   * Attempts to acquire an image from the camera that corresponds to the current frame. Depending
   * on device performance, can throw {@link com.google.ar.core.exceptions.NotYetAvailableException}
   * for several frames after session start, and for a few frames at a time while the session is
   * running.
   *
   * @return an <a href="https://developer.android.com/reference/android/media/Image">Android Image
   *     object</a> that contains the image data from the camera. The returned image object format
   *     is <a
   *     href="https://developer.android.com/ndk/reference/group/media#group___media_1gga9c3dace30485a0f28163a882a5d65a19aea9797f9b5db5d26a2055a43d8491890">{@code
   *     AIMAGE_FORMAT_YUV_420_888}</a>.
   * @throws java.lang.NullPointerException if {@code session} or {@code frame} is null.
   * @throws com.google.ar.core.exceptions.DeadlineExceededException if the input frame is not the
   *     current frame.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if the caller app has exceeded
   *     maximum number of images that it can hold without releasing.
   * @throws com.google.ar.core.exceptions.NotYetAvailableException if the image with the timestamp
   *     of the input frame did not become available within a bounded amount of time, or if the
   *     camera failed to produce the image.
   */
  @androidx.annotation.NonNull
  public android.media.Image acquireCameraImage()
      throws com.google.ar.core.exceptions.NotYetAvailableException {
    // Method contents removed.
  }

  /**
   * Attempts to acquire a depth <a
   * href="https://developer.android.com/reference/android/media/Image">Android Image object</a>
   * that corresponds to the current frame.
   *
   * <p>The depth image has a single 16-bit plane at index 0, stored in little-endian format. Each
   * pixel contains the distance in millimeters to the camera plane. Currently, the three most
   * significant bits are always set to 000. The remaining thirteen bits express values from 0 to
   * 8191, representing depth in millimeters. To extract distance from a depth map, see <a
   * href="https://developers.google.com/ar/develop/java/depth/developer-guide#extract-distance">the
   * Depth API developer guide</a>.
   *
   * <p>The actual size of the depth image depends on the device and its display aspect ratio. The
   * size of the depth image is typically around 160x120 pixels, with higher resolutions up to
   * 640x480 on some devices. These sizes may change in the future. The outputs of {@link
   * com.google.ar.core.Frame#acquireDepthImage }, {@link
   * com.google.ar.core.Frame#acquireRawDepthImage Frame#acquireRawDepthImage} and {@link
   * com.google.ar.core.Frame#acquireRawDepthConfidenceImage } will all have the exact same size.
   *
   * <p>Optimal depth accuracy occurs between 500 millimeters (50 centimeters) and 5000 millimeters
   * (5 meters) from the camera. Error increases quadratically as distance from the camera
   * increases.
   *
   * <p>Depth is estimated using data from the world-facing cameras, user motion, and hardware depth
   * sensors such as a time-of-flight sensor (or ToF sensor) if available. As the user moves their
   * device through the environment, 3D depth data is collected and cached which improves the
   * quality of subsequent depth images and reducing the error introduced by camera distance.
   *
   * <p>If an up-to-date depth image isn't ready for the current frame, the most recent depth image
   * available from an earlier frame will be returned instead. This is expected only to occur on
   * compute-constrained devices. An up-to-date depth image should typically become available again
   * within a few frames.
   *
   * <p>The image must be released via {@link android.media.Image#close} once it is no longer
   * needed.
   *
   * @return The depth image corresponding to the frame.
   * @throws com.google.ar.core.exceptions.NotYetAvailableException if the number of observed frames
   *     is not yet sufficient for depth estimation; or depth estimation was not possible due to
   *     poor lighting, camera occlusion, or insufficient motion observed.
   * @throws com.google.ar.core.exceptions.NotTrackingException if the {@link
   *     com.google.ar.core.Session Session} is not in the {@link
   *     com.google.ar.core.TrackingState#TRACKING TrackingState#TRACKING} state, which is required
   *     to acquire depth images.
   * @throws java.lang.IllegalStateException if a supported depth mode was not enabled in Session
   *     configuration.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if the caller app has exceeded
   *     maximum number of depth images that it can hold without releasing.
   * @throws com.google.ar.core.exceptions.DeadlineExceededException if the method is called on not
   *     the current frame.
   * @deprecated Please use {@link #acquireDepthImage16Bits} instead, which expands the depth range
   *     from 8191mm to 65535mm. This deprecated version may be slower than {@link
   *     #acquireDepthImage16Bits} due to the clearing of the top 3 bits per pixel.
   */
  @Deprecated
  @androidx.annotation.NonNull
  public android.media.Image acquireDepthImage()
      throws com.google.ar.core.exceptions.NotYetAvailableException {
    // Method contents removed.
  }

  /**
   * Attempts to acquire a depth <a
   * href="https://developer.android.com/reference/android/media/Image">Android Image object</a>
   * that corresponds to the current frame.
   *
   * <p>The depth image has format <a
   * href="https://developer.android.com/reference/android/hardware/HardwareBuffer#D_16">
   * HardwareBuffer.D_16</a>, which is a single 16-bit plane at index 0, stored in little-endian
   * format. Each pixel contains the distance in millimeters to the camera plane, with the
   * representable depth range between 0 millimeters and 65535 millimeters, or about 65 meters.
   *
   * <p>To extract distance from a depth map, see <a
   * href="https://developers.google.com/ar/develop/java/depth/developer-guide#extract-distance">the
   * Depth API developer guide</a>.
   *
   * <p>The actual size of the depth image depends on the device and its display aspect ratio. The
   * size of the depth image is typically around 160x120 pixels, with higher resolutions up to
   * 640x480 on some devices. These sizes may change in the future. The outputs of {@link
   * com.google.ar.core.Frame#acquireDepthImage16Bits }, {@link
   * com.google.ar.core.Frame#acquireRawDepthImage16Bits Frame#acquireRawDepthImage16Bits} and
   * {@link com.google.ar.core.Frame#acquireRawDepthConfidenceImage } will all have the exact same
   * size.
   *
   * <p>Optimal depth accuracy occurs between 500 millimeters (50 centimeters) and 15000 millimeters
   * (15 meters) from the camera, with depth reliably observed up to 25000 millimeters (25 meters).
   * Error increases quadratically as distance from the camera increases.
   *
   * <p>Depth is estimated using data from the world-facing cameras, user motion, and hardware depth
   * sensors such as a time-of-flight sensor (or ToF sensor) if available. As the user moves their
   * device through the environment, 3D depth data is collected and cached which improves the
   * quality of subsequent depth images and reducing the error introduced by camera distance. The
   * depth accuracy improves as the user moves more while looking at a desired location.
   *
   * <p>If an up-to-date depth image isn't ready for the current frame, the most recent depth image
   * available from an earlier frame will be returned instead. This is expected only to occur on
   * compute-constrained devices. An up-to-date depth image should typically become available again
   * within a few frames.
   *
   * <p>When the Geospatial API and the Depth API are enabled, output images from the Depth API will
   * include terrain and building geometry when in a location with VPS coverage. See the <a
   * href="https://developers.google.com/ar/develop/java/depth/geospatial-depth">Geospatial Depth
   * Developer Guide</a> for more information.
   *
   * <p>The image must be released via {@link android.media.Image#close} once it is no longer
   * needed.
   *
   * @return The depth image corresponding to the frame.
   * @throws com.google.ar.core.exceptions.NotYetAvailableException if the number of observed frames
   *     is not yet sufficient for depth estimation; or depth estimation was not possible due to
   *     poor lighting, camera occlusion, or insufficient motion observed.
   * @throws com.google.ar.core.exceptions.NotTrackingException if the {@link
   *     com.google.ar.core.Session Session} is not in the {@link
   *     com.google.ar.core.TrackingState#TRACKING TrackingState#TRACKING} state, which is required
   *     to acquire depth images.
   * @throws java.lang.IllegalStateException if a supported depth mode was not enabled in Session
   *     configuration.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if the caller app has exceeded
   *     maximum number of depth images that it can hold without releasing.
   * @throws com.google.ar.core.exceptions.DeadlineExceededException if the method is called on not
   *     the current frame.
   */
  @androidx.annotation.NonNull
  public android.media.Image acquireDepthImage16Bits()
      throws com.google.ar.core.exceptions.NotYetAvailableException {
    // Method contents removed.
  }

  /**
   * Attempts to acquire a "raw", mostly unfiltered, depth <a
   * href="https://developer.android.com/reference/android/media/Image">Android Image object</a>
   * that corresponds to the current frame.
   *
   * <p>The raw depth image is sparse and does not provide valid depth for all pixels. Pixels
   * without a valid depth estimate have a pixel value of 0 and a corresponding confidence value of
   * 0 (see {@link com.google.ar.core.Frame#acquireRawDepthConfidenceImage()
   * Frame#acquireRawDepthConfidenceImage()}).
   *
   * <p>The depth image has a single 16-bit plane at index 0, stored in little-endian format. Each
   * pixel contains the distance in millimeters to the camera plane. Currently, the three most
   * significant bits are always set to 000. The remaining thirteen bits express values from 0 to
   * 8191, representing depth in millimeters. To extract distance from a depth map, see <a
   * href="https://developers.google.com/ar/develop/java/depth/developer-guide#extract-distance">the
   * Depth API developer guide</a>.
   *
   * <p>The actual size of the depth image depends on the device and its display aspect ratio. The
   * size of the depth image is typically around 160x120 pixels, with higher resolutions up to
   * 640x480 on some devices. These sizes may change in the future. The outputs of {@link
   * com.google.ar.core.Frame#acquireDepthImage }, {@link
   * com.google.ar.core.Frame#acquireRawDepthImage Frame#acquireRawDepthImage} and {@link
   * com.google.ar.core.Frame#acquireRawDepthConfidenceImage } will all have the exact same size.
   *
   * <p>Optimal depth accuracy occurs between 500 millimeters (50 centimeters) and 5000 millimeters
   * (5 meters) from the camera. Error increases quadratically as distance from the camera
   * increases.
   *
   * <p>Depth is primarily estimated using data from the motion of world-facing cameras. As the user
   * moves their device through the environment, 3D depth data is collected and cached, improving
   * the quality of subsequent depth images and reducing the error introduced by camera distance.
   * Depth accuracy and robustness improves if the device has a hardware depth sensor, such as a
   * time-of-flight (ToF) camera.
   *
   * <p>Not every raw depth image contains a new depth estimate. Typically there are about 10
   * updates to the raw depth data per second. The depth images between those updates are a 3D
   * reprojection which transforms each depth pixel into a 3D point in space and renders those 3D
   * points into a new raw depth image based on the current camera pose. This effectively transforms
   * raw depth image data from a previous frame to account for device movement since the depth data
   * was calculated. For some applications it may be important to know whether the raw depth image
   * contains new depth data or is a 3D reprojection (for example, to reduce the runtime cost of 3D
   * reconstruction). To do that, compare the current raw depth image timestamp, obtained via {@link
   * android.media.Image#getTimestamp}, with the previously recorded raw depth image timestamp. If
   * they are different, the depth image contains new information.
   *
   * <p>The image must be released via {@link android.media.Image#close()} once it is no longer
   * needed.
   *
   * @return The raw depth image corresponding to the frame.
   * @throws com.google.ar.core.exceptions.NotYetAvailableException if the number of observed frames
   *     is not yet sufficient for depth estimation; or depth estimation was not possible due to
   *     poor lighting, camera occlusion, or insufficient motion observed.
   * @throws com.google.ar.core.exceptions.NotTrackingException if the {@link
   *     com.google.ar.core.Session Session} is not in the {@link
   *     com.google.ar.core.TrackingState#TRACKING TrackingState#TRACKING} state, which is required
   *     to acquire depth images.
   * @throws java.lang.IllegalStateException if a supported {@link
   *     com.google.ar.core.Config.DepthMode Config.DepthMode} was not enabled in Session
   *     configuration.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if the caller app has exceeded
   *     maximum number of depth images that it can hold without releasing.
   * @throws com.google.ar.core.exceptions.DeadlineExceededException if the method is called on not
   *     the current frame.
   * @deprecated Please use {@link #acquireRawDepthImage16Bits} instead, which expands the depth
   *     range from 8191mm to 65535mm. This deprecated version may be slower than {@link
   *     #acquireRawDepthImage16Bits} due to the clearing of the top 3 bits per pixel.
   */
  @Deprecated
  @androidx.annotation.NonNull
  public android.media.Image acquireRawDepthImage()
      throws com.google.ar.core.exceptions.NotYetAvailableException {
    // Method contents removed.
  }

  /**
   * Attempts to acquire a "raw", mostly unfiltered, depth <a
   * href="https://developer.android.com/reference/android/media/Image">Android Image object</a>
   * that corresponds to the current frame.
   *
   * <p>The raw depth image is sparse and does not provide valid depth for all pixels. Pixels
   * without a valid depth estimate have a pixel value of 0 and a corresponding confidence value of
   * 0 (see {@link com.google.ar.core.Frame#acquireRawDepthConfidenceImage()
   * Frame#acquireRawDepthConfidenceImage()}).
   *
   * <p>The depth image has format <a
   * href="https://developer.android.com/reference/android/hardware/HardwareBuffer#D_16">
   * HardwareBuffer.D_16</a>, which is a single 16-bit plane at index 0, stored in little-endian
   * format. Each pixel contains the distance in millimeters to the camera plane, with the
   * representable depth range between 0 millimeters and 65535 millimeters, or about 65 meters.
   *
   * <p>To extract distance from a depth map, see <a
   * href="https://developers.google.com/ar/develop/java/depth/developer-guide#extract_distance_from_a_depth_image">
   * the Depth API developer guide</a>.
   *
   * <p>The actual size of the depth image depends on the device and its display aspect ratio. The
   * size of the depth image is typically around 160x120 pixels, with higher resolutions up to
   * 640x480 on some devices. These sizes may change in the future. The outputs of {@link
   * com.google.ar.core.Frame#acquireDepthImage16Bits }, {@link
   * com.google.ar.core.Frame#acquireRawDepthImage16Bits Frame#acquireRawDepthImage16Bits} and
   * {@link com.google.ar.core.Frame#acquireRawDepthConfidenceImage } will all have the exact same
   * size.
   *
   * <p>Optimal depth accuracy occurs between 500 millimeters (50 centimeters) and 15000 millimeters
   * (15 meters) from the camera, with depth reliably observed up to 25000 millimeters (25 meters).
   * Error increases quadratically as distance from the camera increases.
   *
   * <p>Depth is primarily estimated using data from the motion of world-facing cameras. As the user
   * moves their device through the environment, 3D depth data is collected and cached, improving
   * the quality of subsequent depth images and reducing the error introduced by camera distance.
   * The depth accuracy improves as the user moves more while looking at a desired location. Depth
   * accuracy and robustness improves if the device has a hardware depth sensor, such as a
   * time-of-flight (ToF) camera.
   *
   * <p>Not every raw depth image contains a new depth estimate. Typically there are about 10
   * updates to the raw depth data per second. The depth images between those updates are a 3D
   * reprojection which transforms each depth pixel into a 3D point in space and renders those 3D
   * points into a new raw depth image based on the current camera pose. This effectively transforms
   * raw depth image data from a previous frame to account for device movement since the depth data
   * was calculated. For some applications it may be important to know whether the raw depth image
   * contains new depth data or is a 3D reprojection (for example, to reduce the runtime cost of 3D
   * reconstruction). To do that, compare the current raw depth image timestamp, obtained via {@link
   * android.media.Image#getTimestamp}, with the previously recorded raw depth image timestamp. If
   * they are different, the depth image contains new information.
   *
   * <p>When the Geospatial API and the Depth API are enabled, output images from the Depth API will
   * include terrain and building geometry when in a location with VPS coverage. See the <a
   * href="https://developers.google.com/ar/develop/java/depth/geospatial-depth">Geospatial Depth
   * Developer Guide</a> for more information.
   *
   * <p>The image must be released via {@link android.media.Image#close()} once it is no longer
   * needed.
   *
   * @return The raw depth image corresponding to the frame.
   * @throws com.google.ar.core.exceptions.NotYetAvailableException if the number of observed frames
   *     is not yet sufficient for depth estimation; or depth estimation was not possible due to
   *     poor lighting, camera occlusion, or insufficient motion observed.
   * @throws com.google.ar.core.exceptions.NotTrackingException if the {@link
   *     com.google.ar.core.Session Session} is not in the {@link
   *     com.google.ar.core.TrackingState#TRACKING TrackingState#TRACKING} state, which is required
   *     to acquire depth images.
   * @throws java.lang.IllegalStateException if a supported {@link
   *     com.google.ar.core.Config.DepthMode Config.DepthMode} was not enabled in Session
   *     configuration.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if the caller app has exceeded
   *     maximum number of depth images that it can hold without releasing.
   * @throws com.google.ar.core.exceptions.DeadlineExceededException if the method is called on not
   *     the current frame.
   */
  @androidx.annotation.NonNull
  public android.media.Image acquireRawDepthImage16Bits()
      throws com.google.ar.core.exceptions.NotYetAvailableException {
    // Method contents removed.
  }

  /**
   * Attempts to acquire the confidence <a
   * href="https://developer.android.com/reference/android/media/Image">Android Image object</a>
   * corresponding to the raw depth image of the current frame.
   *
   * <p>The image must be released via {@link android.media.Image#close} once it is no longer
   * needed.
   *
   * <p>Each pixel is an 8-bit unsigned integer representing the estimated confidence of the
   * corresponding pixel in the raw depth image. The confidence value is between 0 and 255,
   * inclusive, with 0 representing the lowest confidence and 255 representing the highest
   * confidence in the measured depth value. Pixels without a valid depth estimate have a confidence
   * value of 0 and a corresponding depth value of 0 (see {@link
   * com.google.ar.core.Frame#acquireRawDepthImage() Frame#acquireRawDepthImage()}).
   *
   * <p>The scaling of confidence values is linear and continuous within this range. Expect to see
   * confidence values represented across the full range of 0 to 255, with values increasing as
   * better observations are made of each location. If an application requires filtering out
   * low-confidence pixels, removing depth pixels below a confidence threshold of half confidence
   * (128) tends to work well.
   *
   * <p>The actual size of the depth image depends on the device and its display aspect ratio. The
   * size of the depth image is typically around 160x120 pixels, with higher resolutions up to
   * 640x480 on some devices. These sizes may change in the future. The outputs of {@link
   * com.google.ar.core.Frame#acquireDepthImage }, {@link
   * com.google.ar.core.Frame#acquireRawDepthImage Frame#acquireRawDepthImage} and {@link
   * com.google.ar.core.Frame#acquireRawDepthConfidenceImage } will all have the exact same size.
   *
   * @return The confidence image corresponding to the raw depth of the frame.
   * @throws com.google.ar.core.exceptions.NotYetAvailableException if the number of observed frames
   *     is not yet sufficient for depth estimation; or depth estimation was not possible due to
   *     poor lighting, camera occlusion, or insufficient motion observed.
   * @throws com.google.ar.core.exceptions.NotTrackingException if the {@link
   *     com.google.ar.core.Session Session} is not in the {@link
   *     com.google.ar.core.TrackingState#TRACKING TrackingState#TRACKING} state, which is required
   *     to acquire depth images.
   * @throws java.lang.IllegalStateException if a supported {@link
   *     com.google.ar.core.Config.DepthMode Config.DepthMode} was not enabled in Session
   *     configuration.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if the caller app has exceeded
   *     maximum number of depth images that it can hold without releasing.
   * @throws com.google.ar.core.exceptions.DeadlineExceededException if the method is called on not
   *     the current frame.
   */
  @androidx.annotation.NonNull
  public android.media.Image acquireRawDepthConfidenceImage()
      throws com.google.ar.core.exceptions.NotYetAvailableException {
    // Method contents removed.
  }

  /**
   * Attempts to acquire the semantic image corresponding to the current frame. Each pixel in the
   * image is an 8-bit unsigned integer representing a semantic class label: see {@link
   * com.google.ar.core.SemanticLabel SemanticLabel} for a list of pixel labels and the <a
   * href="https://developers.google.com/ar/develop/java/scene-semantics">Scene Semantics Developer
   * Guide</a> for more information.
   *
   * <p>The image must be released via {@link android.media.Image#close()} once it is no longer
   * needed.
   *
   * <p>In order to obtain a valid result from this function, you must set the session's {@link
   * com.google.ar.core.Config.SemanticMode Config.SemanticMode} to {@link
   * com.google.ar.core.Config.SemanticMode#ENABLED Config.SemanticMode#ENABLED}. Use {@link
   * com.google.ar.core.Session#isSemanticModeSupported(com.google.ar.core.Config.SemanticMode) } to
   * query for support for Scene Semantics.
   *
   * <p>The width of the semantic image is currently 256 pixels. The height of the image depends on
   * the device and will match its display aspect ratio.
   *
   * @throws com.google.ar.core.exceptions.NotYetAvailableException if no semantic image is
   *     available that corresponds to the frame.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if the caller app has exceeded
   *     maximum number of images that it can hold without releasing.
   * @throws com.google.ar.core.exceptions.DeadlineExceededException if the method is called on not
   *     the current frame.
   */
  @androidx.annotation.NonNull
  public android.media.Image acquireSemanticImage()
      throws com.google.ar.core.exceptions.NotYetAvailableException {
    // Method contents removed.
  }

  /**
   * Attempts to acquire the semantic confidence image corresponding to the current frame. Each
   * pixel is an 8-bit integer representing the estimated confidence of the corresponding pixel in
   * the semantic image. See the <a
   * href="https://developers.google.com/ar/develop/java/scene-semantics">Scene Semantics Developer
   * Guide</a> for more information.
   *
   * <p>The confidence value is between 0 and 255, inclusive, with 0 representing the lowest
   * confidence and 255 representing the highest confidence in the semantic class prediction from
   * {@link com.google.ar.core.Frame#acquireSemanticImage() Frame#acquireSemanticImage()}.
   *
   * <p>The image must be released via {@link android.media.Image#close()} once it is no longer
   * needed.
   *
   * <p>In order to obtain a valid result from this function, you must set the session's {@link
   * com.google.ar.core.Config.SemanticMode Config.SemanticMode} to {@link
   * com.google.ar.core.Config.SemanticMode#ENABLED Config.SemanticMode#ENABLED}. Use {@link
   * com.google.ar.core.Session#isSemanticModeSupported(com.google.ar.core.Config.SemanticMode) } to
   * query for support for Scene Semantics.
   *
   * <p>The size of the semantic confidence image is the same size as the image obtained by {@link
   * com.google.ar.core.Frame#acquireSemanticImage() }.
   *
   * @throws com.google.ar.core.exceptions.NotYetAvailableException if no semantic image is
   *     available that corresponds to the frame.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if the caller app has exceeded
   *     maximum number of images that it can hold without releasing.
   * @throws com.google.ar.core.exceptions.DeadlineExceededException if the method is called on not
   *     the current frame.
   */
  @androidx.annotation.NonNull
  public android.media.Image acquireSemanticConfidenceImage()
      throws com.google.ar.core.exceptions.NotYetAvailableException {
    // Method contents removed.
  }

  /**
   * Retrieves the percentage of pixels in the most recent semantics image that are {@code
   * queryLabel}.
   *
   * <p>Queries the semantic image provided by {@link
   * com.google.ar.core.Frame#acquireSemanticImage() Frame#acquireSemanticImage()} for pixels
   * labelled by {@code queryLabel}. This call is more efficient than retrieving the {@link
   * android.media.Image} and performing a pixel-wise search for the detected label.
   *
   * @throws com.google.ar.core.exceptions.NotYetAvailableException if no semantic image has been
   *     detected since the start of the session.
   */
  @androidx.annotation.NonNull
  public float getSemanticLabelFraction(com.google.ar.core.SemanticLabel queryLabel)
      throws com.google.ar.core.exceptions.NotYetAvailableException {
    // Method contents removed.
  }

  /**
   * Transforms a list of 2D coordinates from one 2D coordinate system to another 2D coordinate
   * system.
   *
   * <p>For Android view coordinates ({@link com.google.ar.core.Coordinates2d#VIEW
   * Coordinates2d#VIEW}, {@link com.google.ar.core.Coordinates2d#VIEW_NORMALIZED }), the view
   * information is taken from the most recent call to {@link
   * com.google.ar.core.Session#setDisplayGeometry(int,int,int) Session#setDisplayGeometry(int, int,
   * int)}.
   *
   * <p>Must be called on the most recently obtained {@link com.google.ar.core.Frame Frame} object.
   * If this function is called on an older frame, a log message will be printed and {@code
   * outputVertices2d} will remain unchanged.
   *
   * <p>Some examples of useful conversions:
   *
   * <ul>
   *   <li>To transform from {@code [0,1]} range to screen-quad coordinates for rendering: {@link
   *       com.google.ar.core.Coordinates2d#VIEW_NORMALIZED } -> {@link
   *       com.google.ar.core.Coordinates2d#TEXTURE_NORMALIZED Coordinates2d#TEXTURE_NORMALIZED}
   *   <li>To transform from {@code [-1,1]} range to screen-quad coordinates for rendering: {@link
   *       com.google.ar.core.Coordinates2d#OPENGL_NORMALIZED_DEVICE_COORDINATES } -> {@link
   *       com.google.ar.core.Coordinates2d#TEXTURE_NORMALIZED }
   *   <li>To transform a point found by a computer vision algorithm in a CPU image into a point on
   *       the screen that can be used to place an Android View (e.g. Button) at that location:
   *       {@link com.google.ar.core.Coordinates2d#IMAGE_PIXELS Coordinates2d#IMAGE_PIXELS} ->
   *       {@link com.google.ar.core.Coordinates2d#VIEW Coordinates2d#VIEW}
   *   <li>To transform a point found by a computer vision algorithm in a CPU image into a point to
   *       be rendered using GL in clip-space ({@code [-1,1]} range): {@link
   *       com.google.ar.core.Coordinates2d#IMAGE_PIXELS } -> {@link
   *       com.google.ar.core.Coordinates2d#OPENGL_NORMALIZED_DEVICE_COORDINATES
   *       Coordinates2d#OPENGL_NORMALIZED_DEVICE_COORDINATES}
   * </ul>
   *
   * <p>Read-only array-backed buffers are not supported by {@code inputVertices2d} for performance
   * reasons.
   *
   * <p>If {@code inputCoordinates} is same as {@code outputCoordinates}, the input vertices will be
   * copied to the output vertices unmodified.
   *
   * @param inputCoordinates The coordinate system used by {@code inputVertices2d}.
   * @param inputVertices2d Input 2D vertices to transform.
   * @param outputCoordinates The coordinate system to convert to.
   * @param outputVertices2d Buffer to put the transformed 2D vertices into.
   * @throws java.lang.IllegalArgumentException If the buffer sizes don't match, or the input/output
   *     buffers have odd size.
   * @throws java.nio.ReadOnlyBufferException If this buffer is a read-only array backed buffer.
   */
  public void transformCoordinates2d(
      com.google.ar.core.Coordinates2d inputCoordinates,
      java.nio.FloatBuffer inputVertices2d,
      com.google.ar.core.Coordinates2d outputCoordinates,
      java.nio.FloatBuffer outputVertices2d) {
    // Method contents removed.
  }

  /**
   * Transforms a list of 2D coordinates from one 2D coordinate system to another 2D coordinate
   * system.
   *
   * <p>Same as {@link
   * #transformCoordinates2d(com.google.ar.core.Coordinates2d,java.nio.FloatBuffer,com.google.ar.core.Coordinates2d,java.nio.FloatBuffer)},
   * but taking float arrays.
   *
   * @param inputCoordinates The coordinate system used by {@code inputVertices2d}.
   * @param inputVertices2d Input 2D vertices to transform.
   * @param outputCoordinates The coordinate system to convert to.
   * @param outputVertices2d Buffer to put the transformed 2D vertices into.
   */
  public void transformCoordinates2d(
      com.google.ar.core.Coordinates2d inputCoordinates,
      float[] inputVertices2d,
      com.google.ar.core.Coordinates2d outputCoordinates,
      float[] outputVertices2d) {
    // Method contents removed.
  }

  /**
   * Transforms a list of 2D coordinates from one 2D coordinate system to 3D coordinate space. See
   * the <a
   * href="https://developers.google.com/ar/develop/java/electronic-image-stabilization">Electronic
   * Image Stabilization developer guide</a> for more information.
   *
   * <p>The view information is taken from the most recent call to {@link
   * com.google.ar.core.Session#setDisplayGeometry(int,int,int) }.
   *
   * <p>If Electronic Image Stabilization is off, the device coordinates return {@code (-1, -1, 0)
   * -> (1, 1, 0)} and texture coordinates return the same coordinates as {@link
   * com.google.ar.core.Frame#transformCoordinates2d(com.google.ar.core.Coordinates2d,float[],com.google.ar.core.Coordinates2d,float[])
   * }} with the Z component set to 1.0f.
   *
   * <p>In order to use EIS, your app should use EIS compensated screen coordinates and camera
   * texture coordinates to pass on to shaders. Use the 2D NDC space coordinates as input to obtain
   * EIS compensated 3D screen coordinates and matching camera texture coordinates.
   *
   * @param inputCoordinates The coordinate system used by {@code inputVertices2d}.
   * @param inputVertices2d Input 2D vertices to transform.
   * @param outputCoordinates The 3D coordinate system to output in {@code outputVertices3d}.
   * @param outputVertices3d Buffer to put the transformed 3D vertices into.
   */
  public void transformCoordinates3d(
      com.google.ar.core.Coordinates2d inputCoordinates,
      java.nio.FloatBuffer inputVertices2d,
      com.google.ar.core.Coordinates3d outputCoordinates,
      java.nio.FloatBuffer outputVertices3d) {
    // Method contents removed.
  }

  /**
   * Transforms a list of 2D coordinates from one 2D coordinate system to another 3D coordinate
   * system.
   *
   * <p>Same as {@link
   * #transformCoordinates3d(com.google.ar.core.Coordinates2d,java.nio.FloatBuffer,com.google.ar.core.Coordinates3d,java.nio.FloatBuffer)},
   * but taking float arrays.
   *
   * @param inputCoordinates The coordinate system used by {@code inputVertices2d}.
   * @param inputVertices2d Input 2D vertices to transform.
   * @param outputCoordinates The 3D coordinate system to output in {@code outputVertices3d}.
   * @param outputVertices3d Buffer to put the transformed 3D vertices into.
   */
  public void transformCoordinates3d(
      com.google.ar.core.Coordinates2d inputCoordinates,
      float[] inputVertices2d,
      com.google.ar.core.Coordinates3d outputCoordinates,
      float[] outputVertices3d) {
    // Method contents removed.
  }

  /**
   * Returns the OpenGL ES camera texture name (id) associated with this frame. This is guaranteed
   * to be one of the texture names previously set via {@link
   * com.google.ar.core.Session#setCameraTextureNames Session#setCameraTextureNames} or {@link
   * com.google.ar.core.Session#setCameraTextureName Session#setCameraTextureName}. Texture names
   * (ids) are returned in a round robin fashion in sequential frames.
   *
   * @return the OpenGL ES texture name (id).
   */
  public int getCameraTextureName() {
    // Method contents removed.
  }

  /**
   * Gets the <a
   * href="https://developer.android.com/reference/android/hardware/HardwareBuffer">HardwareBuffer
   * </a> for this frame. See <a href="https://developers.google.com/ar/develop/java/vulkan">Vulkan
   * Rendering developer guide</a> for more information.
   *
   * <p>Should only be called when a configuration is active that uses {@link
   * com.google.ar.core.Config.TextureUpdateMode#EXPOSE_HARDWARE_BUFFER }.
   *
   * @throws com.google.ar.core.exceptions.NotYetAvailableException when the camera failed to
   *     produce the image.
   * @throws com.google.ar.core.exceptions.DeadlineExceededException if this method was called on a
   *     frame that isn't the latest frame.
   */
  public android.hardware.HardwareBuffer getHardwareBuffer()
      throws java.lang.IllegalStateException,
          com.google.ar.core.exceptions.NotYetAvailableException {
    // Method contents removed.
  }
}
