/*
 * Copyright 2023 Google LLC
 *
 * Licensed for use under "ARCore Additional Terms of Service". You may obtain
 * a copy of the license at https://developers.google.com/ar/develop/terms.
 */

package com.google.ar.core;

/**
 * Futures represent the eventual completion of an asynchronous operation. A future has one of three
 * {@link com.google.ar.core.FutureState FutureState}s, which can be obtained with {@link
 * com.google.ar.core.Future#getState Future#getState}:
 *
 * <ul>
 *   <li>{@link com.google.ar.core.FutureState#PENDING FutureState#PENDING} - The operation is still
 *       pending. The result of the operation isn't available yet and any associated callback hasn't
 *       yet been invoked.
 *   <li>{@link com.google.ar.core.FutureState#DONE FutureState#DONE} - The operation is complete,
 *       and a result is available.
 *   <li>{@link com.google.ar.core.FutureState#CANCELLED FutureState#CANCELLED} - The operation has
 *       been cancelled.
 * </ul>
 *
 * <p>A {@link com.google.ar.core.Future Future} starts in the {@link
 * com.google.ar.core.FutureState#PENDING FutureState#PENDING} state and transitions to {@link
 * com.google.ar.core.FutureState#DONE } upon completion. If the future is cancelled using {@link
 * com.google.ar.core.Future#cancel Future#cancel}, then its state may become {@link
 * com.google.ar.core.FutureState#CANCELLED FutureState#CANCELLED} (see <a
 * href="#future_cancellation">cancelling a future</a> for caveats).
 *
 * <h3 id="future_results">Obtaining results from a Future</h3>
 *
 * <p>There are two ways of obtaining results from a {@link com.google.ar.core.Future Future}:
 *
 * <h4 id="future_polling">Polling a Future</h4>
 *
 * <p>When the {@link com.google.ar.core.Future Future} is created, its {@link
 * com.google.ar.core.FutureState FutureState} is set to {@link
 * com.google.ar.core.FutureState#PENDING }. You may poll the future using {@link
 * com.google.ar.core.Future#getState Future#getState} to query the state of the asynchronous
 * operation. When its state is {@link com.google.ar.core.FutureState#DONE FutureState#DONE}, you
 * can obtain the operation's result.
 *
 * <h4 id="future_callback">Using a callback to obtain Future results</h4>
 *
 * <p>The operation's result can be reported via a {@code callback}. When providing a callback,
 * ARCore will invoke the given function when the operation is complete, unless the future has been
 * cancelled using {@link com.google.ar.core.Future#cancel Future#cancel}. This callback will be
 * invoked on the <a
 * href="https://developer.android.com/guide/components/processes-and-threads#Threads">main
 * thread</a>.
 *
 * <h4 id="future_cancellation">Cancelling a Future</h4>
 *
 * <p>You can try to cancel a {@link com.google.ar.core.Future Future} by calling {@link
 * com.google.ar.core.Future#cancel Future#cancel}. Due to multi-threading, it is possible that the
 * cancel operation is not successful. The return value indicates if the cancellation was
 * successful.
 *
 * <p>If the cancellation is successful, then any <a href="#future_callback">associated callback</a>
 * will never be called.
 */
@SuppressWarnings({"unchecked", "deprecation", "all"})
public interface Future {

  /** Get the current state of the future. */
  @androidx.annotation.NonNull
  public com.google.ar.core.FutureState getState();

  /**
   * Tries to cancel the execution of this operation. If the operation was cancelled by this
   * invocation, this method returns {@code true} and the associated callback (if any) will never be
   * invoked.
   */
  public boolean cancel();
}
