/*
 * Copyright 2018 Google LLC
 *
 * Licensed for use under "ARCore Additional Terms of Service". You may obtain
 * a copy of the license at https://developers.google.com/ar/develop/terms.
 */

package com.google.ar.core;

/**
 * Describes a specific location, elevation, and orientation relative to Earth. It is comprised of:
 *
 * <ul>
 *   <li>Latitude and longitude are specified in degrees, with positive values being north of the
 *       equator and east of the prime meridian as defined by the <a
 *       href="https://en.wikipedia.org/wiki/World_Geodetic_System">WGS84 specification</a>.
 *   <li>Altitude is specified in meters above the WGS84 ellipsoid, which is roughly equivalent to
 *       meters above sea level.
 *   <li>Orientation approximates the direction the user is facing in the EUS coordinate system. The
 *       EUS coordinate system has X+ pointing east, Y+ pointing up, and Z+ pointing south.
 *   <li>Accuracy of the latitude, longitude, altitude, and orientation are available as numeric
 *       confidence intervals where a large value (large interval) means low confidence and small
 *       value (small interval) means high confidence.
 * </ul>
 *
 * An {@link com.google.ar.core.GeospatialPose GeospatialPose} can be retrieved from {@link
 * com.google.ar.core.Earth#getCameraGeospatialPose() Earth#getCameraGeospatialPose()} and by
 * converting a {@link com.google.ar.core.Pose Pose} using {@link
 * com.google.ar.core.Earth#getGeospatialPose(com.google.ar.core.Pose)
 * Earth#getGeospatialPose(Pose)}.
 */
@SuppressWarnings({"unchecked", "deprecation", "all"})
@com.google.ar.core.annotations.UsedByNative("session_jni_wrapper.cc")
public class GeospatialPose {

  GeospatialPose() {
    // Method contents removed.
  }

  /** @hide */
  protected void finalize() throws java.lang.Throwable {
    // Method contents removed.
  }

  /**
   * Returns the {@link com.google.ar.core.GeospatialPose GeospatialPose}'s latitude in degrees.
   * Positive values are north of the equator as defined by the <a
   * href="https://en.wikipedia.org/wiki/World_Geodetic_System">WGS84 specification</a>.
   */
  public double getLatitude() {
    // Method contents removed.
  }

  /**
   * Returns the {@link com.google.ar.core.GeospatialPose GeospatialPose}'s longitude in degrees.
   * Positive values are east of the prime meridian as defined by the <a
   * href="https://en.wikipedia.org/wiki/World_Geodetic_System">WGS84 specification</a>.
   */
  public double getLongitude() {
    // Method contents removed.
  }

  /**
   * Returns the {@link com.google.ar.core.GeospatialPose GeospatialPose}'s altitude in meters as
   * elevation above the <a href="https://en.wikipedia.org/wiki/World_Geodetic_System">WGS84
   * ellipsoid</a>.
   */
  public double getAltitude() {
    // Method contents removed.
  }

  /**
   * Returns the {@link com.google.ar.core.GeospatialPose GeospatialPose}'s heading. Heading is
   * specified in degrees clockwise from true north and approximates the direction the device is
   * facing. The value returned when facing north is 0°, when facing east is 90°, when facing south
   * is +/-180°, and when facing west is -90°.
   *
   * <p>This function will return heading values for {@link com.google.ar.core.GeospatialPose
   * GeospatialPose} from {@link com.google.ar.core.Earth#getCameraGeospatialPose() } and returns 0
   * for all other {@link com.google.ar.core.GeospatialPose GeospatialPose} objects. You may use
   * {@link com.google.ar.core.GeospatialPose#getEastUpSouthQuaternion()
   * GeospatialPose#getEastUpSouthQuaternion()} instead, which provides a more detailed orientation
   * in 3D space.
   *
   * <p>The heading approximation is based on the rotation of the device in its current orientation
   * mode (i.e. portrait or landscape) and pitch. For example, when the device is held vertically or
   * upright, the heading is based on the camera optical axis. If the device is held horizontally,
   * looking downwards, the heading is based on the top of the device, with respect to the
   * orientation mode.
   *
   * <p>Note: Heading is currently only supported in the device's default orientation mode, which is
   * portrait mode for most supported devices.
   *
   * @return the heading component of this pose's orientation in [-180.0, 180.0] degree range.
   * @deprecated This function has been deprecated in favor of {@link #getEastUpSouthQuaternion()}
   *     which provides orientation values in 3D space. To determine a value analogous to the
   *     heading value, calculate the yaw, pitch, and roll values from {@link
   *     #getEastUpSouthQuaternion()}. When the device is pointing downwards, i.e. perpendicular to
   *     the ground, heading is analoguous to roll, and when the device is upright in the device's
   *     default orientation mode, heading is analogous to yaw.
   */
  @Deprecated
  public double getHeading() {
    // Method contents removed.
  }

  /**
   * Returns the {@link com.google.ar.core.GeospatialPose GeospatialPose}'s estimated heading
   * accuracy.
   *
   * <p>This function will return values for {@link com.google.ar.core.GeospatialPose
   * GeospatialPose} from {@link com.google.ar.core.Earth#getCameraGeospatialPose() } and returns 0
   * for all other {@link com.google.ar.core.GeospatialPose GeospatialPose} objects.
   *
   * <p>We define heading accuracy as the radius of the 68th percentile confidence level around
   * {@link com.google.ar.core.GeospatialPose#getHeading() GeospatialPose#getHeading()}. In other
   * words, there is a 68% probability that the true heading is within {@link
   * com.google.ar.core.GeospatialPose#getHeadingAccuracy() GeospatialPose#getHeadingAccuracy()} of
   * this {@link com.google.ar.core.GeospatialPose GeospatialPose}'s heading. Larger values indicate
   * lower accuracy.
   *
   * <p>For example, if the estimated heading is 60°, and the heading accuracy is 10°, then there is
   * an estimated 68% probability of the true heading being between 50° and 70°.
   *
   * @return the accuracy of the heading confidence in degrees.
   * @deprecated This function has deprecated in favor of {@link #getOrientationYawAccuracy()},
   *     which provides the accuracy analogous to the heading accuracy when the device is held
   *     upright in the default orientation mode.
   */
  @Deprecated
  public double getHeadingAccuracy() {
    // Method contents removed.
  }

  /**
   * Returns the {@link com.google.ar.core.GeospatialPose GeospatialPose}'s estimated orientation
   * yaw angle accuracy. Yaw rotation is the angle between the pose's compass direction and north,
   * and can be determined from {@link #getEastUpSouthQuaternion()}.
   *
   * <p>We define yaw accuracy as the estimated radius of the 68th percentile confidence level
   * around yaw angles from {@link com.google.ar.core.GeospatialPose#getEastUpSouthQuaternion()
   * GeospatialPose#getEastUpSouthQuaternion()}. In other words, there is a 68% probability that the
   * true yaw angle is within {@link com.google.ar.core.GeospatialPose#getOrientationYawAccuracy() }
   * of this {@link com.google.ar.core.GeospatialPose GeospatialPose}'s orientation. Larger values
   * indicate lower accuracy.
   *
   * <p>For example, if the estimated yaw angle is 60°, and the orientation yaw accuracy is 10°,
   * then there is an estimated 68% probability of the true yaw angle being between 50° and 70°.
   *
   * @return the accuracy of the orientation yaw confidence in degrees.
   */
  public double getOrientationYawAccuracy() {
    // Method contents removed.
  }

  /**
   * Returns the {@link com.google.ar.core.GeospatialPose GeospatialPose}'s estimated horizontal
   * accuracy in meters with respect to latitude and longitude.
   *
   * <p>We define horizontal accuracy as the radius of the 68th percentile confidence level around
   * the estimated horizontal location. In other words, if you draw a circle centered at this {@link
   * com.google.ar.core.GeospatialPose GeospatialPose}'s latitude and longitude, and with a radius
   * equal to the horizontal accuracy, then there is a 68% probability that the true location is
   * inside the circle. Larger numbers indicate lower accuracy.
   *
   * <p>For example, if the latitude is 10°, longitude is 10°, and the returned value is 15, then
   * there is a 68% probability that the true location is within 15 meters of the (10°, 10°)
   * latitude/longitude coordinate.
   *
   * @return the estimated horizontal accuracy with respect to latitude and longitude, radial, in
   *     meters.
   */
  public double getHorizontalAccuracy() {
    // Method contents removed.
  }

  /**
   * Returns the {@link com.google.ar.core.GeospatialPose GeospatialPose}'s estimated altitude
   * accuracy in meters.
   *
   * <p>We define vertical accuracy as the radius of the 68th percentile confidence level around the
   * estimated altitude. In other words, there is a 68% probability that the true altitude is within
   * the output value (in meters) of this {@link com.google.ar.core.GeospatialPose GeospatialPose}'s
   * altitude (above or below). Larger numbers indicate lower accuracy.
   *
   * <p>For example, if this {@link com.google.ar.core.GeospatialPose GeospatialPose}'s altitude is
   * 100 meters, and the output value is 20 meters, there is a 68% chance that the true altitude is
   * within 20 meters of 100 meters.
   *
   * @return the estimated vertical accuracy in meters.
   */
  public double getVerticalAccuracy() {
    // Method contents removed.
  }

  /**
   * Extracts the orientation from a Geospatial pose. The output quaternion represents the rotation
   * matrix transforming a vector from the target to the east-up-south (EUS) coordinate system (i.e.
   * X+ points east, Y+ points up, and Z+ points south). Values are written in the order {x, y, z,
   * w}.
   */
  @androidx.annotation.NonNull
  public float[] getEastUpSouthQuaternion() {
    // Method contents removed.
  }
}
