/*
 * Copyright 2020 Google LLC
 *
 * Licensed for use under "ARCore Additional Terms of Service". You may obtain
 * a copy of the license at https://developers.google.com/ar/develop/terms.
 */

package com.google.ar.core;

/**
 * Trackable Instant Placement point returned by {@link
 * com.google.ar.core.Frame#hitTestInstantPlacement(float,float,float)
 * Frame#hitTestInstantPlacement(float, float, float)}.
 *
 * <p>If ARCore has an accurate 3D pose for the {@link com.google.ar.core.InstantPlacementPoint
 * InstantPlacementPoint} returned by {@link
 * com.google.ar.core.Frame#hitTestInstantPlacement(float,float,float) } it will start with tracking
 * method {@link com.google.ar.core.InstantPlacementPoint.TrackingMethod#FULL_TRACKING }. Otherwise,
 * it will start with tracking method {@link
 * com.google.ar.core.InstantPlacementPoint.TrackingMethod#SCREENSPACE_WITH_APPROXIMATE_DISTANCE },
 * and will transition to {@link
 * com.google.ar.core.InstantPlacementPoint.TrackingMethod#FULL_TRACKING } once ARCore has an
 * accurate 3D pose. Once the tracking method is {@link
 * com.google.ar.core.InstantPlacementPoint.TrackingMethod#FULL_TRACKING
 * TrackingMethod#FULL_TRACKING} it will not revert to {@link
 * com.google.ar.core.InstantPlacementPoint.TrackingMethod#SCREENSPACE_WITH_APPROXIMATE_DISTANCE }.
 *
 * <p>When the tracking method changes from {@link
 * com.google.ar.core.InstantPlacementPoint.TrackingMethod#SCREENSPACE_WITH_APPROXIMATE_DISTANCE }
 * in one frame to {@link com.google.ar.core.InstantPlacementPoint.TrackingMethod#FULL_TRACKING } in
 * the next frame, the pose will jump from its initial location based on the provided approximate
 * distance to a new location at an accurate distance.
 *
 * <p>This instantaneous change in pose will change the apparent scale of any objects that are
 * anchored to the {@link com.google.ar.core.InstantPlacementPoint InstantPlacementPoint}. That is,
 * an object will suddenly appear larger or smaller than it was in the previous frame.
 *
 * <p>To avoid the visual jump due to the sudden change in apparent object scale, use the following
 * procedure:
 *
 * <ol>
 *   <li>Keep track of the pose and tracking method of the {@link
 *       com.google.ar.core.InstantPlacementPoint InstantPlacementPoint} in each frame.
 *   <li>Wait for the tracking method to change to {@link
 *       com.google.ar.core.InstantPlacementPoint.TrackingMethod#FULL_TRACKING
 *       TrackingMethod#FULL_TRACKING}.
 *   <li>Use the pose from the previous frame and the pose in the current frame to determine the
 *       object's distance to the device in both frames.
 *   <li>Calculate the apparent change in scale due to the change in distance from the camera.
 *   <li>Adjust the scale of the object to counteract the perceived change in scale, so that
 *       visually the object does not appear to change in size.
 *   <li>Optionally, smoothly adjust the scale of the object back to its original value over several
 *       frames.
 * </ol>
 *
 * <p>The following snippet shows how to use Instant Placement to place one object. The wrapper
 * below shows how to use steps 1 through 5 to avoid the visual jump.
 *
 * <pre class="prettyprint lang-java">{@code class ArPlacementActivity {
 *   private Session session = null;
 *   private placementIsDone = false;
 *
 *   public void createSession() {
 *     session = new Session(this);
 *     Config config = new Config(session);
 *     config.setInstantPlacementMode(InstantPlacementMode.LOCAL_Y_UP);
 *     session.configure(config);
 *   }
 *
 *   public void onDrawFrame(GL10 gl) {
 *     Frame frame = session.update();
 *
 *     // Place an object on tap.
 *     if (!placementIsDone && didUserTap()) {
 *       // Use estimated distance from the user's device to the real world, based
 *       // on how we expect the app to be used by users.
 *       float approximateDistanceMeters = 1.0f;
 *
 *       List<HitResult> results =
 *          frame.hitTestInstantPlacement(tapX, tapY, approximateDistanceMeters);
 *       if (!results.isEmpty()) {
 *         HitResult hit = results.get(0);
 *         InstantPlacementPoint point = (InstantPlacementPoint) hit.getTrackable();
 *         anchor = point.createAnchor(point.getPose());
 *         placementIsDone = true;
 *         disableInstantPlacement();
 *       }
 *     }
 *   }
 *
 *   private void disableInstantPlacement() {
 *     // When the placement is done, you can disable Instant Placement to save computing.
 *     Config config = new Config(session);
 *     config.setInstantPlacementMode(InstantPlacementMode.DISABLED);
 *     session.configure(config);
 *   }
 * }}</pre>
 *
 * <p>This wrapper of InstantPlacementPoint can be used to offset the apparent scale shift that
 * occurs when the tracking method changes from {@link
 * com.google.ar.core.InstantPlacementPoint.TrackingMethod#SCREENSPACE_WITH_APPROXIMATE_DISTANCE }
 * to {@link com.google.ar.core.InstantPlacementPoint.TrackingMethod#FULL_TRACKING
 * TrackingMethod#FULL_TRACKING}. For AR experiences where a sudden apparent scale change is
 * acceptable, no wrapper is required.
 *
 * <pre class="prettyprint lang-java">{@code class WrappedInstantPlacement {
 *   public InstantPlacementPoint point;
 *   public TrackingMethod previousTrackingMethod;
 *   public float previousDistanceToCamera;
 *   public float scaleFactor = 1.0f;
 *
 *   public WrappedInstantPlacement(
 *       InstantPlacementPoint point,
 *       TrackingMethod previousTrackingMethod,
 *       float previousDistanceToCamera) {
 *     this.point = point;
 *     this.previousTrackingMethod = previousTrackingMethod;
 *     this.previousDistanceToCamera = previousDistanceToCamera;
 *   }
 * }
 *
 * ArrayList<WrappedInstantPlacement> wrappedPoints =
 *     new ArrayList<WrappedInstantPlacement>();
 *
 * public void onDrawFrame(GL10 gl) {
 *   Frame frame = session.update();
 *
 *   // Place an object on tap.
 *   if (didUserTap()) {
 *     // Use estimated distance from the user's device to the real world, based
 *     // on how we expect the app to be used by users.
 *     float approximateDistanceMeters = 2.0f;
 *
 *     // Returns a single result if the hit test was successful.
 *     List<HitResult> results = frame.hitTestInstantPlacement(tapX, tapY,
 *         approximateDistanceMeters);
 *      for (HitResult hit : results) {
 *        InstantPlacementPoint point = (InstantPlacementPoint) hit.getTrackable();
 *        wrappedPoints.add(
 *            new WrappedInstantPlacement(point, point.getTrackingMethod(),
 *                                        distance(camera.getPose(), point.getPose())));
 *      }
 *   }
 *
 *   for (WrappedInstantPlacement wrappedPoint : wrappedPoints) {
 *     InstantPlacementPoint point = wrappedPoint.point;
 *     if (point.getTrackingState() == TrackingState.STOPPED) {
 *       wrappedPoints.remove(wrappedPoint);
 *       continue;
 *     }
 *     if (point.getTrackingState() == TrackingState.PAUSED) {
 *       continue;
 *     }
 *
 *     if (point.getTrackingMethod() == TrackingMethod.SCREENSPACE_WITH_APPROXIMATE_DISTANCE) {
 *       // Continue to use estimated depth and pose. Record distance to camera for use in
 *       // the next frame if the transition to full tracking happens.
 *       wrappedPoint.previousDistanceToCamera =
 *           distance(point.getPose(), camera.getPose());
 *     } else if (point.getTrackingMethod() == TrackingMethod.FULL_TRACKING) {
 *       if (wrappedPoint.previousPoseType ==
 *           TrackingMethod.SCREENSPACE_WITH_APPROXIMATE_DISTANCE) {
 *        // Transition from estimated pose to real pose. Adjust object scale to
 *        // counter-act apparent change is size due to pose jump.
 *        wrappedPoint.scaleFactor = distance(point.getPose(), camera.getPose()) /
 *            wrappedPoint.previousDistanceToCamera);
 *        // TODO: Apply the scale factor to the model.
 *        // ...
 *        previousPoseType = TrackingMethod.FULL_TRACKING;
 *       }
 *     }
 *   }
 * }}</pre>
 */
@SuppressWarnings({"unchecked", "deprecation", "all"})
public class InstantPlacementPoint implements com.google.ar.core.Trackable {

  InstantPlacementPoint() {
    // Method contents removed.
  }

  /**
   * Returns the current {@link com.google.ar.core.InstantPlacementPoint.TrackingMethod
   * TrackingMethod} for this {@link com.google.ar.core.InstantPlacementPoint
   * InstantPlacementPoint}.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.InstantPlacementPoint.TrackingMethod getTrackingMethod() {
    // Method contents removed.
  }

  /**
   * Gets the pose of the Instant Placement point. Use {@link #getTrackingMethod()} to determine
   * whether depth and scale are accurate or are based on the estimated depth provided to {@link
   * com.google.ar.core.Frame#hitTestInstantPlacement(float,float,float) }.
   *
   * <p>If the tracking method is {@link
   * com.google.ar.core.InstantPlacementPoint.TrackingMethod#SCREENSPACE_WITH_APPROXIMATE_DISTANCE
   * TrackingMethod#SCREENSPACE_WITH_APPROXIMATE_DISTANCE}, the pose will be based on the
   * approximate distance provided to {@link
   * com.google.ar.core.Frame#hitTestInstantPlacement(float,float,float) }.
   *
   * <p>When the tracking method changes from {@link
   * com.google.ar.core.InstantPlacementPoint.TrackingMethod#SCREENSPACE_WITH_APPROXIMATE_DISTANCE }
   * to {@link com.google.ar.core.InstantPlacementPoint.TrackingMethod#FULL_TRACKING
   * TrackingMethod#FULL_TRACKING}, there will be a noticeable pose jump as the depth changes, from
   * one that's estimated based on the approximate distance provided to {@link
   * com.google.ar.core.Frame#hitTestInstantPlacement(float,float,float)
   * Frame#hitTestInstantPlacement(float, float, float)} to the actual distance as determined by
   * ARCore.
   *
   * <p>Consequently, the on-screen size, or apparent scale, of any object anchored to the pose will
   * appear to change abruptly. To prevent a visible pop in 3D asset size when the tracking method
   * changes, the scale of the object can be adjusted to counteract the apparent scale change when
   * the tracking method changes. See the {@link com.google.ar.core.InstantPlacementPoint
   * InstantPlacementPoint} class-level documentation for an example.
   */
  public com.google.ar.core.Pose getPose() {
    // Method contents removed.
  }

  /** Gets this trackable's {@link com.google.ar.core.TrackingState TrackingState}. */
  @androidx.annotation.NonNull
  public com.google.ar.core.TrackingState getTrackingState() {
    // Method contents removed.
  }

  /**
   * Returns a hash code value for the object. This method is supported for the benefit of hash
   * tables such as those provided by {@link java.util.HashMap}.
   *
   * @return a hash code value for this object.
   * @see #equals(Object)
   */
  public int hashCode() {
    // Method contents removed.
  }

  @androidx.annotation.NonNull
  public com.google.ar.core.Anchor createAnchor(com.google.ar.core.Pose pose) {
    // Method contents removed.
  }

  /**
   * Indicates whether some other object is a {@code Trackable} referencing the same logical
   * trackable as this one.
   *
   * @param obj the reference object with which to compare.
   * @return {@code true} if this object is the same as the obj argument; {@code false} otherwise.
   * @see #hashCode()
   * @see java.util.HashMap
   */
  public boolean equals(java.lang.Object obj) {
    // Method contents removed.
  }

  @androidx.annotation.NonNull
  public java.util.Collection<com.google.ar.core.Anchor> getAnchors() {
    // Method contents removed.
  }

  /**
   * Tracking methods for {@link com.google.ar.core.InstantPlacementPoint InstantPlacementPoint}.
   */
  @SuppressWarnings({"unchecked", "deprecation", "all"})
  public enum TrackingMethod {
    /**
     * The {@link com.google.ar.core.InstantPlacementPoint InstantPlacementPoint} is not currently
     * being tracked. The {@link com.google.ar.core.TrackingState TrackingState} is {@link
     * com.google.ar.core.TrackingState#PAUSED TrackingState#PAUSED} or {@link
     * com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED}.
     */
    NOT_TRACKING,
    /**
     * The {@link com.google.ar.core.InstantPlacementPoint InstantPlacementPoint} is currently being
     * tracked in screen space and the pose returned by {@link #getPose()} is being estimated using
     * the approximate distance provided to {@link
     * com.google.ar.core.Frame#hitTestInstantPlacement(float,float,float)
     * Frame#hitTestInstantPlacement(float, float, float)}.
     *
     * <p>ARCore concurrently tracks at most 20 {@link com.google.ar.core.InstantPlacementPoint
     * InstantPlacementPoint}s that are {@link
     * com.google.ar.core.InstantPlacementPoint.TrackingMethod#SCREENSPACE_WITH_APPROXIMATE_DISTANCE
     * }. As additional {@link com.google.ar.core.InstantPlacementPoint InstantPlacementPoint}s with
     * {@link
     * com.google.ar.core.InstantPlacementPoint.TrackingMethod#SCREENSPACE_WITH_APPROXIMATE_DISTANCE
     * TrackingMethod#SCREENSPACE_WITH_APPROXIMATE_DISTANCE} are created, the oldest points will
     * become permanently {@link com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED} in
     * order to maintain the maximum number of concurrently tracked points.
     */
    SCREENSPACE_WITH_APPROXIMATE_DISTANCE,
    /**
     * The {@link com.google.ar.core.InstantPlacementPoint InstantPlacementPoint} is being tracked
     * normally and {@link #getPose()} is using a pose fully determined by ARCore.
     *
     * <p>ARCore doesn't limit the number of {@link com.google.ar.core.InstantPlacementPoint
     * InstantPlacementPoint}s with {@link
     * com.google.ar.core.InstantPlacementPoint.TrackingMethod#FULL_TRACKING } that are being
     * tracked concurrently.
     */
    FULL_TRACKING;
  }
}
