/*
 * Copyright 2017 Google LLC
 *
 * Licensed for use under "ARCore Additional Terms of Service". You may obtain
 * a copy of the license at https://developers.google.com/ar/develop/terms.
 */

package com.google.ar.core;


/**
 * Holds information about the estimated lighting of the real scene. Returned by {@link
 * com.google.ar.core.Frame#getLightEstimate() }
 */
@SuppressWarnings({"unchecked", "deprecation", "all"})
public class LightEstimate {

  LightEstimate() {
    // Method contents removed.
  }

  /** @hide */
  protected void finalize() throws java.lang.Throwable {
    // Method contents removed.
  }

  /**
   * Returns the validity of this lighting estimate. When the estimate is {@link
   * com.google.ar.core.LightEstimate.State#NOT_VALID State#NOT_VALID} it should not be used for
   * rendering.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.LightEstimate.State getState() {
    // Method contents removed.
  }

  /**
   * Returns the pixel intensity of the current camera view in gamma color space. Values are in the
   * range {@code [0.0, 1.0]}, with zero being black and one being white. If {@link
   * com.google.ar.core.Config.LightEstimationMode#AMBIENT_INTENSITY } mode is not set, returns
   * zero.
   *
   * <p>If rendering in gamma color space, divide this value by 0.466, which is middle gray in gamma
   * color space, and multiply against the final calculated color after rendering. If rendering in
   * linear space, first convert this value to linear space by rising to the power 2.2. Normalize
   * the result by dividing it by 0.18, which is middle gray in linear space. Then multiply by the
   * final calculated color after rendering.
   */
  public float getPixelIntensity() {
    // Method contents removed.
  }

  /**
   * Returns RGB scaling factors to be applied to the final color computed in the fragment shader.
   * The last component contains the average pixel intensity. The color correction method uses the
   * green channel as reference baseline and scales the red and blue channels accordingly. In this
   * way the overall intensity will not be significantly changed. If {@link
   * com.google.ar.core.Config.LightEstimationMode#AMBIENT_INTENSITY } mode is not set, returns all
   * zeros.
   *
   * <p>The RGB scale factors can be used independently from the pixel intensity value. They are put
   * together for the convenience of only having to upload one float4 to the fragment shader.
   *
   * <p>The RGB scale factors are not intended to brighten nor dim the scene. They are only to shift
   * the color of the virtual object towards the color of the light; not intensity of the light. The
   * pixel intensity is used to match the intensity of the light in the scene.
   *
   * <p>Color correction values are reported in gamma color space. If rendering in gamma color
   * space, multiply them component-wise against the final calculated color after rendering. If
   * rendering in linear space, first convert the values to linear space by rising to the power 2.2.
   *
   * @param outColorCorrection4 The float array the 4 component color correction values are written
   *     to. The four values are:
   *     <ul>
   *       <li>{@code outColorCorrection4[0]}: Color correction value for the red channel. This
   *           value is larger or equal to zero.
   *       <li>{@code outColorCorrection4[1]}: Color correction value for the green channel. This
   *           value is always 1.0 as the green channel is the reference baseline.
   *       <li>{@code outColorCorrection4[2]}: Color correction value for the blue channel. This
   *           value is larger or equal to zero.
   *       <li>{@code outColorCorrection4[3]}: This value is identical to the average pixel
   *           intensity from {@link #getPixelIntensity()} in the range {@code [0.0, 1.0]}.
   *     </ul>
   *
   * @param offset The offset into the array the 4 color correction components are written.
   */
  public void getColorCorrection(float[] outColorCorrection4, int offset) {
    // Method contents removed.
  }

  /**
   * Returns the timestamp of the LightEstimate in nanoseconds.
   *
   * <p>ARCore returns a different timestamp when the underlying light estimate has changed.
   * Conversely, the same timestamp is returned if the light estimate has not changed.
   *
   * <p>This timestamp uses the same time base as {@link com.google.ar.core.Frame#getTimestamp()
   * Frame#getTimestamp()}.
   */
  public long getTimestamp() {
    // Method contents removed.
  }

  /**
   * Returns the direction of the main directional light based on the inferred Environmental HDR
   * light estimation. If {@link com.google.ar.core.Config.LightEstimationMode#ENVIRONMENTAL_HDR
   * Config.LightEstimationMode#ENVIRONMENTAL_HDR} mode is not set, returns {@code [0.0, 1.0, 0.0]},
   * representing a light shining straight down from above.
   *
   * @return The normalized output lighting direction of the main directional light as 3 floats
   *     {@code [x, y, z]}.
   */
  public float[] getEnvironmentalHdrMainLightDirection() {
    // Method contents removed.
  }

  /**
   * Returns the intensity of the main directional light based on the inferred Environmental HDR
   * lighting estimation. All return values are larger or equal to zero. If {@link
   * com.google.ar.core.Config.LightEstimationMode#ENVIRONMENTAL_HDR } mode is not set, returns zero
   * for all elements of the array.
   *
   * @return The output lighting intensity of the main directional light as 3 floats [r, g, b].
   */
  public float[] getEnvironmentalHdrMainLightIntensity() {
    // Method contents removed.
  }

  /**
   * Gets the spherical harmonics coefficients for the ambient illumination based on the inferred
   * Environmental HDR lighting.
   *
   * @return The output spherical harmonics coefficients for the ambient illumination. This array
   *     contains 9 sets of per-channel coefficients, or a total of 27 values of 32-bit floating
   *     point type. The coefficients are stored in a channel-major fashion e.g. {@code [r0, g0, b0,
   *     r1, g1, b1, ... , r8, g8, b8]}. If {@link
   *     com.google.ar.core.Config.LightEstimationMode#ENVIRONMENTAL_HDR
   *     Config.LightEstimationMode#ENVIRONMENTAL_HDR} mode is not set, returns zero for all 27
   *     coefficients.
   */
  public float[] getEnvironmentalHdrAmbientSphericalHarmonics() {
    // Method contents removed.
  }

  /**
   * Gets the 6 cubemap textures in OpenGL texture format based on the inferred Environmental HDR
   * lighting.
   *
   * @return An array that contains 6 images in <a
   *     href="https://developer.android.com/ndk/reference/group/media#aimage_formats">{@code
   *     AIMAGE_FORMAT_RGBA_FP16}</a> format for the HDR cubemap. The memory layout for the image
   *     data is identical to {@code GL_RGBA16F}. The pixel values are in linear color space. The
   *     order of the images corresponds to the cubemap order in the following fashion:
   *     <ul>
   *       <li>{@code cubemap[0]}: {@code GL_TEXTURE_CUBE_MAP_POSITIVE_X}
   *       <li>{@code cubemap[1]}: {@code GL_TEXTURE_CUBE_MAP_NEGATIVE_X}
   *       <li>{@code cubemap[2]}: {@code GL_TEXTURE_CUBE_MAP_POSITIVE_Y}
   *       <li>{@code cubemap[3]}: {@code GL_TEXTURE_CUBE_MAP_NEGATIVE_Y}
   *       <li>{@code cubemap[4]}: {@code GL_TEXTURE_CUBE_MAP_POSITIVE_Z}
   *       <li>{@code cubemap[5]}: {@code GL_TEXTURE_CUBE_MAP_NEGATIVE_Z}
   *     </ul>
   *     If {@link com.google.ar.core.Config.LightEstimationMode#ENVIRONMENTAL_HDR
   *     Config.LightEstimationMode#ENVIRONMENTAL_HDR} mode is not set, all textures will be
   *     assigned with zero pixel values. All 6 acquired images must be released with {@link
   *     android.media.Image#close()} once they are no longer needed.
   */
  public android.media.Image[] acquireEnvironmentalHdrCubeMap() {
    // Method contents removed.
  }

  /** Light Estimate States. */
  @SuppressWarnings({"unchecked", "deprecation", "all"})
  public enum State {
    /** The light estimate is not valid this frame */
    NOT_VALID,
    /** The light estimate is valid this frame */
    VALID;
  }
}
