/*
 * Copyright 2017 Google LLC
 *
 * Licensed for use under "ARCore Additional Terms of Service". You may obtain
 * a copy of the license at https://developers.google.com/ar/develop/terms.
 */

package com.google.ar.core;

/**
 * Represents an immutable rigid transformation from one coordinate space to another. As provided
 * from all ARCore APIs, Poses always describe the transformation from object's local coordinate
 * space to the <b>world coordinate space</b> (see below). That is, Poses from ARCore APIs can be
 * thought of as equivalent to OpenGL model matrices.
 *
 * <p>The transformation is defined using a quaternion rotation about the origin followed by a
 * translation.
 *
 * <p>Coordinate system is right-handed, like OpenGL conventions.
 *
 * <p>Translation units are meters.
 *
 * <h2>World Coordinate Space</h2>
 *
 * <p>As ARCore's understanding of the environment changes, it adjusts its model of the world to
 * keep things consistent. When this happens, the numerical location (coordinates) of the camera and
 * {@link com.google.ar.core.Anchor Anchor}s can change significantly to maintain appropriate
 * relative positions of the physical locations they represent.
 *
 * <p>These changes mean that every frame should be considered to be in a completely unique world
 * coordinate space. The numerical coordinates of anchors and the camera should never be used
 * outside the rendering frame during which they were retrieved. If a position needs to be
 * considered beyond the scope of a single rendering frame, either an anchor should be created or a
 * position relative to a nearby existing anchor should be used.
 */
@SuppressWarnings({"unchecked", "deprecation", "all"})
@com.google.ar.core.annotations.UsedByNative("session_jni_wrapper.cc")
public class Pose {

  /**
   * Returns a new pose having the specified translation and rotation.
   *
   * <p>Formally, the translation and rotation of an Pose are defined as follows:
   *
   * <p>Translation is the position vector from the destination (usually world) coordinate space to
   * the local coordinate frame, expressed in destination (world) coordinates.
   *
   * <p>Rotation is a quaternion following the Hamilton convention. Assume the destination and local
   * coordinate spaces are initially aligned, and the local coordinate space is then rotated
   * counter-clockwise about a unit-length axis, k, by an angle, theta. The quaternion parameters
   * are hence:<br>
   * {@code x = k.x * sin(theta/2)}<br>
   * {@code y = k.y * sin(theta/2)}<br>
   * {@code z = k.z * sin(theta/2)}<br>
   * {@code w = cos(theta/2)}
   *
   * <p>The contents of both input arrays will be copied - later modifications will not affect the
   * constructed Pose.
   *
   * @param translation The translation component of the pose. The first three elements will be used
   *     as the X, Y, and Z components of the translation respectively.
   * @param rotation The rotation component of the pose as described above. Order is {x, y, z, w}.
   */
  public Pose(float[] translation, float[] rotation) {
    // Method contents removed.
  }

  /**
   * Creates a translation-only pose. See {@link #Pose(float[],float[])} for definition of the
   * translation.
   *
   * @param tx X component of the translation.
   * @param ty Y component of the translation.
   * @param tz Z component of the translation.
   */
  @androidx.annotation.NonNull
  public static com.google.ar.core.Pose makeTranslation(float tx, float ty, float tz) {
    // Method contents removed.
  }

  /**
   * Creates a translation-only pose. See {@link #Pose(float[],float[])} for definition of the
   * translation.
   *
   * @param translation Components of the translation vector in the order {x, y, z}
   */
  @androidx.annotation.NonNull
  public static com.google.ar.core.Pose makeTranslation(float[] translation) {
    // Method contents removed.
  }

  /**
   * Creates a rotation-only pose. See {@link #Pose(float[],float[])} for details of the quaternion
   * definition.
   *
   * @param x sin(theta/2)*rx
   * @param y sin(theta/2)*ry
   * @param z sin(theta/2)*rz
   * @param w cos(theta/2)
   */
  @androidx.annotation.NonNull
  public static com.google.ar.core.Pose makeRotation(float x, float y, float z, float w) {
    // Method contents removed.
  }

  /**
   * Creates a rotation-only pose. See {@link #Pose(float[],float[])} for details of the quaternion
   * definition.
   *
   * @param quaternion Components of the rotation quaternion in the order {x, y, z, w}
   */
  @androidx.annotation.NonNull
  public static com.google.ar.core.Pose makeRotation(float[] quaternion) {
    // Method contents removed.
  }

  /**
   * Returns the result of composing {@code this} with {@code rhs}. That is, transforming a point by
   * the resulting pose will be equivalent to transforming that point first by {@code rhs}, and then
   * transforming the result by {@code this}, or in code:
   *
   * <p>The result satisfies the following relationship: {@code result.toMatrix() == this.toMatrix()
   * * rhs.toMatrix()}.
   *
   * @param rhs the pose to combine, as described above
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.Pose compose(com.google.ar.core.Pose rhs) {
    // Method contents removed.
  }

  /**
   * Returns a new pose that blends between two input poses. Linear and spherical-linear
   * interpolation are performed on the translation and rotation respectively.
   *
   * <p>Rotation interpolation always takes the short path, negating the components of {@code b}'s
   * rotation if the result is more similar to {@code a}'s rotation. As a result, while the
   * resulting transformation will approach {@code b}'s transformation as {@code t} approaches 1,
   * the numerical representation as a quaternion may not.
   *
   * <p>The returned value is equal to {@code a} when {@code t == 0}, and equal to {@code b} when
   * {@code t == 1}. Values of {@code t} outside the range [0, 1] will result in overshoot of the
   * transformation, though correct operation well outside that range is not guaranteed.
   *
   * @param a The pose to return when t == 0
   * @param b The pose to return when t == 1
   * @param t the blending factor
   */
  @androidx.annotation.NonNull
  public static com.google.ar.core.Pose makeInterpolated(
      com.google.ar.core.Pose a, com.google.ar.core.Pose b, float t) {
    // Method contents removed.
  }

  /**
   * Returns a pose that performs the opposite transformation.
   *
   * <p>{@code pose.compose(pose.inverse())} will, allowing for floating point precision errors,
   * produce an identity pose.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.Pose inverse() {
    // Method contents removed.
  }

  /** Returns a pose having the rotation of this pose but no translation. */
  @androidx.annotation.NonNull
  public com.google.ar.core.Pose extractRotation() {
    // Method contents removed.
  }

  /** Returns a pose having the translation of this pose but no rotation. */
  public com.google.ar.core.Pose extractTranslation() {
    // Method contents removed.
  }

  /**
   * Converts this pose to a model matrix, placing the matrix in column-major order into entries
   * {@code offset} through {@code offset+15} of the dest array.
   *
   * @param dest Array in which to write the matrix at indices offset..offset+15
   * @param offset Location within dest of the first element to write
   */
  public void toMatrix(float[] dest, int offset) {
    // Method contents removed.
  }

  /** Returns the X component of this pose's translation. */
  public float tx() {
    // Method contents removed.
  }

  /** Returns the Y component of this pose's translation. */
  public float ty() {
    // Method contents removed.
  }

  /** Returns the Z component of this pose's translation. */
  public float tz() {
    // Method contents removed.
  }

  /**
   * Returns the X component of this pose's rotation quaternion.
   *
   * @see #Pose(float[], float[])
   */
  public float qx() {
    // Method contents removed.
  }

  /**
   * Returns the Y component of this pose's rotation quaternion.
   *
   * @see #Pose(float[], float[])
   */
  public float qy() {
    // Method contents removed.
  }

  /**
   * Returns the Z component of this pose's rotation quaternion.
   *
   * @see #Pose(float[], float[])
   */
  public float qz() {
    // Method contents removed.
  }

  /**
   * Returns the W component of this pose's rotation quaternion.
   *
   * @see #Pose(float[], float[])
   */
  public float qw() {
    // Method contents removed.
  }

  /**
   * Copies the translation vector into a float array starting at offset.
   *
   * @param dest Array in which to write the translation at indices offset..offset+2.
   * @param offset Location within dest of the first element to write.
   */
  public void getTranslation(float[] dest, int offset) {
    // Method contents removed.
  }

  /** Returns a {@code float[3]} containing the translation component of this pose. */
  public float[] getTranslation() {
    // Method contents removed.
  }

  /**
   * Copies the rotation quaternion into a float array starting at offset. The values are written in
   * the order {x, y, z, w}.
   *
   * @param dest Array in which to write the quaternion at indices offset..offset+3.
   * @param offset Location within dest of the first element to write.
   */
  public void getRotationQuaternion(float[] dest, int offset) {
    // Method contents removed.
  }

  /**
   * Returns a {@code float[4]} containing the rotation component of this pose. The quaternion
   * values are written in the order {x, y, z, w}.
   */
  public float[] getRotationQuaternion() {
    // Method contents removed.
  }

  /**
   * Rotates the provided vector by the pose's rotation. Does not apply translation. In-place
   * operation is allowed.
   *
   * <p>Equivalent to taking the rotation matrix portion of this pose and doing: out = R * v.
   *
   * @param vectorIn Array containing the input vector.
   * @param inOffset Location within vectorIn of the first element to read.
   * @param vectorOut Array in which to write the output vector.
   * @param outOffset Location within vectorOut of the first element to write.
   */
  public void rotateVector(float[] vectorIn, int inOffset, float[] vectorOut, int outOffset) {
    // Method contents removed.
  }

  /**
   * Rotates the provided vector by the pose's rotation. Does not apply translation.
   *
   * <p>Equivalent to taking the rotation matrix portion of this pose and doing: out = R * v.
   *
   * @param vectorIn A float[3] containing the input vector.
   * @return A float[3] containing the rotated vector.
   */
  public float[] rotateVector(float[] vectorIn) {
    // Method contents removed.
  }

  /**
   * Transforms the provided point by the pose. In-place operation is allowed. Applies the pose's
   * transformation to {@code pointIn[inOffset..inOffset+2]}, placing the result in {@code
   * pointOut[outOffset..outOffset+2]}
   *
   * <p>Equivalent to taking the matrix M from toMatrix and doing {@code out = M * in}.
   *
   * @param pointIn Array containing the input point.
   * @param inOffset Location within pointIn of the first element to read.
   * @param pointOut Array in which to write the output point.
   * @param outOffset Location within pointOut of the first element to write.
   */
  public void transformPoint(float[] pointIn, int inOffset, float[] pointOut, int outOffset) {
    // Method contents removed.
  }

  /**
   * Transforms the provided point by this pose.
   *
   * <p>Letting x = point_local, this is semantically equivalent to<br>
   * {@code point_world = this.toMatrix() * point_local}
   *
   * @param x A 3-element array containing the point to transform.
   * @return A newly-allocated 3-element array containing the transformed point.
   */
  public float[] transformPoint(float[] x) {
    // Method contents removed.
  }

  /**
   * Computes the transformed direction of a local axis, outputting into a float array.
   *
   * @param axis the axis index 0=X, 1=Y, 2=Z
   * @param scale length of the resulting vector
   * @param dest destination array
   * @param offset index of the first element to set
   */
  public void getTransformedAxis(int axis, float scale, float[] dest, int offset) {
    // Method contents removed.
  }

  /**
   * Returns the transformed direction of a local axis.
   *
   * @param axis the axis index 0=X, 1=Y, 2=Z
   * @param scale length of the resulting vector
   */
  public float[] getTransformedAxis(int axis, float scale) {
    // Method contents removed.
  }

  /** Returns a 3-element array containing the direction of the transformed X axis. */
  public float[] getXAxis() {
    // Method contents removed.
  }

  /** Returns a 3-element array containing the direction of the transformed Y axis. */
  public float[] getYAxis() {
    // Method contents removed.
  }

  /** Returns a 3-element array containing the direction of the transformed Z axis. */
  public float[] getZAxis() {
    // Method contents removed.
  }

  /** Returns a human-readable representation of this pose. */
  public java.lang.String toString() {
    // Method contents removed.
  }

  /**
   * The identity pose. {@link #toMatrix(float[],int) toMatrix()} will produce the identity matrix.
   */
  public static final com.google.ar.core.Pose IDENTITY;

  static {
    IDENTITY = null;
  }
}
