/*
 * Copyright 2022 Google LLC
 *
 * Licensed for use under "ARCore Additional Terms of Service". You may obtain
 * a copy of the license at https://developers.google.com/ar/develop/terms.
 */

package com.google.ar.core;

/**
 * Defines the labels the Scene Semantics API is able to detect and maps human-readable names to
 * per-pixel semantic labels. See the <a
 * href="https://developers.google.com/ar/develop/java/scene-semantics">Scene Semantics Developer
 * Guide</a> for more information.
 *
 * <p>Use {@link com.google.ar.core.Frame#acquireSemanticImage() Frame#acquireSemanticImage()} to
 * obtain an image containing these pixels and {@link
 * com.google.ar.core.Frame#getSemanticLabelFraction(com.google.ar.core.SemanticLabel) } to query
 * what percentage of the image contains these pixels.
 */
@SuppressWarnings({"unchecked", "deprecation", "all"})
public enum SemanticLabel {
  /** Pixels with no semantic label available in the API output. */
  UNLABELED,
  /**
   * Pixels of the open sky, including clouds. Thin electrical wires in front of the sky are
   * included, but leaves/vegetation are not included.
   */
  SKY,
  /**
   * Pixels of buildings, including houses, garages, etc. Includes all structures attached to the
   * building, such as signs, solar panels, scaffolding, etc.
   */
  BUILDING,
  /**
   * Pixels of non-walkable vegetation, like trees and shrubs. In contrast, 'terrain' specifies
   * walkable vegetation, like grass.
   */
  TREE,
  /** Pixels of drivable surfaces for vehicles, including paved, unpaved, dirt, driveways, etc. */
  ROAD,
  /** Pixels of sidewalks for pedestrians and cyclists, including associated curbs. */
  SIDEWALK,
  /**
   * Pixels of walkable vegetation areas, including grass, soil, sand, etc. In contrast, 'tree'
   * specifies non-walkable vegetation, like trees and bushes.
   */
  TERRAIN,
  /**
   * Pixels of structures that are not buildings, including fences, guardrails, stand-alone walls,
   * tunnels, bridges, mountains, etc.
   */
  STRUCTURE,
  /**
   * Pixels of general temporary and permanent objects and obstacles, including mailboxes, fire
   * hydrants, poles, street lights, phone booths, bus stop enclosures, cones, parking meters, etc.
   */
  OBJECT,
  /**
   * Pixels of vehicles, including cars, vans, buses, trucks, motorcycles, bicycles, trains, etc.
   */
  VEHICLE,
  /** Pixels of humans and animals. */
  PERSON,
  /** Pixels of ground surfaces covered by water, including lakes, rivers, etc. */
  WATER;
}
