/*
 * Copyright 2017 Google LLC
 *
 * Licensed for use under "ARCore Additional Terms of Service". You may obtain
 * a copy of the license at https://developers.google.com/ar/develop/terms.
 */

package com.google.ar.core;


/**
 * Manages AR system state and handles the session lifecycle. This class is the main entry point to
 * the ARCore API. This class allows the user to create a session, configure it, start or stop it
 * and, most importantly, receive frames that allow access to camera image and device pose.
 *
 * <p><strong>Important:</strong>
 *
 * <p><b>Before creating a {@link com.google.ar.core.Session Session}, you must first verify that
 * ARCore is installed and up to date</b>. If ARCore isn't installed, then session creation will
 * fail and any subsequent installation or upgrade of ARCore will require an app restart, and might
 * cause Android to kill the app. You can verify ARCore is installed and up to date by:
 *
 * <ul>
 *   <li>Checking {@link com.google.ar.core.ArCoreApk#checkAvailability(android.content.Context)
 *       ArCoreApk#checkAvailability(Context)} returns {@link
 *       com.google.ar.core.ArCoreApk.Availability#SUPPORTED_INSTALLED }, or by checking
 *   <li>{@link com.google.ar.core.ArCoreApk#requestInstall(Activity,boolean)
 *       ArCoreApk#requestInstall(Activity, boolean)} (or another form of {@code requestInstall()})
 *       returns {@link com.google.ar.core.ArCoreApk.InstallStatus#INSTALLED
 *       ArCoreApk.InstallStatus#INSTALLED}.
 * </ul>
 *
 * <p>
 *
 * <p>This class owns a significant amount of native heap memory. When the AR session is no longer
 * needed, call {@link #close()} to release native resources. If your app contains a single
 * AR-enabled activity, it is recommended that you call {@link #close()} from the activity's {@code
 * onDestroy} method. Failure to close the session explicitly may cause your app to run out of
 * native memory and crash.
 */
@SuppressWarnings({"unchecked", "deprecation", "all"})
public class Session {

  /**
   * Creates a new ARCore session.
   *
   * <p>Before calling this constructor, your app must check that a compatible version of ARCore is
   * installed. See the {@link com.google.ar.core.Session Session} class-level documentation for
   * details.
   *
   * <p>When created through this constructor, ARCore holds exclusive access to the camera while the
   * session is running. If you want to share the camera with ARCore (for example, to enable fast
   * switching between AR and non-AR modes), consider using {@link
   * com.google.ar.core.Session.Feature#SHARED_CAMERA Session.Feature#SHARED_CAMERA}.
   *
   * <p>This class owns a significant amount of native heap memory. When the AR session is no longer
   * needed, call {@link #close()} to release native resources. If your app contains a single
   * AR-enabled activity, it is recommended that you call {@link #close()} from the activity's
   * {@code onDestroy} method. Failure to close the session explicitly may cause your app to run out
   * of native memory and crash.
   *
   * @param context the {@link android.content.Context Context} for your app
   * @throws com.google.ar.core.exceptions.FatalException if an internal error occurred while
   *     creating the session. {@code adb logcat} may contain useful information.
   * @throws java.lang.SecurityException if your app does not have the {@link
   *     android.Manifest.permission#CAMERA} permission.
   * @throws com.google.ar.core.exceptions.UnavailableArcoreNotInstalledException if the ARCore APK
   *     is not present. This can be prevented by the installation check described in the {@link
   *     com.google.ar.core.Session Session} class-level documentation.
   * @throws com.google.ar.core.exceptions.UnavailableSdkTooOldException if the ARCore SDK that this
   *     app was built with is too old and is no longer supported by the installed ARCore APK.
   * @throws com.google.ar.core.exceptions.UnavailableApkTooOldException if the installed ARCore APK
   *     is too old for the ARCore SDK with which this app was built. This can be prevented by the
   *     installation check described in the {@link com.google.ar.core.Session Session} class-level
   *     documentation.
   * @throws com.google.ar.core.exceptions.UnavailableDeviceNotCompatibleException if the device is
   *     not compatible with ARCore. If encountered after completing the installation check, this
   *     usually indicates that ARCore has been side-loaded onto an incompatible device.
   */
  public Session(android.content.Context context)
      throws com.google.ar.core.exceptions.UnavailableApkTooOldException,
          com.google.ar.core.exceptions.UnavailableArcoreNotInstalledException,
          com.google.ar.core.exceptions.UnavailableDeviceNotCompatibleException,
          com.google.ar.core.exceptions.UnavailableSdkTooOldException {
    // Method contents removed.
  }

  /**
   * Creates a new ARCore session requesting additional features.
   *
   * <p>Before calling this constructor, your app must check that a compatible version of ARCore is
   * installed. See the {@link com.google.ar.core.Session Session} class-level documentation for
   * details.
   *
   * <p>This class owns a significant amount of native heap memory. When the AR session is no longer
   * needed, call {@link #close()} to release native resources. If your app contains a single
   * AR-enabled activity, it is recommended that you call {@link #close()} from the activity's
   * {@code onDestroy} method. Failure to close the session explicitly may cause your app to run out
   * of native memory and crash.
   *
   * @param context the {@link android.content.Context Context} for your app
   * @param features the features requested
   * @throws com.google.ar.core.exceptions.FatalException if an internal error occurred while
   *     creating the session. {@code adb logcat} may contain useful information.
   * @throws java.lang.SecurityException if your app does not have the {@link
   *     android.Manifest.permission#CAMERA} permission.
   * @throws com.google.ar.core.exceptions.UnavailableArcoreNotInstalledException if the ARCore APK
   *     is not present. This can be prevented by the installation check described in the {@link
   *     com.google.ar.core.Session Session} class-level documentation.
   * @throws com.google.ar.core.exceptions.UnavailableSdkTooOldException if the ARCore SDK that this
   *     app was built with is too old and is no longer supported by the installed ARCore APK.
   * @throws com.google.ar.core.exceptions.UnavailableApkTooOldException if the installed ARCore APK
   *     is too old for the ARCore SDK with which this app was built. This can be prevented by the
   *     installation check described in the {@link com.google.ar.core.Session Session} class-level
   *     documentation.
   * @throws java.lang.IllegalArgumentException if the requested features are mutually incompatible.
   *     See {@link com.google.ar.core.Session.Feature Session.Feature} for details. This is not
   *     possible with the currently available features.
   * @throws com.google.ar.core.exceptions.UnavailableDeviceNotCompatibleException if the device is
   *     not compatible with ARCore. If encountered after completing the installation check, this
   *     usually indicates that ARCore has been side-loaded onto an incompatible device.
   */
  public Session(
      android.content.Context context, java.util.Set<com.google.ar.core.Session.Feature> features)
      throws java.lang.IllegalArgumentException,
          com.google.ar.core.exceptions.UnavailableApkTooOldException,
          com.google.ar.core.exceptions.UnavailableArcoreNotInstalledException,
          com.google.ar.core.exceptions.UnavailableDeviceNotCompatibleException,
          com.google.ar.core.exceptions.UnavailableSdkTooOldException {
    // Method contents removed.
  }

  /**
   * Gets the SharedCamera object if the session was created for camera sharing using {@link
   * com.google.ar.core.Session.Feature#SHARED_CAMERA }.
   *
   * @throws java.lang.IllegalStateException if session is not created for shared camera.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.SharedCamera getSharedCamera() {
    // Method contents removed.
  }

  /** @hide */
  protected void finalize() throws java.lang.Throwable {
    // Method contents removed.
  }

  /**
   * Releases resources (a significant amount of native heap memory) used by an ARCore session.
   *
   * <p>Failure to close sessions explicitly may cause your app to run out of native memory and
   * crash. If your app contains a single AR-enabled activity, it is recommended that you call
   * {@link #close()} from the activity's onDestroy method.
   *
   * <p>This method will take several seconds to complete. To prevent blocking the main thread, call
   * {@link #pause()} on the main thread, and then call {@link #close()} on a background thread.
   *
   * <p>It is not safe to call methods on this session or other objects obtained from this session
   * while the session is being closed and after the session is closed.
   *
   * <p>When closing a session that was created with {@link
   * com.google.ar.core.Session.Feature#SHARED_CAMERA Session.Feature#SHARED_CAMERA}, the camera
   * device must be closed via {@code CameraDevice#close()} before calling this method.
   *
   * <p>Creation of a new session on a different thread will not be blocked by the ongoing call to
   * {@link #close()} on a background thread.
   */
  public void close() {
    // Method contents removed.
  }

  /**
   * Configures the session and verifies that the enabled features in the specified session config
   * are supported with the currently set camera config.
   *
   * <p>Should be called after {@link
   * com.google.ar.core.Session#setCameraConfig(com.google.ar.core.CameraConfig)
   * Session#setCameraConfig(CameraConfig)} to verify that all requested session config features are
   * supported. Features not supported with the current camera config will otherwise be silently
   * disabled when the session is resumed by calling {@link com.google.ar.core.Session#resume() }.
   *
   * <p>The following configurations are unsupported and will throw {@link
   * com.google.ar.core.exceptions.UnsupportedConfigurationException}:
   *
   * <ul>
   *   <li>When using the default back-facing (world) camera:
   *       <ul>
   *         <li>The {@link com.google.ar.core.Config.AugmentedFaceMode#MESH3D
   *             Config.AugmentedFaceMode#MESH3D} enum in {@link
   *             com.google.ar.core.Config.AugmentedFaceMode Config.AugmentedFaceMode}.
   *         <li>The {@link com.google.ar.core.Config.DepthMode#AUTOMATIC
   *             Config.DepthMode#AUTOMATIC} enum in {@link com.google.ar.core.Config.DepthMode
   *             Config.DepthMode} on devices that do not support this Depth API mode. See {@link
   *             #isDepthModeSupported}.
   *         <li>The {@link com.google.ar.core.Config.GeospatialMode#ENABLED
   *             Config.GeospatialMode#ENABLED} enum in {@link
   *             com.google.ar.core.Config.GeospatialMode Config.GeospatialMode} on devices that do
   *             not support this Geospatial mode. See {@link #isGeospatialModeSupported}.
   *       </ul>
   *   <li>When using the front-facing (selfie) camera:
   *       <ul>
   *         <li>Any config using {@link
   *             com.google.ar.core.Config#setAugmentedImageDatabase(com.google.ar.core.AugmentedImageDatabase)
   *             Config#setAugmentedImageDatabase(AugmentedImageDatabase)}.
   *         <li>The {@link com.google.ar.core.Config.CloudAnchorMode#ENABLED
   *             Config.CloudAnchorMode#ENABLED} enum in {@link
   *             com.google.ar.core.Config.CloudAnchorMode Config.CloudAnchorMode}.
   *         <li>The {@link com.google.ar.core.Config.LightEstimationMode#ENVIRONMENTAL_HDR
   *             Config.LightEstimationMode#ENVIRONMENTAL_HDR} enum in {@link
   *             com.google.ar.core.Config.LightEstimationMode Config.LightEstimationMode}.
   *         <li>The {@link com.google.ar.core.Config.DepthMode#AUTOMATIC
   *             Config.DepthMode#AUTOMATIC} enum in {@link com.google.ar.core.Config.DepthMode
   *             Config.DepthMode}.
   *         <li>The {@link com.google.ar.core.Config.GeospatialMode#ENABLED
   *             Config.GeospatialMode#ENABLED} enum in {@link
   *             com.google.ar.core.Config.GeospatialMode Config.GeospatialMode}.
   *       </ul>
   * </ul>
   *
   * @param config The new configuration setting for the session.
   * @throws com.google.ar.core.exceptions.UnsupportedConfigurationException if the requested
   *     session config is not supported. See above restrictions.
   * @throws com.google.ar.core.exceptions.GooglePlayServicesLocationLibraryNotLinkedException if
   *     {@link com.google.ar.core.Config.GeospatialMode Config.GeospatialMode} is {@link
   *     com.google.ar.core.Config.GeospatialMode#ENABLED Config.GeospatialMode#ENABLED} and the <a
   *     href="https://developers.google.com/android/guides/setup#declare-dependencies">Google Play
   *     Services Location Library</a> could not be found. See {@link
   *     com.google.ar.core.exceptions.GooglePlayServicesLocationLibraryNotLinkedException} for
   *     additional troubleshooting steps.
   * @throws com.google.ar.core.exceptions.FineLocationPermissionNotGrantedException if {@link
   *     com.google.ar.core.Config.GeospatialMode Config.GeospatialMode} is {@link
   *     com.google.ar.core.Config.GeospatialMode#ENABLED Config.GeospatialMode#ENABLED} and the
   *     {@code ACCESS_FINE_LOCATION} has not been granted.
   * @throws java.lang.SecurityException if {@link com.google.ar.core.Config.GeospatialMode
   *     Config.GeospatialMode} is {@link com.google.ar.core.Config.GeospatialMode#ENABLED } or
   *     {@link com.google.ar.core.Config.CloudAnchorMode Config.CloudAnchorMode} is {@link
   *     com.google.ar.core.Config.CloudAnchorMode#ENABLED } and the Android <a
   *     href="https://developer.android.com/training/basics/network-ops/connecting">{@code
   *     INTERNET}</a> permission has not been granted.
   */
  public void configure(com.google.ar.core.Config config) {
    // Method contents removed.
  }

  /**
   * Gets the current config. More specifically, returns a copy of the config most recently set by
   * {@link #configure(com.google.ar.core.Config)}.
   *
   * <p>Note: if the session was not explicitly configured, a default configuration is returned
   * (same as {@code new Config(session)}).
   *
   * <p>To avoid allocations, use {@link #getConfig(com.google.ar.core.Config)} instead.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.Config getConfig() {
    // Method contents removed.
  }

  /**
   * Similar to {@link #getConfig()}, but takes a config object to fill. Use this form to avoid
   * extra object allocations.
   *
   * @see #getConfig()
   */
  public void getConfig(com.google.ar.core.Config configToFill) {
    // Method contents removed.
  }

  /**
   * Starts or resumes the ARCore Session.
   *
   * <p>Typically this should be called from <a
   * href="https://developer.android.com/reference/android/app/Activity.html#onResume()">{@code
   * onResume()}</a>.
   *
   * <p>Note that if the camera configuration has been changed by {@link
   * #setCameraConfig(com.google.ar.core.CameraConfig)} since the last call to {@link #resume()},
   * all images previously acquired using {@link com.google.ar.core.Frame#acquireCameraImage()
   * Frame#acquireCameraImage()} must be released by calling {@link android.media.Image#close()}
   * before resuming. If there are open images, {@link #resume()} will throw {@link
   * java.lang.IllegalStateException IllegalStateException}.
   *
   * @throws com.google.ar.core.exceptions.SessionNotPausedException if the session was not paused.
   * @throws com.google.ar.core.exceptions.CameraNotAvailableException if the camera isn't available
   *     yet.
   * @throws java.lang.SecurityException if the camera permission is not granted.
   * @throws java.lang.IllegalStateException if there are any acquired images still open.
   * @throws com.google.ar.core.exceptions.FatalException if an unrecoverable error occurs. Check
   *     the logcat for more details.
   */
  public void resume() throws com.google.ar.core.exceptions.CameraNotAvailableException {
    // Method contents removed.
  }

  /**
   * Updates the state of the ARCore system. This includes: receiving a new camera frame, updating
   * the location of the device, updating the location of tracking anchors, updating detected
   * planes, etc.
   *
   * <p>This call may cause off-screen OpenGL activity. Because of this, to avoid unnecessary frame
   * buffer flushes and reloads, this call should not be made in the middle of rendering a frame or
   * offscreen buffer.
   *
   * <p>This call may update the pose of all created anchors and detected planes. The set of updated
   * objects is accessible through {@link
   * com.google.ar.core.Frame#getUpdatedTrackables(java.lang.Class)
   * Frame#getUpdatedTrackables(Class)}.
   *
   * <p>This call in blocking mode (see {@link com.google.ar.core.Config.UpdateMode
   * Config.UpdateMode}) will wait until a new camera image is available, or until the built-in
   * timeout (currently 66ms) is reached. If the camera image does not arrive by the built-in
   * timeout, then update() will return the most recent {@link com.google.ar.core.Frame Frame}
   * object. For some applications it may be important to know if a new frame was actually obtained
   * (for example, to avoid redrawing if the camera did not produce a new frame). To do that,
   * compare the current frame's timestamp, obtained via {@link
   * com.google.ar.core.Frame#getTimestamp() Frame#getTimestamp()}, with the previously recorded
   * frame timestamp. If they are different, this is a new frame.
   *
   * <p>During startup the camera system may not produce actual images immediately. In this common
   * case, a frame with timestamp = 0 will be returned.
   *
   * @return The most recent Frame received
   * @throws com.google.ar.core.exceptions.CameraNotAvailableException if the camera could not be
   *     opened.
   * @throws com.google.ar.core.exceptions.SessionPausedException if the session is currently
   *     paused.
   * @throws com.google.ar.core.exceptions.MissingGlContextException if there is no OpenGL context
   *     available.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.Frame update()
      throws com.google.ar.core.exceptions.CameraNotAvailableException {
    // Method contents removed.
  }

  /**
   * Pause the current session. This method will stop the camera feed and release resources. The
   * session can be restarted again by calling {@link #resume()}.
   *
   * <p>Typically this should be called from <a
   * href="https://developer.android.com/reference/android/app/Activity.html#onPause()">{@code
   * onPause()}</a>.
   *
   * <p>Note that ARCore might continue consuming substantial computing resources for up to 10
   * seconds after calling this method.
   */
  public void pause() {
    // Method contents removed.
  }

  /**
   * Sets the OpenGL texture names (ids) that will be assigned to incoming camera frames in sequence
   * in a ring buffer. The textures must be bound to the {@link
   * android.opengl.GLES11Ext#GL_TEXTURE_EXTERNAL_OES GLES11Ext#GL_TEXTURE_EXTERNAL_OES} target for
   * use. Shaders accessing these textures must use a {@code samplerExternalOES} sampler.
   *
   * <p>The texture contents are not guaranteed to remain valid after another call to {@link
   * #setCameraTextureName} or {@link #setCameraTextureNames}, and additionally are not guaranteed
   * to remain valid after a call to {@link #pause} or {@link #close}.
   *
   * <p>Passing multiple textures allows for a multithreaded rendering pipeline, unlike {@link
   * #setCameraTextureName}.
   *
   * @param textureIds an array of valid OpenGL texture names (ids)
   * @throws java.lang.IllegalArgumentException if textureIds is null or empty.
   */
  public void setCameraTextureNames(int[] textureIds) {
    // Method contents removed.
  }

  /**
   * Sets the OpenGL texture name (id) that will allow GPU access to the camera image. The provided
   * ID should have been created with {@link android.opengl.GLES20#glGenTextures(int,int[],int)
   * GLES20#glGenTextures(int, int[], int)}. The resulting texture must be bound to the {@link
   * android.opengl.GLES11Ext#GL_TEXTURE_EXTERNAL_OES GLES11Ext#GL_TEXTURE_EXTERNAL_OES} target for
   * use. Shaders accessing this texture must use a {@code samplerExternalOES} sampler.
   *
   * <p>The texture contents are not guaranteed to remain valid after another call to {@link
   * #setCameraTextureName} or {@link #setCameraTextureNames}, and additionally are not guaranteed
   * to remain valid after a call to {@link #pause} or {@link #close}.
   *
   * @param textureId a valid OpenGL texture name (id)
   */
  public void setCameraTextureName(int textureId) {
    // Method contents removed.
  }

  /**
   * Sets the aspect ratio, coordinate scaling, and display rotation. This data is used by UV
   * conversion, projection matrix generation, and hit test logic.
   *
   * @param displayRotation display rotation specified by {@link android.view.Surface} constants:
   *     {@link android.view.Surface#ROTATION_0}, {@link android.view.Surface#ROTATION_90}, {@link
   *     android.view.Surface#ROTATION_180}, {@link android.view.Surface#ROTATION_270},
   * @param widthPx width of the view, in pixels
   * @param heightPx height of the view, in pixels
   */
  public void setDisplayGeometry(int displayRotation, int widthPx, int heightPx) {
    // Method contents removed.
  }

  /**
   * Before release 1.2.0: Checks if the provided configuration is usable on the this device. If
   * this method returns false, calls to {@link #configure(com.google.ar.core.Config)} with this
   * configuration will throw an exception.
   *
   * <p>This function now always returns true. See documentation for each configuration entry to
   * know which configuration options & combinations are supported.
   *
   * @param config the configuration to test
   * @deprecated Please refer to (<a
   *     href="https://github.com/google-ar/arcore-android-sdk/releases/tag/v1.2.0">release notes
   *     1.2.0</a>).
   */
  @Deprecated
  public boolean isSupported(com.google.ar.core.Config config) {
    // Method contents removed.
  }

  /**
   * Returns all known anchors, including those not currently tracked. Anchors forgotten by ARCore
   * due to a call to {@link com.google.ar.core.Anchor#detach() Anchor#detach()} or entering the
   * {@link com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED} state will not be
   * included.
   */
  @androidx.annotation.NonNull
  public java.util.Collection<com.google.ar.core.Anchor> getAllAnchors() {
    // Method contents removed.
  }

  /**
   * Returns the list of all known trackables. This includes {@link com.google.ar.core.Plane Plane}s
   * if plane detection is enabled, as well as {@link com.google.ar.core.Point Point}s created as a
   * side effect of calls to {@link #createAnchor(com.google.ar.core.Pose)} or {@link
   * com.google.ar.core.Frame#hitTest(float,float) Frame#hitTest(float, float)}.
   *
   * @param filterType The desired trackable type, or {@code Trackable.class} to retrieve all
   *     trackables.
   */
  @androidx.annotation.NonNull
  public <T extends com.google.ar.core.Trackable> java.util.Collection<T> getAllTrackables(
      java.lang.Class<T> filterType) {
    // Method contents removed.
  }

  /**
   * Defines a tracked location in the physical world. See {@link com.google.ar.core.Anchor Anchor}
   * for more details.
   *
   * <p>Anchors incur ongoing processing overhead within ARCore. To release unneeded anchors use
   * {@link com.google.ar.core.Anchor#detach() Anchor#detach()}.
   *
   * @param pose a pose to anchor to the physical world
   * @see com.google.ar.core.HitResult#getHitPose()
   * @throws com.google.ar.core.exceptions.SessionPausedException if the Session is currently
   *     paused.
   * @throws com.google.ar.core.exceptions.NotTrackingException if the Camera is not currently
   *     tracking.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if too many resources have
   *     already been acquired.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.Anchor createAnchor(com.google.ar.core.Pose pose) {
    // Method contents removed.
  }

  /**
   * Estimates the quality of the visual features seen by ARCore in the preceding few seconds and
   * visible from the provided camera pose.
   *
   * <p>Cloud Anchors hosted using higher quality features will generally result in easier and more
   * accurately resolved Cloud Anchor poses.
   *
   * @param pose The camera pose to use in estimating the quality.
   * @return The estimated quality of the visual features seen by ARCore in the preceding few
   *     seconds and visible from the provided camera pose.
   * @throws com.google.ar.core.exceptions.SessionPausedException if the Session is currently
   *     paused.
   * @throws com.google.ar.core.exceptions.NotTrackingException if the Session is not currently
   *     tracking.
   * @throws com.google.ar.core.exceptions.CloudAnchorsNotConfiguredException if support for Cloud
   *     Anchors has not been enabled in the Session Config.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.Session.FeatureMapQuality estimateFeatureMapQualityForHosting(
      com.google.ar.core.Pose pose) {
    // Method contents removed.
  }

  /**
   * Uses the pose and other data from {@code anchor} to create a new anchor that will be hosted.
   * The returned {@code anchor} will have the Cloud Anchor state {@link
   * com.google.ar.core.Anchor.CloudAnchorState#TASK_IN_PROGRESS }.
   *
   * @param anchor The anchor whose pose is to be hosted.
   * @return The new anchor with the same pose as {@code anchor} which will be hosted.
   * @throws com.google.ar.core.exceptions.NotTrackingException if the Camera is not currently
   *     tracking.
   * @throws com.google.ar.core.exceptions.CloudAnchorsNotConfiguredException if support for Cloud
   *     Anchors has not been enabled in the Session Config.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if too many resources have
   *     already been acquired.
   * @throws com.google.ar.core.exceptions.SessionPausedException if the session is currently
   *     paused.
   * @throws com.google.ar.core.exceptions.AnchorNotSupportedForHostingException if the anchor is
   *     not supported for hosting.
   * @deprecated Use {@link
   *     #hostCloudAnchorAsync(com.google.ar.core.Anchor,int,java.util.function.BiConsumer)} with
   *     ttlDays=1 instead.
   */
  @Deprecated
  @androidx.annotation.NonNull
  public com.google.ar.core.Anchor hostCloudAnchor(com.google.ar.core.Anchor anchor) {
    // Method contents removed.
  }

  /**
   * Uses the pose and other data from {@code anchor} to create a new anchor with a given lifetime
   * in days that will be hosted. The returned {@code anchor} will have the Cloud Anchor state
   * {@link com.google.ar.core.Anchor.CloudAnchorState#TASK_IN_PROGRESS
   * Anchor.CloudAnchorState#TASK_IN_PROGRESS}.
   *
   * @param anchor The anchor whose pose is to be hosted.
   * @param ttlDays The lifetime of the anchor in days. Must be positive. The maximum allowed value
   *     is 1 if using an API Key to authenticate with the ARCore Cloud Anchor service, otherwise
   *     the maximum allowed value is 365.
   * @return The new anchor with the same pose as {@code anchor} which will be hosted.
   * @deprecated Use {@link
   *     #hostCloudAnchorAsync(com.google.ar.core.Anchor,int,java.util.function.BiConsumer)} with
   *     ttlDays=1 instead.
   */
  @Deprecated
  @androidx.annotation.NonNull
  public com.google.ar.core.Anchor hostCloudAnchorWithTtl(
      com.google.ar.core.Anchor anchor, int ttlDays) {
    // Method contents removed.
  }

  /**
   * Uses the pose and other data from {@code anchor} to host a new Cloud Anchor. A Cloud Anchor is
   * assigned an identifier that can be used to create an {@link com.google.ar.core.Anchor Anchor}
   * in the same position in subsequent sessions across devices using {@link
   * #resolveCloudAnchorAsync}. See the <a
   * href="https://developers.google.com/ar/develop/java/cloud-anchors/developer-guide">Cloud
   * Anchors developer guide</a> for more information.
   *
   * <p>The duration that a Cloud Anchor can be resolved for is specified by {@code ttlDays}. When
   * using <a
   * href="https://developers.google.com/ar/develop/java/cloud-anchors/developer-guide#keyless-authorization">Keyless
   * authorization</a>, the maximum allowed value is 365 days. When using an <a
   * href="https://developers.google.com/ar/develop/java/cloud-anchors/developer-guide#api-key-authorization">API
   * Key</a> to authenticate with the ARCore API, the maximum allowed value is 1 day.
   *
   * <p>Cloud Anchors requires a {@link com.google.ar.core.Config Config} with {@link
   * com.google.ar.core.Config.CloudAnchorMode#ENABLED Config.CloudAnchorMode#ENABLED} set on this
   * session. Use {@link com.google.ar.core.Config#setCloudAnchorMode Config#setCloudAnchorMode} to
   * set the Cloud Anchor API mode and {@link com.google.ar.core.Session#configure } to configure
   * the session.
   *
   * <p>Hosting a Cloud Anchor works best when ARCore is able to create a good feature map around
   * the {@link com.google.ar.core.Anchor Anchor}. Use {@link #estimateFeatureMapQualityForHosting}
   * to determine the quality of visual features seen by ARCore in the preceding few seconds. Cloud
   * Anchors hosted using higher quality features will generally result in quicker and more
   * accurately resolved Cloud Anchor poses.
   *
   * <p>This launches an asynchronous operation used to query the Google Cloud ARCore API. See
   * {@link com.google.ar.core.Future Future} for information on obtaining results and cancelling
   * the operation.
   *
   * <p>ARCore can have up to 40 simultaneous Cloud Anchor operations, including resolved anchors
   * and active hosting operations.
   *
   * @param anchor The anchor whose pose is to be hosted.
   * @param ttlDays The lifetime of the anchor in days. Must be positive.
   * @param callback An optional <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_callback">callback</a>,
   *     called when the operation is finished, unless the future is <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_cancellation">cancelled</a>.
   *     May be null.
   * @return A handler that can be <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_polling">polled</a>
   *     or <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_cancellation">cancelled</a>.
   * @throws java.lang.IllegalArgumentException if the provided anchor or TTL are invalid.
   * @throws com.google.ar.core.exceptions.NotTrackingException if the Camera is not currently
   *     tracking.
   * @throws com.google.ar.core.exceptions.CloudAnchorsNotConfiguredException if the Cloud Anchors
   *     API has not been enabled in the session configuration.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if too many resources have
   *     already been acquired.
   * @throws com.google.ar.core.exceptions.SessionPausedException if the session is currently
   *     paused.
   * @throws com.google.ar.core.exceptions.AnchorNotSupportedForHostingException if the anchor is
   *     not supported for hosting.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.HostCloudAnchorFuture hostCloudAnchorAsync(
      com.google.ar.core.Anchor anchor,
      int ttlDays,
      @androidx.annotation.Nullable
          java.util.function.BiConsumer<
                  java.lang.String, com.google.ar.core.Anchor.CloudAnchorState>
              callback) {
    // Method contents removed.
  }

  /**
   * Creates a new anchor, the pose of which ARCore will try to resolve using the ARCore Cloud
   * Anchor service and the provided {@code cloudAnchorId}. The returned {@code anchor} will have
   * the Cloud Anchor state {@link com.google.ar.core.Anchor.CloudAnchorState#TASK_IN_PROGRESS
   * Anchor.CloudAnchorState#TASK_IN_PROGRESS}. ARCore will try to resolve up to 40 Cloud Anchors
   * simultaneously. ARCore will begin attempting additional Cloud Anchors once one or more of the
   * pending resolve requests succeeds or is cancelled.
   *
   * @param cloudAnchorId The Cloud Anchor ID of the Cloud Anchor.
   * @return The new anchor whose pose will be resolved in a later call to {@link
   *     com.google.ar.core.Session#update() Session#update()}.
   * @deprecated Use {@link
   *     #resolveCloudAnchorAsync(java.lang.String,java.util.function.BiConsumer)} instead.
   */
  @Deprecated
  @androidx.annotation.NonNull
  public com.google.ar.core.Anchor resolveCloudAnchor(java.lang.String cloudAnchorId) {
    // Method contents removed.
  }

  /**
   * Attempts to resolve a Cloud Anchor using the provided {@code cloudAnchorId}. The Cloud Anchor
   * must previously have been hosted by {@link #hostCloudAnchorAsync} or another Cloud Anchor
   * hosting method within the allotted {@code ttlDays}. See the <a
   * href="https://developers.google.com/ar/develop/java/cloud-anchors/developer-guide">Cloud
   * Anchors developer guide</a> for more information.
   *
   * <p>When resolving a Cloud Anchor, the ARCore API periodically compares visual features from the
   * scene against the anchor's 3D feature map to pinpoint the user's position and orientation
   * relative to the anchor. When it finds a match, the task completes.
   *
   * <p>This launches an asynchronous operation used to query the Google Cloud ARCore API. See
   * {@link com.google.ar.core.Future Future} for information on obtaining results and cancelling
   * the operation.
   *
   * <p>Cloud Anchors requires a {@link com.google.ar.core.Config Config} with {@link
   * com.google.ar.core.Config.CloudAnchorMode#ENABLED Config.CloudAnchorMode#ENABLED} set on this
   * session. Use {@link com.google.ar.core.Config#setCloudAnchorMode Config#setCloudAnchorMode} to
   * set the Cloud Anchor API mode and {@link com.google.ar.core.Session#configure } to configure
   * the session.
   *
   * <p>ARCore can have up to 40 simultaneous Cloud Anchor operations, including resolved anchors
   * and active hosting operations.
   *
   * @param cloudAnchorId The Cloud Anchor ID of the Cloud Anchor.
   * @param callback An optional <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_callback">callback</a>,
   *     called when the operation is finished, unless the future is <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_cancellation">cancelled</a>.
   *     May be null.
   * @return A handler that can be <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_polling">polled</a>
   *     or <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_cancellation">cancelled</a>.
   * @throws java.lang.IllegalArgumentException if the provided Cloud Anchor ID is invalid.
   * @throws com.google.ar.core.exceptions.CloudAnchorsNotConfiguredException if the Cloud Anchors
   *     API has not been enabled in the session configuration.
   * @throws com.google.ar.core.exceptions.ResourceExhaustedException if too many resources have
   *     already been acquired.
   * @throws com.google.ar.core.exceptions.SessionPausedException if the session is currently
   *     paused.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.ResolveCloudAnchorFuture resolveCloudAnchorAsync(
      java.lang.String cloudAnchorId,
      @androidx.annotation.Nullable
          java.util.function.BiConsumer<
                  com.google.ar.core.Anchor, com.google.ar.core.Anchor.CloudAnchorState>
              callback) {
    // Method contents removed.
  }

  /**
   * Gets a list of camera configs supported by the camera being used by the session.
   *
   * <p>Can be called at any time.
   *
   * <p>Each config will contain a different CPU resolution. The GPU texture resolutions will be the
   * same in all configs. Most devices provide a GPU texture resolution of 1920 x 1080, but the
   * actual resolution will vary with device capabilities.
   *
   * <p>When the session camera is a back-facing camera:
   *
   * <uL>
   *   <li>The list will always contain three camera configs.
   *   <li>The CPU image resolutions returned will be VGA, a middle resolution, and a large
   *       resolution matching the GPU texture resolution. The middle resolution is typically 1280 x
   *       720, but the actual resolution will vary with device capabilities.
   * </ul>
   *
   * <p>When the session camera is front-facing (selfie) camera, the list will contain at least one
   * supported camera config.
   *
   * <p>Notes:
   *
   * <ul>
   *   <li>Prior to ARCore SDK 1.6, the middle CPU image resolution was guaranteed to be 1280 x 720
   *       on all devices.
   *   <li>In ARCore SDK 1.7 and 1.8, when the session camera was a front-facing (selfie) camera,
   *       the list contained three identical camera configs.
   * </ul>
   *
   * @return The list of CameraConfig {@link com.google.ar.core.CameraConfig CameraConfig} objects
   *     supported by the camera being used by the session.
   * @deprecated Please use instead: {@link
   *     com.google.ar.core.Session#getSupportedCameraConfigs(com.google.ar.core.CameraConfigFilter)
   *     Session#getSupportedCameraConfigs(CameraConfigFilter)}.
   */
  @Deprecated
  @androidx.annotation.NonNull
  public java.util.List<com.google.ar.core.CameraConfig> getSupportedCameraConfigs() {
    // Method contents removed.
  }

  /**
   * Gets the list of supported camera configs that satisfy the provided filter settings.
   *
   * <p>The returned camera configs might vary at runtime depending on device capabilities and
   * ARCore supported cameras. Overly restrictive filtering can result in the returned list being
   * empty on some devices.
   *
   * <p>Beginning with ARCore SDK 1.15.0, some devices support additional camera configs with lower
   * GPU texture resolutions than the device's default GPU texture resolution. See the <a
   * href="https://developers.google.com/ar/devices">ARCore supported devices</a> page for details.
   *
   * <p>Beginning with ARCore SDK 1.23.0, the list of returned camera configs will include
   * front-facing (selfie) and back-facing (world) camera configs. In previous SDKs, returned camera
   * configs included only front-facing (selfie) or only back-facing (world) camera configs,
   * depending on whether the deprecated {@link com.google.ar.core.Session.Feature#FRONT_CAMERA
   * Session.Feature#FRONT_CAMERA} feature was used.
   *
   * <p>In the case of a non-empty list, element 0 will contain the camera config that best matches
   * the session feature and filter settings, according to the following priority:
   *
   * <ol>
   *   <li>Prefer {@link com.google.ar.core.CameraConfig.StereoCameraUsage#REQUIRE_AND_USE} over
   *       {@link com.google.ar.core.CameraConfig.StereoCameraUsage#DO_NOT_USE}
   *   <li>Prefer {@link com.google.ar.core.CameraConfig.TargetFps#TARGET_FPS_60} over {@link
   *       com.google.ar.core.CameraConfig.TargetFps#TARGET_FPS_30}
   *   <li>Prefer {@link com.google.ar.core.CameraConfig.DepthSensorUsage#REQUIRE_AND_USE} over
   *       {@link com.google.ar.core.CameraConfig.DepthSensorUsage#DO_NOT_USE}
   * </ol>
   *
   * <p>No guarantees are made about the order in which the remaining elements are returned.
   *
   * <p>Can be called at any time.
   *
   * @param cameraConfigFilter The {@link com.google.ar.core.CameraConfigFilter CameraConfigFilter}
   *     that defines the configuration filter(s).
   * @return The list of CameraConfig {@link com.google.ar.core.CameraConfig CameraConfig} objects
   *     that provide supported configurations.
   * @throws java.lang.IllegalArgumentException if the {@link com.google.ar.core.CameraConfigFilter
   *     CameraConfigFilter} is null.
   */
  @androidx.annotation.NonNull
  public java.util.List<com.google.ar.core.CameraConfig> getSupportedCameraConfigs(
      com.google.ar.core.CameraConfigFilter cameraConfigFilter) {
    // Method contents removed.
  }

  /**
   * Sets the camera config to use. The config must be one returned by {@link
   * #getSupportedCameraConfigs(com.google.ar.core.CameraConfigFilter)}.
   *
   * <p>The new camera config will be applied once the session is resumed.
   *
   * <p>The session must be paused. All previously acquired frame images must be released via {@link
   * android.media.Image#close()} before resuming. Failure to do so will cause {@link #resume()} to
   * throw IllegalStateException.
   *
   * <p>Changing the camera config for an existing session may affect which ARCore features can be
   * used. Unsupported session features are silently disabled when the session is resumed. Call
   * {@link com.google.ar.core.Session#configure(com.google.ar.core.Config)
   * Session#configure(Config)} after setting a camera config to verify that all configured session
   * features are supported with the new camera config.
   *
   * <p>Changing the current session's camera config to one that uses a different camera will cause
   * all internal session states to be reset when the session is next resumed by calling {@link
   * com.google.ar.core.Session#resume() }.
   *
   * <p>Note: Starting in ARCore SDK 1.12.0, changing the active camera config may cause the
   * tracking state on certain devices to become permanently {@link
   * com.google.ar.core.TrackingState#PAUSED TrackingState#PAUSED}. For consistent behavior across
   * all supported devices, release any previously created anchors and trackables when setting a new
   * camera config.
   *
   * @param cameraConfig Reference to one of the {@link com.google.ar.core.CameraConfig
   *     CameraConfig} that was obtained by calling {@link
   *     com.google.ar.core.Session#getSupportedCameraConfigs(com.google.ar.core.CameraConfigFilter)
   *     Session#getSupportedCameraConfigs(CameraConfigFilter)} method.
   * @throws com.google.ar.core.exceptions.SessionNotPausedException if called when session is not
   *     paused.
   */
  public void setCameraConfig(com.google.ar.core.CameraConfig cameraConfig) {
    // Method contents removed.
  }

  /**
   * Gets the current camera config used by the session.
   *
   * @return The object {@link com.google.ar.core.CameraConfig CameraConfig} that contains the
   *     current camera config. If the camera config was not explicitly set using {@link
   *     #setCameraConfig(com.google.ar.core.CameraConfig)} then it returns the default camera
   *     config.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.CameraConfig getCameraConfig() {
    // Method contents removed.
  }

  /**
   * Returns the {@link com.google.ar.core.Earth Earth} object for the session. This object is
   * long-lived; it may be used for the entire duration of the ARCore session or until a {@link
   * com.google.ar.core.Config Config} with {@link com.google.ar.core.Config.GeospatialMode#DISABLED
   * } is applied on the {@link com.google.ar.core.Session Session}.
   *
   * <p>{@link com.google.ar.core.Earth Earth} can only be obtained when a {@link
   * com.google.ar.core.Config Config} with {@link com.google.ar.core.Config.GeospatialMode#ENABLED
   * } has been set on this session. See {@link
   * com.google.ar.core.Config#setGeospatialMode(GeospatialMode) } to enable the Geospatial API.
   *
   * @return The {@link com.google.ar.core.Earth Earth} object for the session, or {@code null} if
   *     the session's config {@link com.google.ar.core.Config.GeospatialMode Config.GeospatialMode}
   *     is set to {@link com.google.ar.core.Config.GeospatialMode#DISABLED
   *     Config.GeospatialMode#DISABLED}.
   */
  @androidx.annotation.Nullable
  public com.google.ar.core.Earth getEarth() {
    // Method contents removed.
  }

  /**
   * Checks whether the provided {@link com.google.ar.core.Config.DepthMode Config.DepthMode} is
   * supported on this device with the selected camera configuration. The current list of supported
   * devices is documented on the <a href="https://developers.google.com/ar/devices">ARCore
   * supported devices</a> page.
   *
   * @param mode The desired depth mode to check.
   * @return true if the depth mode is supported on this device.
   */
  public boolean isDepthModeSupported(com.google.ar.core.Config.DepthMode mode) {
    // Method contents removed.
  }

  /**
   * Checks whether the provided {@link com.google.ar.core.Config.SemanticMode Config.SemanticMode}
   * is supported on this device with the selected camera configuration. The current list of
   * supported devices is documented on the <a
   * href="https://developers.google.com/ar/devices">ARCore supported devices</a> page.
   *
   * @param mode The desired semantic mode to check.
   * @return true if the semantic mode is supported on this device.
   */
  public boolean isSemanticModeSupported(com.google.ar.core.Config.SemanticMode mode) {
    // Method contents removed.
  }

  /**
   * Checks whether the provided {@link com.google.ar.core.Config.ImageStabilizationMode
   * Config.ImageStabilizationMode} is supported on this device with the selected camera
   * configuration. See <a
   * href="https://developers.google.com/ar/develop/java/electronic-image-stabilization">Enabling
   * Electronic Image Stabilization</a> for more information.
   *
   * <p>Attempting to use {@link #configure(com.google.ar.core.Config)} to configure a {@link
   * com.google.ar.core.Config.ImageStabilizationMode Config.ImageStabilizationMode} mode on a
   * device that isn't supported, {@link #configure(com.google.ar.core.Config)} will throw {@link
   * com.google.ar.core.exceptions.UnsupportedConfigurationException}.
   *
   * @param imageStabilizationMode Desired image stabilization mode to check.
   * @return true if the image stabilization mode requested is supported.
   */
  public boolean isImageStabilizationModeSupported(
      com.google.ar.core.Config.ImageStabilizationMode imageStabilizationMode) {
    // Method contents removed.
  }

  /**
   * Checks whether the provided {@link com.google.ar.core.Config.GeospatialMode
   * Config.GeospatialMode} is supported on this device. The current list of supported devices is
   * documented on the <a href="https://developers.google.com/ar/devices">ARCore supported
   * devices</a> page. A device may be incompatible with a given {@link
   * com.google.ar.core.Config.GeospatialMode Config.GeospatialMode} due to insufficient sensor
   * capabilities.
   *
   * @param geospatialMode The desired Geospatial mode to check.
   * @return true if the given {@link com.google.ar.core.Config.GeospatialMode
   *     Config.GeospatialMode} is supported on this device.
   */
  public boolean isGeospatialModeSupported(
      com.google.ar.core.Config.GeospatialMode geospatialMode) {
    // Method contents removed.
  }

  /**
   * Gets the availability of the Visual Positioning System (VPS) at a specified horizontal
   * position. The availability of VPS in a given location helps to improve the quality of
   * Geospatial localization and tracking accuracy.
   *
   * <p>This launches an asynchronous operation used to query the Google Cloud ARCore API. See
   * {@link com.google.ar.core.Future Future} for information on obtaining results and cancelling
   * the operation.
   *
   * <p>This may be called without calling {@link com.google.ar.core.Session#resume()
   * Session#resume()} or {@link com.google.ar.core.Session#configure(com.google.ar.core.Config) }.
   *
   * <p>Your app must be properly set up to communicate with the Google Cloud ARCore API in order to
   * obtain a result from this call. See <a
   * href="https://developers.google.com/ar/develop/java/geospatial/check-vps-availability">Check
   * VPS Availability</a> for more details on setup steps and usage examples.
   *
   * @param latitude The latitude in degrees.
   * @param longitude The longitude in degrees.
   * @param callback An optional <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_callback">callback</a>,
   *     called when the operation is finished, unless the future is <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_cancellation">cancelled</a>.
   *     May be null.
   * @return A handler that can be <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_polling">polled</a>
   *     or <a
   *     href="https://developers.google.com/ar/reference/java/com/google/ar/core/Future#future_cancellation">cancelled</a>.
   * @throws java.lang.SecurityException if the Android <a
   *     href="https://developer.android.com/training/basics/network-ops/connecting">{@code
   *     INTERNET}</a> permission has not been granted.
   */
  @androidx.annotation.NonNull
  public com.google.ar.core.VpsAvailabilityFuture checkVpsAvailabilityAsync(
      double latitude,
      double longitude,
      @androidx.annotation.Nullable
          java.util.function.Consumer<com.google.ar.core.VpsAvailability> callback) {
    // Method contents removed.
  }

  /**
   * Sets a MP4 dataset file to play back instead of live camera feed.
   *
   * <p>Restrictions:
   *
   * <ul>
   *   <li>Can only be called while the session is paused. Playback of the MP4 dataset file will
   *       start once the session is resumed.
   *   <li>The MP4 dataset file must use the same camera facing direction as is configured in the
   *       session.
   *   <li>Due to the way session data is processed, ARCore APIs may sometimes produce different
   *       results during playback than during recording and produce different results during
   *       subsequent playback sessions. For example, the number of detected planes and other
   *       trackables, the precise timing of their detection and their pose over time may be
   *       different in subsequent playback sessions.
   *   <li>Once playback has started (due to the first call to {@link #resume()}), pausing the
   *       session (by calling {@link #pause()}) will suspend processing of all camera image frames
   *       and any other recorded sensor data in the dataset. Camera image frames and sensor frame
   *       data that is discarded in this way will not be reprocessed when the session is again
   *       resumed (by calling {@link #resume()}). AR tracking for the session will generally suffer
   *       due to the gap in processed data.
   * </ul>
   *
   * <p>When an MP4 dataset file is set:
   *
   * <ul>
   *   <li>All existing trackables ({@link com.google.ar.core.Anchor Anchor} and {@link
   *       com.google.ar.core.Trackable Trackable}) immediately enter tracking state {@link
   *       com.google.ar.core.TrackingState#STOPPED TrackingState#STOPPED}.
   *   <li>The desired focus mode {@link
   *       com.google.ar.core.Config#setFocusMode(com.google.ar.core.Config.FocusMode)
   *       Config#setFocusMode(Config.FocusMode)} is ignored, and will not affect the previously
   *       recorded camera images.
   *   <li>The current camera configuration ({@link com.google.ar.core.CameraConfig CameraConfig})
   *       is immediately set to the default for the device the MP4 dataset file was recorded on.
   *   <li>Calls to {@link
   *       com.google.ar.core.Session#getSupportedCameraConfigs(com.google.ar.core.CameraConfigFilter)
   *       Session#getSupportedCameraConfigs(CameraConfigFilter)} will return camera configs
   *       supported by the device the MP4 dataset file was recorded on.
   *   <li>Setting a previously obtained camera config to {@link
   *       com.google.ar.core.Session#setCameraConfig(com.google.ar.core.CameraConfig) } will have
   *       no effect.
   * </ul>
   *
   * <p>Throws {@link com.google.ar.core.exceptions.SessionUnsupportedException} when incompatible
   * session features are specified:
   *
   * <ul>
   *   <li>{@link com.google.ar.core.Session.Feature#SHARED_CAMERA Session.Feature#SHARED_CAMERA}
   * </ul>
   *
   * @throws com.google.ar.core.exceptions.SessionNotPausedException if called when session is not
   *     paused.
   * @throws com.google.ar.core.exceptions.SessionUnsupportedException if playback is incompatible
   *     with selected features.
   * @throws com.google.ar.core.exceptions.PlaybackFailedException for other playback failures.
   * @deprecated Please use {@link #setPlaybackDatasetUri(android.net.Uri)} to play back datasets.
   */
  @Deprecated
  public void setPlaybackDataset(java.lang.String mp4DatasetFilePath)
      throws com.google.ar.core.exceptions.PlaybackFailedException {
    // Method contents removed.
  }

  /**
   * Sets a MP4 dataset to play back instead of live camera feed.
   *
   * <p>The {@link android.net.Uri Uri} must represent a seekable resource. File descriptors opened
   * with {@link android.content.ContentResolver#openAssetFileDescriptor(android.net.Uri, String)}
   * must support {@link android.system.Os#lseek(java.io.FileDescriptor, long, int)}.
   *
   * <p>See {@link #setPlaybackDataset(java.lang.String)} for additional restrictions and details.
   *
   * @throws java.lang.IllegalArgumentException if {@code mp4DatasetUri} does not represent a
   *     seekable resource.
   * @throws com.google.ar.core.exceptions.SessionNotPausedException if called when session is not
   *     paused.
   * @throws com.google.ar.core.exceptions.SessionUnsupportedException if playback is incompatible
   *     with selected feature.
   * @throws com.google.ar.core.exceptions.PlaybackFailedException for other playback failures.
   */
  public void setPlaybackDatasetUri(android.net.Uri mp4DatasetUri)
      throws com.google.ar.core.exceptions.PlaybackFailedException {
    // Method contents removed.
  }

  /** Returns the current playback status. */
  @androidx.annotation.NonNull
  public com.google.ar.core.PlaybackStatus getPlaybackStatus() {
    // Method contents removed.
  }

  /**
   * Starts a new MP4 dataset file recording that is written to the specific filesystem path.
   *
   * <p>Existing files will be overwritten.
   *
   * <p>The MP4 video stream (VGA) bitrate is 5Mbps (40Mb per minute).
   *
   * <p>Recording introduces additional overhead and may affect app performance.
   *
   * <p>Session recordings may contain sensitive information. See <a
   * href="https://developers.google.com/ar/develop/recording-and-playback#what%E2%80%99s_in_a_recording">documentation
   * on Recording and Playback</a> to learn which data is saved in a recording.
   *
   * @param recordingConfig The {@link com.google.ar.core.RecordingConfig RecordingConfig} that
   *     defines the recording configuration.
   * @throws java.lang.IllegalArgumentException if {@code recordingConfig} is null or ARCore cannot
   *     write to the configured output.
   * @throws java.lang.IllegalStateException if recording has been started already. In this case,
   *     the existing recording continues.
   * @throws com.google.ar.core.exceptions.RecordingFailedException for other recording failures.
   */
  public void startRecording(com.google.ar.core.RecordingConfig recordingConfig)
      throws com.google.ar.core.exceptions.RecordingFailedException {
    // Method contents removed.
  }

  /**
   * Stops recording and flushes unwritten data to disk. The MP4 dataset file is ready to read after
   * this call returns.
   *
   * <p>Recording can also be stopped automatically when {@link #pause()} is called if {@link
   * com.google.ar.core.RecordingConfig#getAutoStopOnPause() } is true in {@link
   * com.google.ar.core.RecordingConfig RecordingConfig}. Recording errors that would be thrown in
   * stopRecording() are silently ignored in {@link #pause()}.
   *
   * <p>Calling this method when no recording is in progress is a no-op.
   *
   * @throws com.google.ar.core.exceptions.RecordingFailedException if the file cannot be written
   *     and fully flushed to disk.
   */
  public void stopRecording() throws com.google.ar.core.exceptions.RecordingFailedException {
    // Method contents removed.
  }

  /** Returns the current recording status. */
  @androidx.annotation.NonNull
  public com.google.ar.core.RecordingStatus getRecordingStatus() {
    // Method contents removed.
  }

  /**
   * Fundamental session features that can be requested using {@link
   * #Session(android.content.Context,java.util.Set)}.
   *
   * <p>All currently defined features are mutually compatible.
   */
  @SuppressWarnings({"unchecked", "deprecation", "all"})
  public enum Feature {
    /**
     * Use the front-facing (selfie) camera. When the front camera is selected, ARCore's behavior
     * changes in the following ways:
     *
     * <ul>
     *   <li>The display will be mirrored. Specifically, {@link
     *       com.google.ar.core.Camera#getProjectionMatrix(float[],int,float,float)
     *       Camera#getProjectionMatrix(float[], int, float, float)} will include a horizontal flip
     *       in the generated projection matrix and APIs that reason about things in screen space
     *       such as {@link
     *       com.google.ar.core.Frame#transformCoordinates2d(com.google.ar.core.Coordinates2d,float[],com.google.ar.core.Coordinates2d,float[])
     *       } will mirror screen coordinates. Open GL apps should consider using {@code
     *       glFrontFace} to render mirrored assets without changing their winding direction.
     *   <li>{@link com.google.ar.core.Camera#getTrackingState() Camera#getTrackingState()} will
     *       always return {@link com.google.ar.core.TrackingState#PAUSED TrackingState#PAUSED}.
     *   <li>All forms of {@code Frame.hitTest()} will always return an empty list.
     *   <li>{@link com.google.ar.core.Camera#getDisplayOrientedPose()
     *       Camera#getDisplayOrientedPose()} will always return an identity pose.
     *   <li>{@link #createAnchor(com.google.ar.core.Pose)} will always throw {@link
     *       com.google.ar.core.exceptions.NotTrackingException}.
     *   <li>Planes will never be detected.
     *   <li>{@link com.google.ar.core.Session#configure(com.google.ar.core.Config)
     *       Session#configure(Config)} will throw if the supplied configuration requests Cloud
     *       Anchors or Augmented Images.
     * </ul>
     *
     * @deprecated To create a session using the front-facing (selfie) camera, use {@link
     *     com.google.ar.core.Session#setCameraConfig(com.google.ar.core.CameraConfig) } with the
     *     desired config retrieved from {@link
     *     com.google.ar.core.Session#getSupportedCameraConfigs(com.google.ar.core.CameraConfigFilter)
     *     }.
     */
    @Deprecated
    FRONT_CAMERA,
    /**
     * Enable sharing the camera between ARCore and the app. See the class-level documentation for
     * {@link com.google.ar.core.SharedCamera SharedCamera} for an example of how to use this
     * method.
     *
     * <p>Note:
     *
     * <ul>
     *   <li>You must check that a compatible version of ARCore is installed before calling this
     *       method. See the class-level documentation for {@link com.google.ar.core.Session
     *       Session} for details.
     *   <li>You must wrap {@code CameraDevice.StateCallback} and {@code
     *       CameraCaptureSession.StateCallback} callbacks using {@link
     *       com.google.ar.core.SharedCamera#createARDeviceStateCallback(android.hardware.camera2.CameraDevice.StateCallback,android.os.Handler)
     *       } and {@link
     *       com.google.ar.core.SharedCamera#createARSessionStateCallback(android.hardware.camera2.CameraCaptureSession.StateCallback,android.os.Handler)
     *       } respectively.
     * </ul>
     *
     * <p>While ARCore is paused you can use the Camera2 APIs directly without restriction.
     *
     * <p>While ARCore is active, the app may continue to use Camera2 APIs, except it must not call
     * {@code CameraCaptureSession#setRepeatingRequest} as this will interfere with ARCore
     * operation.
     */
    SHARED_CAMERA;
  }

  /**
   * Indicates the quality of the visual features seen by ARCore in the preceding few seconds from a
   * given camera {@link com.google.ar.core.Pose Pose}. A higher quality indicates a Cloud Anchor
   * hosted at the current time with the current set of recently seen feature points will generally
   * be easier to resolve more accurately. See the <a
   * href="https://developers.google.com/ar/develop/java/cloud-anchors/developer-guide">Cloud
   * Anchors developer guide</a> for more information.
   */
  @SuppressWarnings({"unchecked", "deprecation", "all"})
  public enum FeatureMapQuality {
    /**
     * The quality of features seen from the pose in the preceding few seconds is low. This state
     * indicates that ARCore will likely have more difficulty {@link
     * com.google.ar.core.Session#resolveCloudAnchorAsync resolving} the Cloud Anchor. Encourage the
     * user to move the device, so that the desired position of the Cloud Anchor to be hosted is
     * seen from different angles.
     */
    INSUFFICIENT,
    /**
     * The quality of features seen from the pose in the preceding few seconds is likely sufficient
     * for ARCore to successfully {@link com.google.ar.core.Session#resolveCloudAnchorAsync resolve}
     * the Cloud Anchor, although the accuracy of the resolved pose will likely be reduced.
     * Encourage the user to move the device, so that the desired position of the Cloud Anchor to be
     * hosted is seen from different angles.
     */
    SUFFICIENT,
    /**
     * The quality of features seen from the pose in the preceding few seconds is likely sufficient
     * for ARCore to successfully {@link com.google.ar.core.Session#resolveCloudAnchorAsync resolve}
     * a Cloud Anchor with a high degree of accuracy.
     */
    GOOD;
  }
}
